/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.model.java.index

import groovy.transform.EqualsAndHashCode
import groovy.transform.ToString
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.ClassCm
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.metamodel.model.data.ArrayType
import pl.metaprogramming.metamodel.model.data.DataType
import pl.metaprogramming.metamodel.model.data.MapType

import static pl.metaprogramming.codemodel.model.java.JavaDefs.*

class ClassIndex {

    @EqualsAndHashCode
    static class IndexKey {
        def metaModel
        def classType

        String toString() {
            metaModel == null ? classType
                    : "$classType:$metaModel"
        }
    }

    @ToString
    static class IndexEntry {
        ClassCm classCm
        def classType
    }

    List<ClassIndex> dependsOn = []

    String name
    Map<IndexKey, ClassCd> index = [:]
    Set<String> usedNames = []
    Map<ClassCd, List<MethodCm>> mappersIndex = [:]
    Closure classTypeMapper = { it }


    List<IndexEntry> getCodeModels() {
        index.findAll {it.value instanceof ClassCm }.collect {
            new IndexEntry(classCm: (ClassCm) it.value, classType: it.key.classType)
        }
    }

    void put(ClassCd classCd, def metaModel, def classType) {
        assert !usedNames.contains(classCd.canonicalName), "class $classCd.canonicalName already registred"
        usedNames.add(classCd.canonicalName)
        index.put(makeIndexKey(metaModel, classType), classCd)
    }

    ClassCd getClass(Object classType, def metaModel = null) {
        def result = getClassByKey(makeIndexKey(metaModel, classTypeMapper.call(classType)))
        result ?: metaModel != null ? getClassByKey(makeIndexKey(null, classTypeMapper.call(classType))): null
    }

    ClassCd getClassForDataType(DataType dataType, def classType, DataTypeMapper dataTypeMapper) {
        if (dataType && dataTypeMapper) {
            def result = dataTypeMapper.map(dataType)
            if (result) {
                return result
            }
        }
        if (dataType instanceof ArrayType) {
            def genericParam = getClassForDataType(dataType.itemsSchema.dataType, classType, dataTypeMapper)
            assert genericParam, "Undefined item type for collection field: $dataType"
            new ClassCd(T_LIST, [genericParam])
        } else if (dataType instanceof MapType) {
            def genericParam = getClassForDataType(dataType.valuesSchema.dataType, classType, dataTypeMapper)
            if (!genericParam) {
                throw new RuntimeException("Can't find class for $dataType.valuesSchema.dataType (of type $classType)")
            }
            new ClassCd(T_MAP, [T_STRING, genericParam])
        } else {
            getClass(classType, dataType)
        }
    }


    private ClassCd getClassByKey(IndexKey key) {
        index.get(key) ?: dependsOn.findResult { it.getClassByKey(key) }
    }

    static IndexKey makeIndexKey(def metaModel, def classType) {
        if (metaModel != null && metaModel.getClass() == DataType.class) {
            new IndexKey(metaModel: (metaModel as DataType).typeCode)
        } else {
            new IndexKey(metaModel: metaModel, classType: classType)
        }
    }

    void putMappers(MethodCm...mapperMethods) {
        mapperMethods.each {
            if (it.resultType != null && !it.resultType.isGeneric && it.accessModifier == ACCESS_PUBLIC) {
                mappersIndex.putIfAbsent(it.resultType, [])
                def methodsByResultType = mappersIndex.get(it.resultType)
                List<ClassCd> params = it.params.collect { it.type }
                if (matchMethod(methodsByResultType, params)) {
                    printMapperIndex()
                    throw new RuntimeException("Mapper already defined ($params) -> $it.resultType")
                }
                methodsByResultType.add(it)
            }
        }
    }

    private static MethodCm matchMethod(List<MethodCm> methods, List<ClassCd> params) {
        methods?.find { it.params.collect { it.type } == params }
    }

    MethodCm getMapper(ClassCd resultType, List<ClassCd> params) {
        def methods = mappersIndex.get(resultType)
        def result = matchMethod(methods, params)
        if (result == null) {
            result = dependsOn.findResult { it.getMapper(resultType, params)}
        }
        result
    }

    void printMapperIndex() {
        println "Mapper index of $name"
        println '----------------------------------------------'
        mappersIndex.each {
            println "\t$it.key: $it.value"
        }
        println '----------------------------------------------'
        dependsOn.each { it.printMapperIndex()}
    }

    void printIndex() {
        println "Class index of $name"
        println '----------------------------------------------'
        collectClasses().each {
            println "\t$it"
        }
        println '----------------------------------------------'
    }

    List<String> collectClasses(){
        List<String> result = index.collect {
            "$it.value.canonicalName ($it.key.classType for $it.key.metaModel)".toString()
        }
        result.sort()
    }

}
