/*
 * Copyright (c) 2018,2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.model.java

import groovy.transform.EqualsAndHashCode

@EqualsAndHashCode
class ClassCd {
    String packageName
    String className
    List<ClassCd> genericParams
    boolean isGeneric = false
    boolean isEnum = false
    boolean isArray = false

    ClassCd() {}

    ClassCd(ClassCd prototype, List<ClassCd> genericParams = null) {
        this.packageName = prototype.packageName
        this.className = prototype.className
        this.genericParams = genericParams
    }

    ClassCd(String canonicalName, List<ClassCd> genericParams = null) {
        def name = parseCanonicalName(canonicalName)
        this.packageName = name[0]
        this.className = name[1]
        this.genericParams = genericParams
    }

    ClassCd(String packageName, String className, List<ClassCd> genericParams = null) {
        this.packageName = packageName
        this.className = className
        this.genericParams = genericParams
    }

    static private String[] parseCanonicalName(String canonicalName) {
        String[] result = new String[2]
        int lastDot = canonicalName.lastIndexOf('.')
        if (lastDot > 0) {
            result[0] = (canonicalName.substring(0, lastDot))
            result[1] = (canonicalName.substring(lastDot+1, canonicalName.length()))
        } else {
            result[1] = canonicalName
        }
        result
    }

    boolean isClass(String canonicalName) {
        def name = parseCanonicalName(canonicalName)
        name[0] == packageName && name[1] == className
    }

    String getCanonicalName() {
        packageName ? "${packageName}.${className}" : className
    }

    void collectDependencies(Collection<ClassCd> dependencies) {
        dependencies.add(this)
        genericParams?.each { it.collectDependencies(dependencies) }
    }

    ClassCd cloneWithParams(ClassCd...params) {
        new ClassCd(packageName: packageName, className: className, genericParams: Arrays.asList(params))
    }

    @Override
    String toString() {
        def result = getCanonicalName()
        if (genericParams) {
            result += '<' + genericParams.collect { it.toString() }.join(', ') + '>'
        }
        result
    }
}
