/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.formatter

import pl.metaprogramming.codemodel.model.java.*

class JavaCodeFormatter extends BaseJavaCodeFormatter implements CodeFormatter<ClassCm> {

    ClassCm classCm

    String format(ClassCm codeModel) {
        format()
    }

    JavaCodeFormatter(ClassCm classCm) {
        this.classCm = classCm
        initImports()
    }

    void initImports() {
        def excludePackages = [classCm.packageName, 'java.lang'] + classCm.imports.findAll { it.endsWith('.*') }.collect { it.substring(0, it.length()-2)}
        Set<ClassCd> classes = []
        classCm.annotations.each { it.collectDependencies(classes) }
        classCm.extend?.collectDependencies(classes)
        classCm.interfaces.each { it.collectDependencies(classes) }
        classCm.fields.each { it.collectDependencies(classes) }
        classCm.enumItems.each { it.collectDependencies(classes) }
        classCm.methods.each { it.collectDependencies(classes) }
        List<String> result = []
        result.addAll(classCm.imports)
        result.addAll(classes
                .findAll { it.packageName && !excludePackages.contains(it.packageName) && !it.isGeneric }
                .collect { "${it.packageName}.${it.className}".toString() })
        imports = result.unique().sort()
    }

    String format() {
        init(classCm.packageName)

        buf.add("package $classCm.packageName;").newLine()
        buf.addLines(imports.collect {"import $it;"}).newLine()

        if (classCm.description) {
            buf.addLines(formatJavadoc(classCm.description))
        }
        buf.addLines(formatAnnotations(classCm.annotations)).newLine()

        buf.add("public ").add(formatClassHeader()).add(' {')
        buf.indent(1).newLine()

        addEnumItems(buf)
        addFields(buf)

        buf.indent().newLine()
        if (classCm.fields && classCm.methods) buf.newLine()
        addMethods(buf)

        buf.indent().newLine('}').newLine()
        buf.toString()
    }

    private void addEnumItems(CodeBuffer buf) {
        EnumItemCm prevItem
        classCm.enumItems.each {
            if (prevItem) {
                buf.add(',')
                if (prevItem.description) buf.add(" // $prevItem.description")
            }
            buf.addLines(formatAnnotations(it.annotations))
            buf.newLine("${it.name}(${it.value})")
            prevItem = it
        }
        if (prevItem) {
            buf.add(';')
            if (prevItem.description) buf.add(" // $prevItem.description")
            buf.newLine()
        }
    }

    private void addFields(CodeBuffer buf) {

        def valueFields = classCm.fields.findAll { it.value != null }
        if (valueFields) {
            valueFields.each {
                addField(it, buf)
            }
            buf.newLine()
        }
        classCm.fields.findAll { it.value == null } .each {
            addField(it, buf)
        }
    }

    private void addField(FieldCm field, CodeBuffer buf) {
        if (field.description) {
            buf.newLine().addLines(formatJavadoc(field.description))
        }
        buf.newLine().add(formatField(field))
        buf.add(';')
    }

    String formatField(FieldCm field) {
        formatAnnotations(field.annotations, SPACE) +
                field.modifiers + SPACE +
                formatTypeName(field) +
                (field.value == null ? '' : ' = ' + field.value)
    }

    private void addMethods(CodeBuffer buf) {
        classCm.methods.each {
            buf.indent(1)
            if (it.description) {
                buf.addLines(formatJavadoc(it.description))
            }
            buf.addLines(formatAnnotations(it.annotations))
            if (classCm.isInterface) {
                buf.newLine(formatSignature(it)).add(";").newLine()
            } else {
                buf.newLine("$it.accessModifier ${formatSignature(it)} {")
                buf.indent(2).addLines(it.implBody)
                buf.indent(1).newLine("}").indent().newLine()
            }
        }
    }


    String formatClassHeader() {
        def buf = new StringBuffer()
        buf.append(classCm.isInterface ? 'interface' : classCm.isEnum ? 'enum' : 'class')
        buf.append(' ').append(classCm.className)
        if (classCm.genericParams) {
            buf.append('<')
            buf.append(classCm.genericParams.collect { it.className }.join(', '))
            buf.append('>')

        }
        if (classCm.extend) {
            buf.append(" extends ${formatUsage(classCm.extend)}")
        }
        if (classCm.interfaces.size() > 0) {
            buf.append(classCm.isInterface ? ' extends ' : ' implements ')
            buf.append(classCm.interfaces.collect { formatUsage(it) }.join(', '))
        }
        buf.toString()
    }


    String formatParam(FieldCm fieldCm) {
        "${formatAnnotations(fieldCm.annotations, SPACE)}${formatTypeName(fieldCm)}"
    }

    String formatTypeName(FieldCm fieldCm) {
        "${formatUsage(fieldCm.type)} ${fieldCm.name}"
    }

    static List<String> formatJavadoc(String comment) {
        ['/**', ' * ' + comment, ' */']
    }


    String formatSignature(MethodCm methodCm) {
        def buf = new StringBuilder()
        if (methodCm.staticModifier) {
            buf.append('static ')
        }
        def genericParams = genericParams(methodCm)
        if (genericParams) {
            buf.append(genericParams).append(' ')
        }
        def resultType = methodCm.resultType ? formatUsage(methodCm.resultType) : isConstructor(methodCm) ? null : 'void'
        if (resultType) {
            buf.append(resultType).append(' ')
        }
        buf.append("${methodCm.name}(${methodCm.params ? methodCm.params.collect { formatParam(it)}.join(', ') : ''})")
        buf.toString()
    }

    boolean isConstructor(MethodCm methodCm) {
        methodCm.name == classCm.className
    }

    String genericParams(MethodCm methodCm) {
        Set<ClassCd> genericParams = []
        if (methodCm.resultType) {
            collectGenericParams(methodCm.resultType, genericParams)
        }
        methodCm.params?.each { collectGenericParams(it.type, genericParams)}
        formatGenericParams(genericParams)
    }

    static collectGenericParams(ClassCd cm, Set<ClassCd> result) {
        if (cm.isGeneric) {
            result.add(cm)
        } else {
            cm.genericParams?.each { collectGenericParams(it, result) }
        }
    }

    static String toJavaName(String name, boolean capitalize = false) {
        StringBuilder buf = new StringBuilder()
        boolean upperCase = false
        for (int i = 0; i < name.length(); i++) {
            char c = name.charAt(i)
            if (Character.isLetterOrDigit(c)) {
                if (buf.length() == 0 && Character.isDigit(c)) {
                    buf.append('_')
                }
                buf.append(buf.length() == 0 ? (capitalize ? Character.toUpperCase(c) : Character.toLowerCase(c))
                        : upperCase ? Character.toUpperCase(c) : c)
                upperCase = false
            } else {
                upperCase = true
            }
        }
        buf.toString()
    }


    static String toUpperCase(String javaName) {
        StringBuilder buf = new StringBuilder()
        for (int i = 0; i < javaName.length(); i++) {
            char c = javaName.charAt(i)
            if (Character.isUpperCase(c) && i > 0) {
                buf.append('_')
            }
            buf.append(Character.toUpperCase(c))
        }
        buf.toString()
    }

}
