/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.formatter

class CodeBuffer {
    StringBuffer buf
    String indent
    String newLine
    String tab

    CodeBuffer() {
        this(JavaCodeFormatter.NEW_LINE, JavaCodeFormatter.TAB)
    }

    CodeBuffer(String newLine, String tab) {
        this.newLine = newLine
        this.tab = tab
        buf = new StringBuffer()
        indent = ''
    }

    CodeBuffer indent(int tabs = 0) {
        indent = tab * tabs
        this
    }

    CodeBuffer startIf(String condition) {
        newLine("if ($condition) {")
        indent = indent + tab
        this
    }

    CodeBuffer endIf() {
        indent = indent - tab
        newLine('}')
        this
    }

    CodeBuffer ifElseValue(String condition, String ifValue, String elseValue) {
        newLine(condition)
        indent = indent + tab + tab
        newLine('? ' + ifValue)
        newLine(': ' + elseValue)
        indent = indent - (tab + tab)
        this
    }

    CodeBuffer add(def text) {
        buf.append(text)
        this
    }

    CodeBuffer newLine(def text = '') {
        if (buf.size() > 0 || !text) buf.append(newLine)
        if (indent) buf.append(indent)
        buf.append(text)
        this
    }

    CodeBuffer addLines(def lines) {
        if (lines instanceof List) {
            lines.each { newLine(it) }
        } else if (lines instanceof String || lines instanceof GString) {
            lines.eachLine { newLine(it) }
        }
        this
    }

    CodeBuffer addLines(String...lines) {
        lines.each { newLine(it) }
        this
    }

    String toString() {
        buf.toString()
    }
}
