/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.xml

import pl.metaprogramming.codemodel.builder.java.ClassCmBuildHelper
import pl.metaprogramming.codemodel.builder.java.dto.BaseDtoBuildStrategy
import pl.metaprogramming.codemodel.model.java.AnnotationCm
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.ValueCm
import pl.metaprogramming.metamodel.model.data.DataSchema
import pl.metaprogramming.metamodel.model.data.ObjectType

import static pl.metaprogramming.codemodel.builder.java.ClassType.XML_DTO
import static pl.metaprogramming.codemodel.builder.java.ClassType.XML_ENUM
import static pl.metaprogramming.codemodel.builder.java.MetaModelAttribute.JAVA_NAME
import static pl.metaprogramming.codemodel.model.java.JavaDefs.ANNOT_NON_NULL
import static pl.metaprogramming.codemodel.model.java.JavaDefs.ANNOT_NULLABLE

class XmlDtoBuildStrategy extends BaseDtoBuildStrategy {

    static AnnotationCm XML_ACCESSOR_TYPE_ANNOTATION = new AnnotationCm('javax.xml.bind.annotation.XmlAccessorType', [value: ValueCm.value("XmlAccessType.FIELD")])

    @Override
    void makeImplementation(ClassCmBuildHelper<ObjectType> builder) {
        super.makeImplementation(builder)
        builder.interfaces.clear()  // TODO don't add Serializable interface in BaseDtoBuildStrategy
        def metaModel = builder.metaModel
        builder.addImports("javax.xml.bind.annotation.XmlAccessType")
        builder.addAnnotation(XML_ACCESSOR_TYPE_ANNOTATION)
        builder.addAnnotation(prepareXmlTypeAnnotation(metaModel))
        if (metaModel.isRootElement) {
            builder.addAnnotation(prepareXmlRootElementAnnotation(metaModel))
        }
    }

    @Override
    void addField(ClassCmBuildHelper<ObjectType> builder, DataSchema schema) {
        builder.addFields new FieldCm(
                name: schema.getAttribute(JAVA_NAME),
                description: schema.description,
                annotations: getFieldAnnotations(schema),
                type: builder.getClass(isEnum(schema) ? XML_ENUM : XML_DTO, schema.dataType))
    }

    static List<AnnotationCm> getFieldAnnotations(DataSchema schema) {
        def result = []
        result.add(schema.isRequired && !schema.isNillable ? ANNOT_NON_NULL : ANNOT_NULLABLE)
        def xmlElementAttributes = [:]
        if (schema.isRequired) {
            xmlElementAttributes.put('required', ValueCm.value('true'))
        }
        if (schema.isNillable) {
            xmlElementAttributes.put('nillable', ValueCm.value('true'))
        }
        if (xmlElementAttributes) {
            result.add(new AnnotationCm('javax.xml.bind.annotation.XmlElement', xmlElementAttributes))
        }
        result
    }

    static AnnotationCm prepareXmlTypeAnnotation(ObjectType metaModel) {
        new AnnotationCm('javax.xml.bind.annotation.XmlType', [
                name: ValueCm.escaped(metaModel.isRootElement ? '' : metaModel.code),
                propOrder: new ValueCm(
                        value: metaModel.fields.collect { it.code },
                        array: true,
                        escaped: true,
                        itemPerLine: true)
        ])
    }

    static AnnotationCm prepareXmlRootElementAnnotation(ObjectType metaModel) {
        new AnnotationCm('javax.xml.bind.annotation.XmlRootElement', [
                name: ValueCm.escaped(metaModel.code)])
    }
}
