/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.spring

import pl.metaprogramming.codemodel.builder.java.ClassCmBuildHelper
import pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategy
import pl.metaprogramming.codemodel.builder.java.method.BaseMethodCmBuilder
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.metamodel.model.wsdl.WsdlApi

import static pl.metaprogramming.codemodel.builder.java.ClassType.ENDPOINT_PROVIDER
import static pl.metaprogramming.codemodel.builder.java.ClassType.WS_CLIENT

class WsClientConfigurationBuildStrategy extends ClassCmBuildStrategy<WsdlApi> {

    @Override
    void makeImplementation(ClassCmBuildHelper<WsdlApi> builder) {
        builder.addAnnotation(SpringDefs.ANNOT_CONFIGURATION)
        builder.addMethods(new CreateClientMethodBuilder(builder).make())
    }

    static class CreateClientMethodBuilder extends BaseMethodCmBuilder<WsdlApi> {

        CreateClientMethodBuilder(ClassCmBuildHelper<WsdlApi> buildHelper) {
            super(buildHelper)
        }

        @Override
        protected MethodCm prepareMethodCM() {
            def clientClass = classBuilder.getClass(WS_CLIENT, metaModel)
            new MethodCm(
                    name: "create${clientClass.className}",
                    resultType: clientClass,
                    annotations: [SpringDefs.ANNOT_BEAN],
            )
        }

        @Override
        protected void prepareImplBody() {
            def packages = metaModel.schemaPackages.collect { "\"$it\""}.join(", ")
            def resultClass = methodCm.resultType.className
            classBuilder.getClass(WS_CLIENT, metaModel)
            classBuilder.addImports('org.springframework.oxm.jaxb.Jaxb2Marshaller')
            implBodyBuf
                    .newLine('Jaxb2Marshaller marshaller = new Jaxb2Marshaller();')
                    .newLine("marshaller.setContextPaths($packages);")
                    .newLine("$resultClass client = new $resultClass();")
                    .newLine("client.setDefaultUri(${getEndpointUri()});")
                    .newLine("client.setMarshaller(marshaller);")
                    .newLine("client.setUnmarshaller(marshaller);")
                    .newLine("return client;")

        }

        String getEndpointUri() {
            String path = "\"${metaModel.uri}\""
            classBuilder.getClassOptional(ENDPOINT_PROVIDER)
                    .map( {
                        def endpointProviderField = classBuilder.injectDependency(it)
                        "${endpointProviderField.name}.getEndpoint($path)"
                    } ).orElseGet({ path })
        }
    }
}
