/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.spring

import pl.metaprogramming.codemodel.builder.java.ClassCmBuildHelper
import pl.metaprogramming.codemodel.builder.java.method.BaseMethodCmBuilder
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.codemodel.model.java.ValueCm
import pl.metaprogramming.metamodel.model.data.ArrayType
import pl.metaprogramming.metamodel.model.data.DataType
import pl.metaprogramming.metamodel.model.rest.Operation

import static pl.metaprogramming.codemodel.builder.java.ClassType.*
import static pl.metaprogramming.codemodel.model.java.JavaDefs.ANNOT_NON_NULL
import static pl.metaprogramming.codemodel.model.java.JavaDefs.COLLECTIONS
import static pl.metaprogramming.codemodel.model.java.JavaDefs.T_LIST

class ResponseEntityMapperBuilder extends BaseMethodCmBuilder<Operation> {

    ResponseEntityMapperBuilder(ClassCmBuildHelper<Operation> classCmBuilder) {
        super(classCmBuilder)
    }

    @Override
    protected MethodCm prepareMethodCM() {
        new MethodCm(
                name: 'map',
                resultType: SpringDefs.RESPONSE_ENTITY,
                params: [new FieldCm(name: 'response', type: new ClassCd(classBuilder.getClass(RESPONSE_DTO)), annotations: [ANNOT_NON_NULL])],
        )
    }

    @Override
    protected void prepareImplBody() {
        implBodyBuf.addLines(
                'ResponseEntity.BodyBuilder responseBuilder = ResponseEntity.status(response.getValue().getStatus());',
                'Object body = response.isPresent() ? response.getValue().getBody() : null;',
        )
        addObjectsMappings()
        addArraysMappings()
        implBodyBuf.addLines(returnBody('body'))
    }

    private void addObjectsMappings() {
        metaModel.responses.findAll { it.schema?.object }.collect { it.schema.objectType }.unique().each {
            prepareResponseMap(it)
        }
    }

    private void prepareResponseMap(DataType respType) {
        def javaClass = getClass(DTO, respType)
        def jsonClass = getClass(REST_DTO, respType)
        def transformation = makeTransformation(jsonClass, [new FieldCm(type: javaClass, value: ValueCm.value("($javaClass.className) body"))])
        classBuilder.addImport(javaClass)
        implBodyBuf.startIf("body instanceof ${javaClass.className}")
                .newLine(returnBody(transformation))
                .endIf()
    }

    private void addArraysMappings() {
        def arrayTypes = metaModel.responses.findAll { it.schema?.array }.collect { it.schema.arrayType }
        if (!arrayTypes) return
        classBuilder.addImport(COLLECTIONS)
        classBuilder.addImports([T_LIST])
        implBodyBuf.startIf('body instanceof List')
                .newLine('Object item = ((List) body).stream().findFirst().orElse(null);')
        implBodyBuf.startIf('item == null')
                .newLine(returnBody('Collections.emptyList()'))
                .endIf()
        addArraysMappings(arrayTypes)
        implBodyBuf.endIf()
    }

    private void addArraysMappings(Collection<ArrayType> arrayTypes) {
        def handledItems = []
        arrayTypes.each { listSchema ->
            def javaDtoClass = getClass(DTO, listSchema)
            def itemJavaClass = javaDtoClass.genericParams[0].className
            if (!handledItems.contains(itemJavaClass)) {
                handledItems.add(itemJavaClass)
                classBuilder.addImport(javaDtoClass.genericParams[0])
                def fromValue = new FieldCm(
                        type: javaDtoClass,
                        value: ValueCm.value("(${javaDtoClass.className}<${itemJavaClass}>) body"))
                implBodyBuf.startIf("item instanceof $itemJavaClass")
                        .newLine(returnBody(transform(fromValue, getClass(REST_DTO, listSchema)).expression))
                        .endIf()
            }
        }
    }

    private static String returnBody(String body) {
        "return responseBuilder.body($body);"
    }
}
