/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.spring

import pl.metaprogramming.codemodel.builder.java.ClassCmBuildHelper
import pl.metaprogramming.codemodel.builder.java.method.BaseMethodCmBuilder
import pl.metaprogramming.codemodel.model.java.*
import pl.metaprogramming.metamodel.model.data.DataSchema
import pl.metaprogramming.metamodel.model.data.DataType
import pl.metaprogramming.metamodel.model.rest.Operation
import pl.metaprogramming.metamodel.model.rest.Parameter
import pl.metaprogramming.metamodel.model.rest.enums.ParamLocation

import static pl.metaprogramming.codemodel.builder.java.ClassType.REST_DTO
import static pl.metaprogramming.codemodel.formatter.JavaCodeFormatter.toJavaName

class AbstractControllerMethodBuilder extends BaseMethodCmBuilder<Operation> {

    static final List<AnnotationCm> REQUEST_BODY_ANNOTATIONS = [new AnnotationCm('org.springframework.web.bind.annotation.RequestBody')]

    static final Map<ParamLocation, String> PARAM_ANNOTATIONS = [
            (ParamLocation.PATH)  : 'PathVariable',
            (ParamLocation.QUERY) : 'RequestParam',
            (ParamLocation.HEADER): 'RequestHeader',
            (ParamLocation.COOKIE): 'CookieValue',
    ]

    AbstractControllerMethodBuilder(Operation operation, ClassCmBuildHelper classBuilder) {
        super(operation, classBuilder)
    }

    MethodCm prepareMethodCM() {
        new MethodCm(
                name: metaModel.code,
                description: metaModel.description,
                resultType: SpringDefs.RESPONSE_ENTITY,
                annotations: [requestHandlerAnnotation()],
                params: requestHandlerParams(),
        )
    }

    protected AnnotationCm requestHandlerAnnotation() {
        Map<String, ValueCm> params = [
                value   : ValueCm.escaped(metaModel.path),
                produces: ValueCm.escapedArray(metaModel.produces),
        ]
        if (metaModel.type.hasBody()) {
            params.consumes = ValueCm.escapedArray(metaModel.consumes)
        }
        String canonicalName = "org.springframework.web.bind.annotation.${metaModel.type.name().toLowerCase().capitalize()}Mapping"
        new AnnotationCm(canonicalName, params)
    }

    protected List<FieldCm> requestHandlerParams() {
        List<FieldCm> params = []
        if (metaModel.requestBody != null) {
            if (ControllerConfig.payloadField) {
                def payloadField = new FieldCm(ControllerConfig.payloadField)
                payloadField.annotations = REQUEST_BODY_ANNOTATIONS
                params.add(payloadField)
            }
            params.add(new FieldCm(name: 'requestBody', type: getClass(REST_DTO, metaModel.requestBody), annotations: REQUEST_BODY_ANNOTATIONS))
        }
        metaModel.parameters.each {
            String javaName = toJavaName(it.code ?: it.name)
            params.add(new FieldCm(name: javaName, type: getParamType(it), annotations: [requestHandlerParamAnnotation(it, javaName)]))
        }
        params
    }

    protected ClassCd getParamType(DataSchema dataSchema) {
        if (dataSchema.isArray()) {
            new ClassCd(JavaDefs.T_LIST, [getParamType(dataSchema.arrayType.itemsSchema)])
        } else if (dataSchema.dataType == DataType.BINARY) {
            SpringDefs.T_MULTIPART_FILE
        } else {
            JavaDefs.T_STRING
        }
    }

    protected AnnotationCm requestHandlerParamAnnotation(Parameter parameter, String javaName) {
        Map params = [:]
        if (parameter.name != javaName) params.put('value', ValueCm.escaped(parameter.name))
        params.put('required', ValueCm.value('false'))
        String annotation = getParamAnnotation(parameter)
        new AnnotationCm("org.springframework.web.bind.annotation.$annotation", params)
    }

    protected String getParamAnnotation(Parameter parameter) {
        String annotation = PARAM_ANNOTATIONS.get(parameter.location)
        if (parameter.location == ParamLocation.FORMDATA) {
            annotation = parameter.dataType == DataType.BINARY ? 'RequestPart' : 'RequestParam'
        }
        if (!annotation) {
            throw new RuntimeException("[$metaModel.operationId] Can't handle parameter $parameter.code in $parameter.location")
        }
        annotation
    }

    protected FieldCm inject(def classType) {
        classBuilder.injectDependency(getClass(classType))
    }

    protected String getClassName(def classType) {
        getClass(classType).className
    }

    protected String getParamsChain() {
        methodCm.params.collect {it.name }.join(', ')
    }
}
