/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.rest

import pl.metaprogramming.codemodel.builder.java.ClassCmBuildHelper
import pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategy
import pl.metaprogramming.codemodel.builder.java.method.BaseMethodCmBuilder
import pl.metaprogramming.codemodel.formatter.BaseJavaCodeFormatter
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.metamodel.model.rest.HttpResponse
import pl.metaprogramming.metamodel.model.rest.Operation

import static pl.metaprogramming.codemodel.builder.java.ClassType.*
import static pl.metaprogramming.codemodel.model.java.JavaDefs.ANNOT_NON_NULL
import static pl.metaprogramming.codemodel.model.java.JavaDefs.T_INTEGER

class RestResponseBuilder extends ClassCmBuildStrategy<Operation> {

    @Override
    void makeImplementation(ClassCmBuildHelper<Operation> builder) {
        builder.setSuperClass(builder.getGenericClass(VALUE_HOLDER_TEMPLATE, [new ClassCd(builder.getClass(REST_RESPONSE_TEMPLATE))]))
        builder.addMethods(builder.metaModel.responses.collect {
            new SetResultMethodBuilder(it, builder).make()
        } as MethodCm[])
    }

    class SetResultMethodBuilder extends BaseMethodCmBuilder<HttpResponse> {
        private ClassCd bodyClass

        SetResultMethodBuilder(HttpResponse metaModel, ClassCmBuildHelper<Operation> classCmBuilder) {
            super(metaModel, classCmBuilder)
        }

        protected MethodCm prepareMethodCM() {
            def params = []
            if (!isFixedStatus()) {
                params.add(new FieldCm(name: 'status', type: T_INTEGER, annotations: [ANNOT_NON_NULL]))
            }
            if (metaModel.schema) {
                bodyClass = getClass(DTO, metaModel.schema.dataType)
                params.add(new FieldCm(name: 'body', type: bodyClass, annotations: [ANNOT_NON_NULL]))
            }
            new MethodCm(
                    name: isFixedStatus() ? 'set'+metaModel.status : 'set',
                    params: params,
                    resultType: classBuilder.builtClass,
            )
        }

        protected void prepareImplBody() {
            // TODO use formatter of class
            BaseJavaCodeFormatter formatter = new BaseJavaCodeFormatter() {
                boolean isImported(ClassCd classCd) { true }
            }
            implBodyBuf.add("setValue(new RestResponse${bodyClass ? '<'+ formatter.formatUsage(bodyClass)+'>':''}()")
            if (isFixedStatus()) {
                implBodyBuf.add(".setStatus(${metaModel.status})")
            } else {
                implBodyBuf.add('.setStatus(status)')
            }
            if (bodyClass) {
                implBodyBuf.add('.setBody(body)')
            }
            implBodyBuf.add(');')
            implBodyBuf.newLine('return this;')
        }

        private boolean isFixedStatus() {
            metaModel.status != 0
        }

    }

}
