/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.rest

import pl.metaprogramming.codemodel.builder.java.ClassCmBuildHelper
import pl.metaprogramming.codemodel.builder.java.method.BaseMethodCmBuilder
import pl.metaprogramming.codemodel.formatter.JavaCodeFormatter
import pl.metaprogramming.codemodel.model.java.*
import pl.metaprogramming.metamodel.model.data.DataSchema
import pl.metaprogramming.metamodel.model.data.DataTypeCode
import pl.metaprogramming.metamodel.model.data.ObjectType
import pl.metaprogramming.metamodel.model.rest.Parameter

import static pl.metaprogramming.codemodel.builder.java.ClassType.*
import static pl.metaprogramming.codemodel.builder.java.MetaModelAttribute.*

class CheckMethodBuilder extends BaseMethodCmBuilder<ObjectType> {
    ClassCm dtoClass

    CheckMethodBuilder(ObjectType dtoMetaModel, ClassCm dtoClass, ClassCmBuildHelper classBuilder) {
        super(dtoMetaModel, classBuilder)
        this.dtoClass = dtoClass
    }

    @Override
    protected MethodCm prepareMethodCM() {
        def genericParams = classBuilder.superClass ? classBuilder.superClass.genericParams : classBuilder.interfaces[0].genericParams
        new MethodCm(
                name: 'check',
//                    annotations: [ANNOT_OVERRIDE],
                params: [new FieldCm(
                        name: 'ctx',
                        type: classBuilder.getGenericClass(VALIDATION_CONTEXT, genericParams))]
        )
    }

    @Override
    protected void prepareImplBody() {
        metaModel.fields.each {
            addDescriptiveField(it)
            addValidation(it)
        }
    }

    private void addDescriptiveField(DataSchema schema) {
        def fieldClass = classBuilder.getClass(VALIDATION_FIELD)
        def field = dtoClass.fields.find { it.metaModel == schema}
        def descriptiveField = schema.getAttribute(ORIGINAL_NAME_FIELD)
        classBuilder.addFields(new FieldCm(
                name: descriptiveField,
                modifiers: 'private static final',
                type: new ClassCd(fieldClass, [dtoClass, field.type]),
                value: ValueCm.value("new ${fieldClass.className}<>(${getDtoStaticField(descriptiveField)}, ${dtoClass.className}::get${field.name.capitalize()})")
        ))
    }

    private void addValidation(DataSchema schema) {
        def checkers = getCheckers(schema)
        if (checkers) {
            implBodyBuf.newLine("ctx.check(${schema.getAttribute(ORIGINAL_NAME_FIELD)}, ${checkers.join(', ')});")
        }
    }

    private List<String> getCheckers(DataSchema schema, int level = 1) {
        List<String> checkers = []
        if (schema.isRequired) {
            checkers.add('REQUIRED')
        }
        if ([DataTypeCode.BOOLEAN, DataTypeCode.INT32, DataTypeCode.INT64, DataTypeCode.FLOAT, DataTypeCode.DOUBLE].contains(schema.dataType.typeCode)) {
            checkers.add(schema.dataType.typeCode.name())
        }
        addCustomChecker(checkers, schema)
        addObjectChecker(checkers, schema)
        addArrayChecker(checkers, schema, level)
        addEnumChecker(checkers, schema)
        addDataTimeChecker(checkers, schema)
        addPatternChecker(checkers, schema)
        addLengthChecker(checkers, schema)
        addMinMaxChecker(checkers, schema)
        checkers
    }

    private void addCustomChecker(List<String> checkers, DataSchema schema) {
        def validator = CustomValidators.getValidator(schema)
        if (validator) {
            checkers.add(classBuilder.injectDependency(validator).name)
        }
    }

    private void addObjectChecker(List<String> checkers, DataSchema schema) {
        if (!schema.isObject() || schema instanceof Parameter) return
        def objectValidator = classBuilder.injectDependency(classBuilder.getClass(REST_DTO_VALIDATOR, schema.objectType))
        checkers.add(objectValidator.name)
    }

    private void addArrayChecker(List<String> checkers, DataSchema schema, int level) {
        if (!schema.isArray()) return
        if (schema.arrayType.minItems > 0) {
            checkers.add('REQUIRED')
        }
        addListSizeChecker(checkers, schema, level)
        def itemCheckers = getCheckers(schema.arrayType.itemsSchema, level + 1)
        if (!itemCheckers.isEmpty()) {
            checkers.add("list(${itemCheckers.join(', ')})".toString())
        }
    }

    private void addDataTimeChecker(List<String> checkers, DataSchema schema) {
        if (!schema.isType(DataTypeCode.DATE_TIME)) return
        def checkerName = schema.getAttribute(FORMAT_FIELD)
        addCheckerField(checkerName, "dateTime(${getDtoStaticField(schema.getAttribute(FORMAT_FIELD))})")
        checkers.add(checkerName)
    }

    private void addEnumChecker(List<String> checkers, DataSchema schema) {
        if (!schema.isEnum()) return
        def enumClass = classBuilder.getClass(ENUM, schema.dataType)
        def checkerName = JavaCodeFormatter.toUpperCase(enumClass.className)
        if (classBuilder.findFields {it.name == checkerName }.isEmpty()) {
            classBuilder.addImport(enumClass)
            addCheckerField(checkerName, "allow(${enumClass.className}.values())")
        }
        checkers.add(checkerName)
    }

    private void addPatternChecker(List<String> checkers, DataSchema schema) {
        if (!schema.pattern) return
        String checkerName = "${schema.getAttribute(ORIGINAL_NAME_FIELD)}_PATTERN"
        classBuilder.addImports(['java.util.regex.Pattern'])
        addCheckerField(checkerName, "matches(Pattern.compile(\"${schema.pattern.replace('\\', '\\\\')}\"))")
        checkers.add(checkerName)
    }

    private void addMinMaxChecker(List<String> checkers, DataSchema schema) {
        if (!schema.minimum && !schema.maximum) return
        String checkerName = "${schema.getAttribute(ORIGINAL_NAME_FIELD)}_MIN_MAX"
        def type = getClass(schema.dataType)
        addCheckerField(checkerName, "range(${type.className}::valueOf, ${ValueCm.escaped(schema.minimum)}, ${ValueCm.escaped(schema.maximum)})")
        checkers.add(checkerName)
    }

    private void addLengthChecker(List<String> checkers, DataSchema schema) {
        if (!schema.minLength && !schema.maxLength) return
        String checkerName = "${schema.getAttribute(ORIGINAL_NAME_FIELD)}_LENGTH"
        addCheckerField(checkerName, "length(${schema.minLength}, ${schema.maxLength})")
        checkers.add(checkerName)
    }

    private void addListSizeChecker(List<String> checkers, DataSchema schema, int level) {
        if (!schema.arrayType.minItems && !schema.arrayType.maxItems) return
        String checkerName = "${schema.getAttribute(ORIGINAL_NAME_FIELD)}${level > 1 ? '_'+level : ''}_SIZE"
        addCheckerField(checkerName, "size(${schema.arrayType.minItems}, ${schema.arrayType.maxItems})", JavaDefs.LIST_ANONYMOUS)
        checkers.add(checkerName)
    }

    private void addCheckerField(String checkerName, String checkerValue, ClassCd testedType = JavaDefs.T_STRING) {
        classBuilder.addFields(new FieldCm(
                name: checkerName,
                modifiers: 'private static final',
                type: classBuilder.getGenericClass(VALIDATION_CHECKER, [testedType]),
                value: ValueCm.value(checkerValue)
        ))
    }


    String getDtoStaticField(String field) {
        "${dtoClass.className}.${field}"
    }
}
