/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.module

import pl.metaprogramming.codemodel.builder.CodeGenerationTask
import pl.metaprogramming.codemodel.builder.java.ClassCmBuilder
import pl.metaprogramming.codemodel.builder.java.ClassCmBuilderImpl
import pl.metaprogramming.codemodel.builder.java.config.JavaModuleConfig
import pl.metaprogramming.codemodel.builder.java.config.SpringWsOutAdapterConfigurator
import pl.metaprogramming.codemodel.builder.java.xml.XmlPackageInfoBuilder
import pl.metaprogramming.codemodel.formatter.JavaCodeFormatter
import pl.metaprogramming.metamodel.model.data.DataSchema
import pl.metaprogramming.metamodel.model.wsdl.WsdlApi

import static pl.metaprogramming.codemodel.builder.java.ClassType.*

class SpringWsOutAdapterBuilder extends JavaModuleBuilder<WsdlApi> {

    SpringWsOutAdapterBuilder(JavaModuleConfig config) {
        super(config)
    }

    List<CodeGenerationTask> make(WsdlApi model) {
        addSchemaCodes(model)
        addPackageInfo(model)
        addObjectFactory(model.schemas.values())
        addOperationCodes(model)
        makeClassGenerationTasks()
    }

    void addPackageInfo(WsdlApi model) {
        def baseDir = getConfig(XML_DTO).baseDir
        assert baseDir == getConfig(XML_ENUM).baseDir
        def namespaces = model.schemas.values().collect { it.namespace }.unique()
        packageInfoList = namespaces.collect { namespace ->
            XmlPackageInfoBuilder.make(
                    baseDir,
                    namespace,
                    getPackage(namespace),
                    model.namespaceElementFormDefault[namespace])
        }
    }

    void addObjectFactory(Collection<DataSchema> schemas) {
        schemas
                .findAll { !it.enum }
                .groupBy { it.namespace }
                .each { namespace, namespaceSchemas ->
                    createBuilder(namespace, XML_OBJECT_FACTORY, 'ObjectFactory', namespaceSchemas).make()
        }
    }

    void addOperationCodes(WsdlApi model) {
        String clientName = model.name.capitalize()
        createBuilder(WS_CLIENT, model, clientName).make()
        createBuilder(WS_CLIENT_CONFIGURATION, model, clientName).make()
    }

    void addSchemaCodes(WsdlApi model) {
        List<ClassCmBuilder> dtoBuilders = []
        model.schemas.values().each { schema ->
            def builder = createBuilder(schema)
            model.schemaPackages.add(getPackage(schema.namespace))
            if (schema.enum) {
                builder.make()
            } else {
                dtoBuilders.add(builder)
            }
        }
        dtoBuilders
                .each { it.makeDeclaration() }
                .each { it.makeImplementation() }
    }

    ClassCmBuilder createBuilder(DataSchema schema) {
        createBuilder(
                schema.namespace,
                schema.enum ? XML_ENUM : XML_DTO,
                JavaCodeFormatter.toJavaName(schema.code, true),
                schema.dataType
        )
    }

    ClassCmBuilder createBuilder(String namespace, def classType, String className, def metaModel) {
        new ClassCmBuilderImpl<>(
                config: getConfig(classType),
                classIndex: classIndex,
                metaModel: metaModel,
                metaModelName: className,
                packageName: getPackage(namespace)
        )

    }

    String getPackage(String namespace) {
        def packageName = config.params[SpringWsOutAdapterConfigurator.NAMESPACE_2_PACKAGE][namespace]
        assert packageName, "Package not defined for namespace: $namespace"
        packageName
    }
}
