/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.module

import pl.metaprogramming.codemodel.builder.CodeGenerationTask
import pl.metaprogramming.codemodel.builder.java.ClassCmBuilder
import pl.metaprogramming.codemodel.builder.java.config.JavaModuleConfig
import pl.metaprogramming.codemodel.builder.java.config.SpringRestInAdapterConfigurator
import pl.metaprogramming.codemodel.model.java.JavaDefs
import pl.metaprogramming.metamodel.model.rest.Operation
import pl.metaprogramming.metamodel.model.rest.Parameter
import pl.metaprogramming.metamodel.model.rest.RestApi

import java.util.function.Consumer

import static pl.metaprogramming.codemodel.builder.java.ClassType.*

class SpringRestInAdapterBuilder extends JavaModuleBuilder<RestApi> {

    private static final List OPERATION_DTO_CLASSES = [REST_REQUEST_DTO, REQUEST_DTO, RESPONSE_DTO]

    private Consumer<RestOperationBuildHelper> additionalOperationCodeBuilder

    private boolean isMultiOperationControllerStrategy() {
        config.params[SpringRestInAdapterConfigurator.MULTI_OPERATION_CONTROLLER_STRATEGY]
    }

    private List operationClassTypes() {
        def result = [REST_REQUEST_MAPPER, REST_REQUEST_VALIDATOR, RESPONSE_MAPPER]
        if (!isMultiOperationControllerStrategy()) {
            result.add(REST_CONTROLLER)
        }
        result
    }

    SpringRestInAdapterBuilder(JavaModuleConfig config, Consumer<RestOperationBuildHelper> additionalOperationCodeBuilder = {}) {
        super(config)
        this.additionalOperationCodeBuilder = additionalOperationCodeBuilder
    }

    List<CodeGenerationTask> make(RestApi model) {
        addObjectParamsCodes(model.parameters)
        addSchemaCodes(model)
        addOperationCodes(model)
        makeClassGenerationTasks()
    }

    void addObjectParamsCodes(Map<String, Parameter> parameters) {
        parameters.findAll { it.value.isObject() }.each { paramName, paramDef ->
            classIndex.put(JavaDefs.T_STRING, paramDef.dataType, REST_DTO)
            createBuilder(DTO, paramDef.dataType, paramDef.objectType.code).make()
            createBuilder(REST_MAPPER, paramDef.dataType, paramDef.objectType.code).make()
        }
    }

    void addSchemaCodes(RestApi model) {
        def inputObjects = model.getInputObjects()
        List<ClassCmBuilder> dtoBuilders = []
        List<ClassCmBuilder> mapperBuilders = []
        List<ClassCmBuilder> validatorBuilders = []
        model.schemas.findAll { it.value.isEnum() }.each { dataName, schema ->
            createBuilder(ENUM, schema.dataType, dataName).make()
        }
        model.schemas.findAll { it.value.isObject() }.each { dataName, schema ->
            dtoBuilders.add(createBuilder(DTO, schema.dataType, dataName))
            dtoBuilders.add(createBuilder(REST_DTO, schema.dataType, dataName))
            mapperBuilders.add(createBuilder(REST_MAPPER, schema.dataType, dataName))
            if (inputObjects.contains(schema.dataType)) {
                validatorBuilders.add(createBuilder(REST_DTO_VALIDATOR, schema.dataType, dataName))
            }
        }
        dtoBuilders
                .each { it.makeDeclaration() }
                .each { it.makeImplementation() }
        mapperBuilders
                .each { it.makeDeclaration() }
                .each { it.makeImplementation() }
        validatorBuilders
                .each { it.makeDeclaration() }
                .each { it.makeImplementation() }
    }

    void addOperationCodes(RestApi model) {
        model.operations.each { operationGroup, operations ->
            List<ClassCmBuilder> operationBuilders = []
            String resourceCode = operationGroup.capitalize()
            operations.each { operationCode, operation ->
                String dataName = operation.code.capitalize()
                OPERATION_DTO_CLASSES.each { classType ->
                    createBuilder(classType, operation, dataName).make()
                }
                operationClassTypes().each { classType ->
                    operationBuilders.add(createBuilder(classType, operation, dataName).makeDeclaration())
                }
                buildAdditionalOperationCodes(operation)
            }
            def resourceOperations = operations.values().toList()
            createBuilder(FACADE_IMPL, resourceOperations, resourceCode).make().extractInterface(getConfig(FACADE))
            if (isMultiOperationControllerStrategy()) {
                createBuilder(REST_CONTROLLER_MO, resourceOperations, resourceCode).make()
            }
            operationBuilders.each { it.makeImplementation() }
        }
    }

    void buildAdditionalOperationCodes(Operation operation) {
        additionalOperationCodeBuilder.accept(new RestOperationBuildHelper(generationTasksBuilder: this, operation: operation))
    }


    static class RestOperationBuildHelper {
        private JavaModuleBuilder<RestApi> generationTasksBuilder
        private Operation operation

        void addClass(def classType, String metaModelName = defaultMetaModelName()) {
            generationTasksBuilder.createBuilder(classType, operation, metaModelName).make()
        }

        String defaultMetaModelName() {
            operation.group.capitalize() + operation.code.capitalize()
        }
    }

}
