/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.module

import pl.metaprogramming.codemodel.builder.CodeGenerationTask
import pl.metaprogramming.codemodel.builder.ModuleBuilder
import pl.metaprogramming.codemodel.builder.java.ClassBuilderConfig
import pl.metaprogramming.codemodel.builder.java.ClassCmBuilder
import pl.metaprogramming.codemodel.builder.java.ClassCmBuilderImpl
import pl.metaprogramming.codemodel.builder.java.config.JavaModuleConfig
import pl.metaprogramming.codemodel.formatter.GspTemplateFormatter
import pl.metaprogramming.codemodel.formatter.JavaCodeFormatter
import pl.metaprogramming.codemodel.formatter.PackageInfoFormatter
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.ClassCm
import pl.metaprogramming.codemodel.model.java.PackageInfoCm
import pl.metaprogramming.codemodel.model.java.index.ClassIndex

abstract class JavaModuleBuilder<T> implements ModuleBuilder<T> {

    JavaModuleConfig config
    ClassIndex classIndex
    List<CodeGenerationTask> fixedClasses = []
    List<PackageInfoCm> packageInfoList = []


    JavaModuleBuilder(JavaModuleConfig config, String name = null) {
        this.config = config
        classIndex = new ClassIndex(classTypeMapper: { config.mapClassType(it)}, name: name)
    }

    void dependsOn(ClassIndex...dependsOn) {
        classIndex.dependsOn = Arrays.asList(dependsOn)
    }


    protected List<CodeGenerationTask> makeClassGenerationTasks() {
        fixedClasses +
                makeClassGenerationTasksFromClassIndex() +
                makePackageInfoGenerationsTasks()
    }

    protected List<CodeGenerationTask> makeClassGenerationTasksFromClassIndex() {
        classIndex.getCodeModels().collect {
            assert it.classCm.packageName
            new CodeGenerationTask(
                    destFilePath: getClassFilePath(it.classType, it.classCm),
                    formatter: new JavaCodeFormatter(it.classCm),
                    codeModel: it.classCm
            )
        }
    }

    protected List<CodeGenerationTask> makePackageInfoGenerationsTasks() {
        packageInfoList.collect {
            new CodeGenerationTask(
                    destFilePath: getJavaFilePath(it.baseDir, it.packageName, 'package-info'),
                    codeModel: it,
                    formatter: new PackageInfoFormatter(),
            )
        }
    }

    protected String getClassFilePath(Object classType, ClassCm codeModel) {
        def baseDir = getConfig(classType).baseDir
        assert baseDir, "No base dir for class type: $classType"
        getJavaFilePath(baseDir, codeModel.packageName, codeModel.className)
    }

    static protected String getJavaFilePath(String baseDir, String packageName, String className) {
        "$baseDir/${pkgToDir(packageName)}/${className}.java"
    }

    protected ClassCmBuilder createBuilder(def classType, def metaModel, String metaModelName) {
        new ClassCmBuilderImpl<>(
                config: getConfig(classType),
                classIndex: classIndex,
                metaModel: metaModel,
                metaModelName: metaModelName
        )
    }

    protected void addClassBuilder(Object classType) {
        if (!classIndex.getClass(classType)) {
            def classBuilderConfig = getConfig(classType)
            assert classBuilderConfig.className
            if (classBuilderConfig.gspTemplate != null) {
                addFixedClass(classBuilderConfig)
            } else {
                createBuilder(classType, null, classBuilderConfig.className).make()
            }
        }
    }

    protected void addFixedClass(ClassBuilderConfig cfg) {
        ClassCd classCd = new ClassCd(cfg.packageName, cfg.className)
        classIndex.put(classCd, null, cfg.classType)
        Map codeModel = [:]
        codeModel['classCd'] = classCd
        cfg.dependencies?.each {
            codeModel.put(it.toString(), classIndex.getClass(it))
        }
        assert classCd.packageName
        fixedClasses.add(new CodeGenerationTask(
                destFilePath: "${cfg.baseDir}/${pkgToDir(classCd.packageName)}/${classCd.className}.java",
                formatter: new GspTemplateFormatter(cfg.gspTemplate),
                codeModel: codeModel)
        )
    }

    protected ClassBuilderConfig getConfig(def classType) {
        config.getConfig(classType)
    }

    static String pkgToDir(String pkg) {
        pkg.replaceAll('\\.', '/')
    }
}