/*
 * Copyright (c) 2018,2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.method

import pl.metaprogramming.codemodel.builder.java.ClassCmBuildHelper
import pl.metaprogramming.codemodel.model.java.ClassCm
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.codemodel.model.java.ValueCm
import pl.metaprogramming.metamodel.model.data.DataSchema
import pl.metaprogramming.metamodel.model.data.ObjectType

import static pl.metaprogramming.codemodel.model.java.JavaDefs.T_STRING

class MapperMethodBuilder extends BaseMethodCmBuilder<ObjectType> {

    def fromClassType
    def toClassType
    boolean dstObjectInParam
    ClassCm to
    List<FieldCm> from = []

    MapperMethodBuilder(ClassCmBuildHelper<ObjectType> mapperBuilder, def fromClassType, def toClassType, boolean dstObjectInParam = false) {
        super(mapperBuilder.metaModel, mapperBuilder)
        this.fromClassType = fromClassType
        this.toClassType = toClassType
        this.dstObjectInParam = dstObjectInParam
    }

    MethodCm prepareMethodCM() {
        to = (ClassCm) getClass(toClassType)
        if (fromClassType) {
            from.add(new FieldCm(name: 'value', type: getClass(fromClassType)))
        } else {
            from.addAll(to.fields.findAll { !it.isStatic() })
        }

        if (dstObjectInParam) {
            new MethodCm(
                    name: "map",
                    resultType: to,
                    params: [new FieldCm(name: 'result', type: to)] + from,
            )
        } else {
            new MethodCm(
                    name: "map2$to.className",
                    resultType: to,
                    params: from,
            )
        }
    }

    void prepareImplBody() {
        if (dstObjectInParam) {
            if (to.extend) {
                implBodyBuf.add(makeTransformation(to.extend, methodCm.params.collect {
                    new FieldCm(name: it.name, type: (it.type as ClassCm).extend)
                })).add(';').newLine()
            }
            implBodyBuf.add("return result").indent(1)
        } else {
            implBodyBuf.add("return ${fromClassType ? from[0].name + ' == null ? null : ' : ''}new ${to.className}()").indent(1)
        }

        appendMappings()

        implBodyBuf.newLine().add(';')
    }

    void appendMappings() {
        List<FieldCm> fromFields = getFromFields()
        List<FieldCm> mappedFields = []
        metaModel.fields.each {
            def toField = getField(to.fields, it)
            def fromField = getField(fromFields, it)
            mappedFields.add(fromField)
            def transformation = makeTransformation(
                    toField.type, toTransformationParams(fromField, getFieldFormat(it, toField, fromField)))
            implBodyBuf.newLine().add(".set${toField.name.capitalize()}(").add(transformation).add(')')
        }
        (fromFields - mappedFields).each { fromField ->
            def toField = getField(to.fields, fromField.metaModel)
            if (toField?.type == fromField.type) {
                implBodyBuf.newLine().add('.set').add(fromField.name.capitalize()).add('(').add(fromFieldPath(fromField)).add(')')
            }
        }
    }

    private String getFieldFormat(DataSchema schema, FieldCm toField, FieldCm fromField) {
        if (!fromClassType || toField.type == T_STRING && fromField.type == T_STRING) {
            null
        } else {
            schema.isArray() ? schema.arrayType.itemsSchema.format : schema.format
        }
    }


    static FieldCm getField(List<FieldCm> fields, def schema) {
        fields.find { it.metaModel == schema }
    }


    List<FieldCm> toTransformationParams(FieldCm field, String format = null) {
        List<FieldCm> params = []
        params.add(new FieldCm(name: fromFieldPath(field), type: field.type))
        if (format) {
            params.add(new FieldCm(value: ValueCm.escaped(format), type: T_STRING))
        }
        params
    }

    String fromFieldPath(FieldCm field) {
        fromRootField ? "${fromRootField}.${field.name}".toString() : field.name
    }

    List<FieldCm> getFromFields() {
        fromClassType ? ((ClassCm) from[0].type).fields : from
    }

    String getFromRootField() {
        fromClassType ? from[0].name : null
    }
}
