/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.method

import pl.metaprogramming.codemodel.builder.java.ClassCmBuildHelper
import pl.metaprogramming.codemodel.formatter.CodeBuffer
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.MethodCm
import pl.metaprogramming.codemodel.model.java.ValueCm

import static pl.metaprogramming.codemodel.model.java.JavaDefs.*

abstract class BaseMethodCmBuilder<T> extends MethodCmBuilder {
    ClassCmBuildHelper classBuilder
    MethodCm methodCm
    T metaModel
    CodeBuffer implBodyBuf = new CodeBuffer()


    BaseMethodCmBuilder(ClassCmBuildHelper<T> classBuilder) {
        this.metaModel = classBuilder.metaModel
        this.classBuilder = classBuilder
    }

    BaseMethodCmBuilder(T metaModel, ClassCmBuildHelper<?> classBuilder) {
        this.metaModel = metaModel
        this.classBuilder = classBuilder
    }

    MethodCm makeDeclaration() {
        methodCm = prepareMethodCM()
    }

    abstract protected MethodCm prepareMethodCM()

    String makeImplementation() {
        prepareImplBody()
        methodCm.implBody = implBodyBuf.toString()
    }

    protected void prepareImplBody() {
        implBodyBuf.add(methodCm.resultType ? 'return null;' : '')
    }

    String declare(FieldCm fieldCm) {
        classBuilder.addImport(fieldCm.type)
        "$fieldCm.type.className $fieldCm.name = $fieldCm.value;"
    }

    String makeTransformation(ClassCd toType, List<FieldCm> from, int level = 1) {
        assert from.size() > 0
        if (isListTransformation(toType, from)) {
            makeListTransformation(toType, from, level)
        } else if (isEnumTransformation(toType, from)) {
            makeEnumTransformation(toType, from[0])
        } else if ([toType] == from.type) {
            makeTransformationParams(from)
        } else {
            def mapper = classBuilder.findMapper(toType, from.type)
            def mapperField = classBuilder.injectDependency(mapper.ownerClass)
            "${mapperField.name}.${mapper.name}(${makeTransformationParams(from)})"
        }
    }

    static boolean isListTransformation(ClassCd toType, List<FieldCm> from) {
        toType.isClass(T_LIST) && from[0].type.isClass(T_LIST)
    }

    String makeListTransformation(ClassCd toType, List<FieldCm> from, int level) {
        // e.g. baseDataMapper.transformList(raw.getAuthors(), v -> baseDataMapper.toLong(v))
        def mapper = classBuilder.findMapper(LIST_R, [LIST_T, FUN_T_R])
        def mapperField = classBuilder.injectDependency(mapper.ownerClass)
        def varName = 'v' + (level > 1 ? level : '')
        def itemFrom = [new FieldCm(name: varName, type: from[0].type.genericParams.get(0))] + from.subList(1, from.size())
        def itemTransformation = makeTransformation(toType.genericParams.get(0), itemFrom, level+1)
        itemTransformation ? "${mapperField.name}.${mapper.name}(${makeTransformationParams([from[0]])}, ${varName} -> ${itemTransformation})" : null
    }

    static boolean isEnumTransformation(ClassCd toType, List<FieldCm> from) {
        if (from.size() == 1) {
            def fromType = from[0].type
            return toType.isEnum && fromType == T_STRING || toType == T_STRING && fromType.isEnum
        }
        false
    }

    String makeEnumTransformation(ClassCd toType, FieldCm from) {
        def param = path(from.name)
        def buf = new StringBuilder()
        buf.append("$param != null ? ")
        if (toType.isEnum) {
            classBuilder.addImport(toType)
            buf.append("${toType.className}.fromValue(${param})")
        } else {
            classBuilder.addImport(from.type)
            buf.append("${param}.getValue()")
        }
        buf.append(' : null')
        buf.toString()
    }


    FieldCm transform(FieldCm from, def toClassType, String variableName = null) {
        transform([from], toClassType, variableName)
    }

    FieldCm transform(List<FieldCm> from, def toClassType, String variableName = null) {
        def type = classBuilder.getClass(toClassType)
        new FieldCm(
                type: type,
                name: variableName,
                value: ValueCm.value(
                        makeTransformation(type, from)
                ))
    }


    static String makeTransformationParams(List<FieldCm> from) {
        from.collect { it.name ? path(it.name) : it.value}.join(', ')
    }


    static String path(String path) {
        StringBuilder buf = new StringBuilder()
        path.split('\\.').eachWithIndex { String entry, int i ->
            if (i == 0) {
                buf.append(entry)
            } else {
                buf.append('.get').append(entry.capitalize()).append('()')
            }
        }
        buf.toString()
    }

    protected ClassCd getClass(def classType, def metaModel = metaModel) {
        classBuilder.getClass(classType, metaModel)
    }

}
