/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.mapper


import pl.metaprogramming.codemodel.builder.java.ClassCmBuildHelper
import pl.metaprogramming.codemodel.model.java.ClassCd
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.MethodCm

import static pl.metaprogramming.codemodel.model.java.JavaDefs.*

class BaseDataMappersBuilder extends MappersBuildStrategy {

    static final List<String> IMPORTS = [
            'java.util.Calendar',
            'java.util.stream.Collectors',
            'java.text.SimpleDateFormat',
            'javax.xml.bind.DatatypeConverter',
    ]

    static BaseDataMappersBuilder instance = new BaseDataMappersBuilder(methods: prepareMethods())

    @Override
    void makeImplementation(ClassCmBuildHelper builder) {
        super.makeImplementation(builder)
        builder.addImports(IMPORTS)
    }

    static private List<MethodCm> prepareMethods() {
        def rawParam = [new FieldCm(name: 'value', type: T_STRING)]
        def formatParam = new FieldCm(name: 'format', type: T_STRING)
        [
                new MethodCm(name: 'fromString', resultType: GENERIC_T,
                        params: [new FieldCm(name: 'value', type: T_STRING), new FieldCm(name: 'transformer', type: new ClassCd(FUN, [T_STRING, GENERIC_T]))],
                        implBody: 'return value == null || value.isEmpty() ? null : transformer.apply(value);'
                ),
                new MethodCm(name: 'toString', resultType: T_STRING,
                        params: [new FieldCm(name: 'value', type: GENERIC_T), new FieldCm(name: 'transformer', type: new ClassCd(FUN, [GENERIC_T, T_STRING]))],
                        implBody: 'return value == null ? null : transformer.apply(value);'
                ),
                new MethodCm(name: 'transformList', resultType: LIST_R,
                        params: [new FieldCm(name: 'value', type: LIST_T), new FieldCm(name: 'transformer', type: FUN_T_R)],
                        implBody: 'return value == null ? null : value.stream().map(transformer).collect(Collectors.toList());'
                ),
                new MethodCm(name: 'toLong', resultType: T_LONG, params: rawParam,
                        implBody: 'return fromString(value, v -> Long.valueOf(v));'
                ),
                new MethodCm(name: 'toString', resultType: T_STRING, params: [new FieldCm(name: 'value', type: T_LONG)],
                        implBody: 'return toString(value, v -> v.toString());'
                ),
                new MethodCm(name: 'toInteger', resultType: T_INTEGER, params: rawParam,
                        implBody: 'return fromString(value, v -> Integer.valueOf(v));'
                ),
                new MethodCm(name: 'toString', resultType: T_STRING, params: [new FieldCm(name: 'value', type: T_INTEGER)],
                        implBody: 'return toString(value, v -> v.toString());'
                ),
                new MethodCm(name: 'toFloat', resultType: T_FLOAT, params: rawParam,
                        implBody: 'return fromString(value, v -> Float.valueOf(v));'
                ),
                new MethodCm(name: 'toString', resultType: T_STRING, params: [new FieldCm(name: 'value', type: T_FLOAT)],
                        implBody: 'return toString(value, v -> v.toString());'
                ),
                new MethodCm(name: 'toBigDecimal', resultType: T_BIG_DECIMAL, params: rawParam,
                        implBody: 'return fromString(value, v -> new BigDecimal(v));'
                ),
                new MethodCm(name: 'toString', resultType: T_STRING, params: [new FieldCm(name: 'value', type: T_BIG_DECIMAL)],
                        implBody: 'return toString(value, v -> v.toString());'
                ),
                new MethodCm(name: 'toBigDecimal', resultType: T_BIG_DECIMAL, params: rawParam + formatParam,
                        implBody: 'return fromString(value, v -> new BigDecimal(v));'
                ),
                new MethodCm(name: 'toString', resultType: T_STRING, params: [new FieldCm(name: 'value', type: T_BIG_DECIMAL)] + formatParam,
                        implBody: 'return toString(value, v -> v.toString());'
                ),
                new MethodCm(name: 'toDate', resultType: T_DATE, params: rawParam + formatParam,
                        implBody: '''return fromString(value, v -> {
    try {
        return isIso8601(format)
                ? deserializeIso8601(v)
                : new SimpleDateFormat(format).parse(v);
    } catch (Exception e) {
        throw new RuntimeException(e);
    }
});
'''
                ),
                new MethodCm(name: 'toString', resultType: T_STRING, params: [new FieldCm(name: 'value', type: T_DATE)] + formatParam,
                        implBody: '''return toString(value, v -> isIso8601(format)
        ? serializeIso8601(v)
        : new SimpleDateFormat(format).format(v));
'''
                ),

                new MethodCm(name: 'toBoolean', resultType: T_BOOLEAN, params: rawParam,
                        implBody: 'return fromString(value, v -> Boolean.valueOf(v));'
                ),
                new MethodCm(name: 'toString', resultType: T_STRING, params: [new FieldCm(name: 'value', type: T_BOOLEAN)],
                        implBody: 'return toString(value, v -> v.toString());'
                ),

                new MethodCm(name: 'toDouble', resultType: T_DOUBLE, params: rawParam,
                        implBody: 'return fromString(value, v -> Double.valueOf(v));'
                ),
                new MethodCm(name: 'toString', resultType: T_STRING, params: [new FieldCm(name: 'value', type: T_DOUBLE)],
                        implBody: 'return toString(value, v -> v.toString());'
                ),

                new MethodCm(
                        name: 'isIso8601',
                        accessModifier: 'private',
                        resultType: T_BOOLEAN,
                        params: [new FieldCm(name: 'format', type: T_STRING)],
                        implBody: 'return "ISO8601".equals(format);'
                ),
                new MethodCm(
                        name: 'deserializeIso8601',
                        accessModifier: 'private',
                        resultType: T_DATE,
                        params: [new FieldCm(name: 'v', type: T_STRING)],
                        implBody: 'return DatatypeConverter.parseDateTime(v).getTime();'
                ),
                new MethodCm(
                        name: 'serializeIso8601',
                        accessModifier: 'private',
                        resultType: T_STRING,
                        params: [new FieldCm(name: 'v', type: T_DATE)],
                        implBody: '''Calendar c = Calendar.getInstance();
c.setTime(v);
return DatatypeConverter.printDateTime(c);
'''
                ),
        ]
    }

}
