/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.dto

import pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategy
import pl.metaprogramming.codemodel.builder.java.ClassCmBuildHelper
import pl.metaprogramming.codemodel.formatter.JavaCodeFormatter
import pl.metaprogramming.codemodel.model.java.FieldCm
import pl.metaprogramming.codemodel.model.java.JavaDefs
import pl.metaprogramming.codemodel.model.java.ValueCm
import pl.metaprogramming.metamodel.model.data.DataSchema
import pl.metaprogramming.metamodel.model.data.ObjectType

import static pl.metaprogramming.codemodel.model.java.JavaDefs.I_SERIALIZABLE
import static pl.metaprogramming.codemodel.builder.java.MetaModelAttribute.*

abstract class BaseDtoBuildStrategy extends ClassCmBuildStrategy<ObjectType> {

    @Override
    void makeImplementation(ClassCmBuildHelper<ObjectType> builder) {
        builder.addInterfaces(I_SERIALIZABLE)
        if (builder.metaModel.inherits) {
            assert builder.metaModel.inherits.size() == 1
            builder.setSuperClass(builder.classType, builder.metaModel.inherits[0])
        }
        addFields(builder)
    }

    void addFields(ClassCmBuildHelper<ObjectType> builder) {
        builder.metaModel.fields.each { field ->
            try {
                field.setAttribute(JAVA_NAME, { JavaCodeFormatter.toJavaName(field.code) })
                addField(builder, field)
            } catch (Exception e) {
                throw new RuntimeException("Can't handle field '$field.code' for model '$builder.metaModelName'", e)
            }
        }
    }

    abstract void addField(ClassCmBuildHelper<ObjectType> builder, DataSchema schema)

    protected void addDescriptiveFields(ClassCmBuildHelper<ObjectType> builder, DataSchema schema, String originalName) {
        builder.addFields makeDescriptiveField(schema, ORIGINAL_NAME_FIELD, originalName)
        if (schema.format) {
            builder.addFields makeDescriptiveField(schema, FORMAT_FIELD, schema.format, 'FORMAT')
        }
    }

    protected FieldCm makeDescriptiveField(DataSchema schema, def attributeKey, String fieldValue, String suffix = null) {
        String fieldNameConst = 'FIELD_' +
                JavaCodeFormatter.toUpperCase(schema.getAttribute(JAVA_NAME)) +
                (suffix ? '_' + suffix : '')
        schema.setAttribute(attributeKey, { fieldNameConst })
        new FieldCm(
                name: fieldNameConst,
                value: fieldValue ? ValueCm.escaped(fieldValue) : ValueCm.value('null'),
                modifiers: 'public static final',
                type: JavaDefs.T_STRING
        )
    }

    static protected boolean isEnum(DataSchema schema) {
        schema.isEnum() ||
                (schema.isArray() && schema.arrayType.itemsSchema.isEnum()) ||
                (schema.isMap() && schema.mapType.valuesSchema.isEnum())
    }
}
