/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.config

import groovy.transform.builder.Builder
import pl.metaprogramming.codemodel.builder.java.EnumBuildStrategy
import pl.metaprogramming.codemodel.builder.java.dto.DtoBuildStrategy
import pl.metaprogramming.codemodel.builder.java.dto.JsonDtoBuildStrategy
import pl.metaprogramming.codemodel.builder.java.dto.RawDtoBuildStrategy
import pl.metaprogramming.codemodel.builder.java.mapper.MappersBuilders
import pl.metaprogramming.codemodel.builder.java.rest.RestDtoValidatorBuilder
import pl.metaprogramming.codemodel.builder.java.rest.RestResponseBuilder
import pl.metaprogramming.codemodel.builder.java.rest.RestRequestValidatorBuilder
import pl.metaprogramming.codemodel.builder.java.spring.PayloadFieldAppenderBuildStrategy
import pl.metaprogramming.codemodel.builder.java.spring.RestControllerMoBuildStrategy
import pl.metaprogramming.codemodel.builder.java.spring.RestControllerBuildStrategy
import pl.metaprogramming.codemodel.builder.java.spring.FacadeEmptyImplBuildStrategy
import pl.metaprogramming.codemodel.builder.java.spring.SpringDefs
import pl.metaprogramming.metamodel.model.data.ObjectType
import pl.metaprogramming.metamodel.model.rest.Operation

import java.util.function.Function

import static pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategyWrapper.wrap
import static pl.metaprogramming.codemodel.builder.java.ClassType.*

@Builder
class SpringRestInAdapterConfigurator extends JavaModuleConfigurator<SpringRestInAdapterConfigurator> {

    // Generowanie dla każdej operacji oddzielnej klasy kontrolera.
    static final String MULTI_OPERATION_CONTROLLER_STRATEGY = 'MULTI_OPERATION_CONTROLLER_STRATEGY'
    static final Function<Operation, ObjectType> OPERATION_REQUEST_SCHEMA = { Operation o -> o.requestSchema } as Function<Operation, ObjectType>

    SpringRestInAdapterConfigurator multiOperationControllerStrategy() {
        moduleParams[MULTI_OPERATION_CONTROLLER_STRATEGY] = true
        this
    }

    @Override
    SpringRestInAdapterConfigurator init() {
        addComponent(REST_REQUEST_VALIDATOR, 'Validator', RestRequestValidatorBuilder.instance)
        addComponent(REST_DTO_VALIDATOR, 'Validator', RestDtoValidatorBuilder.instance)

        addClass(ENUM, 'Enum', new EnumBuildStrategy())
        addClass(RESPONSE_DTO, 'Response', new RestResponseBuilder())
        addComponent(RESPONSE_MAPPER, 'ResponseMapper', SpringDefs.RESPONSE_MAPPER_BUILDER)

        addLombokData(DTO, 'Dto', DtoBuildStrategy.instance)
        addLombokData(REST_DTO, 'Rdto', JsonDtoBuildStrategy.instance)
        addComponent(REST_MAPPER, 'Mapper', MappersBuilders.REST_DTO_MAPPER_BUILDER)

        addLombokData(REQUEST_DTO, 'Request', wrap(DtoBuildStrategy.instance, OPERATION_REQUEST_SCHEMA))
        addLombokData(REST_REQUEST_DTO, 'Rrequest', PayloadFieldAppenderBuildStrategy.instance, wrap(RawDtoBuildStrategy.instance, OPERATION_REQUEST_SCHEMA))
        addComponent(REST_REQUEST_MAPPER, 'RequestMapper', wrap(MappersBuilders.REST_REQUEST_MAPPER_BUILDER, OPERATION_REQUEST_SCHEMA))

        addClass(REST_CONTROLLER, 'Controller', new RestControllerBuildStrategy(), diStrategy)
        addClass(REST_CONTROLLER_MO, 'Controller', new RestControllerMoBuildStrategy(), diStrategy)
        addClass(FACADE, 'Facade')
        addComponent(FACADE_IMPL, 'FacadeImpl', FacadeEmptyImplBuildStrategy.instance)

        setDataMapper(JavaDataTypeMappers.REST_DATA_TYPE_MAPPER, REST_DTO, REST_REQUEST_DTO, REST_CONTROLLER_MO)
    }

    SpringRestInAdapterConfigurator setRootPackage(String rootPackage) {
        super.setRootPackage(rootPackage)
        setPackage(rootPackage + '.application', FACADE_IMPL)
        setPackage(rootPackage + '.ports.in.rest.dtos', DTO, ENUM, REQUEST_DTO, RESPONSE_DTO)
        setPackage(rootPackage + '.ports.in.rest', FACADE)
        setPackage(rootPackage + '.adapters.in.rest.dtos', REST_DTO, REST_REQUEST_DTO)
        setPackage(rootPackage + '.adapters.in.rest.mappers', REST_MAPPER, REST_REQUEST_MAPPER)
        setPackage(rootPackage + '.adapters.in.rest.validators', REST_DTO_VALIDATOR)
        if (moduleParams[MULTI_OPERATION_CONTROLLER_STRATEGY]) {
            setPackage(rootPackage + '.adapters.in.rest.validators', REST_REQUEST_VALIDATOR)
            setPackage(rootPackage + '.adapters.in.rest.mappers', RESPONSE_MAPPER)
            setPackage(rootPackage + '.adapters.in.rest', REST_CONTROLLER_MO)
        } else {
            setPackage(rootPackage + '.adapters.in.rest.${group}', REST_REQUEST_VALIDATOR, REST_CONTROLLER, RESPONSE_MAPPER)
        }
        setProjectDir(null)
        this
    }

}
