/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java.config

import pl.metaprogramming.codemodel.builder.java.ClassBuilderConfig
import pl.metaprogramming.codemodel.builder.java.ClassCmBuildStrategy
import pl.metaprogramming.codemodel.builder.java.dto.LombokDataBuildStrategy
import pl.metaprogramming.codemodel.builder.java.spring.DiAutowiredBuildStrategy
import pl.metaprogramming.codemodel.builder.java.spring.DiConstructorBuildStrategy
import pl.metaprogramming.codemodel.builder.java.spring.ComponentBuildStrategy
import pl.metaprogramming.codemodel.model.java.index.DataTypeMapper

import java.util.function.Consumer

class JavaModuleConfigurator<C extends JavaModuleConfigurator> {

    protected Map<Object, ClassBuilderConfig> configs = [:]
    protected Map moduleParams = [:]
    protected ClassCmBuildStrategy diStrategy = DiConstructorBuildStrategy.instance
    protected ClassCmBuildStrategy componentStrategy = ComponentBuildStrategy.instance

    JavaModuleConfig getModuleConfig() {
        new JavaModuleConfig(
                dataTypeMapper: JavaDataTypeMappers.JAVA_DATA_TYPE_MAPPER,
                configs: configs.values(),
                params: moduleParams
        )
    }

    C init() {
        (C) this
    }

    C diAutowiredStrategy() {
        diStrategy = DiAutowiredBuildStrategy.instance
        (C) this
    }

    C update(Object classType, Consumer<ClassBuilderConfig> modifier) {
        modifier.accept(configs[classType])
        (C) this
    }

    C setProjectDir(String projectDir, Object...classTypes) {
        setBaseDir((projectDir?:'') + '/src/main/java', classTypes)
    }

    C setBaseDir(String baseDir, Object...classTypes) {
        getClassesConfigs(classTypes).each {
            it.baseDir = baseDir
        }
        (C) this
    }

    C setRootPackage(String rootPackage) {
        setPackage(rootPackage)
    }

    C setPackage(String packageName, Object...classTypes) {
        getClassesConfigs(classTypes).each {
            it.packageName = packageName
        }
        (C) this
    }

    C setDataMapper(DataTypeMapper dataTypeMapper, Object...classTypes) {
        getClassesConfigs(classTypes).each {
            it.dataTypeMapper = dataTypeMapper
        }
        (C) this
    }

    C removeClass(def classType) {
        configs.remove(classType)
        (C) this
    }

    C addClass(ClassBuilderConfig config) {
        config.baseDir = config.baseDir ?: '/src/main/java'
        configs.put(config.classType, config)
        (C) this
    }

    C addClass(def classType, String suffix, ClassCmBuildStrategy...strategies) {
        addClass(classType, suffix, strategies.toList())
    }

    C addClass(def classType, String suffix, List<ClassCmBuildStrategy> strategies) {
        addClass(prepareClassConfig(classType, suffix, strategies))
    }

    C addFixedClass(def classType, String className, ClassCmBuildStrategy...strategies) {
        addClass(prepareFixedClass(classType, className, strategies))
    }

    C addFixedClass(def classType, String className, List dependencies = null) {
        addClass(new ClassBuilderConfig(
                classType: classType,
                className: className,
                gspTemplate: "/pl/metaprogramming/codegen/java/${className}.java.gsp",
                dependencies: dependencies
        ))
    }

    C addLombokData(ClassBuilderConfig classBuilderConfig) {
        classBuilderConfig.strategies = classBuilderConfig.strategies + LombokDataBuildStrategy.instance
        addClass(classBuilderConfig)
    }

    C addLombokData(def classType, String suffix, ClassCmBuildStrategy...strategies) {
        addLombokData(prepareClassConfig(classType, suffix, strategies.toList()))
    }

    C addComponent(ClassBuilderConfig classBuilderConfig) {
        classBuilderConfig.strategies = classBuilderConfig.strategies + [componentStrategy, diStrategy]
        addClass(classBuilderConfig)
    }

    C addComponent(def classType, String suffix, ClassCmBuildStrategy...strategies) {
        addComponent(prepareClassConfig(classType, suffix, strategies.toList()))
    }

    private Collection<ClassBuilderConfig> getClassesConfigs(Object...classTypes) {
        classTypes ? configs.findAll { classTypes.contains(it.key) }.collect { it.value} : configs.values()
    }

    protected static ClassBuilderConfig prepareClassConfig(def classType, String suffix, List<ClassCmBuildStrategy> strategies) {
        new ClassBuilderConfig(
                classType: classType,
                classNameSuffix: suffix,
                strategies: strategies
        )
    }

    protected static ClassBuilderConfig prepareFixedClass(def classType, String className, ClassCmBuildStrategy...strategies) {
        new ClassBuilderConfig(classType: classType, className: className, strategies: strategies.toList())
    }
}
