/*
 * Copyright (c) 2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java

import pl.metaprogramming.codemodel.formatter.CodeBuffer
import pl.metaprogramming.codemodel.formatter.JavaCodeFormatter
import pl.metaprogramming.codemodel.model.java.*
import pl.metaprogramming.metamodel.model.data.EnumType

import static pl.metaprogramming.codemodel.builder.java.ClassType.ENUM_VALUE_INTERFACE
import static pl.metaprogramming.codemodel.model.java.JavaDefs.ACCESS_PRIVATE
import static pl.metaprogramming.codemodel.model.java.JavaDefs.T_STRING

class EnumBuildStrategy extends ClassCmBuildStrategy<EnumType> {
    static final List<Character> ALLOWED_FIRST_LETTERS = ['_' as char]

    @Override
    void makeImplementation(ClassCmBuildHelper<EnumType> builder) {
        builder.addFields(new FieldCm(name: 'value', type: T_STRING))
        builder.addMethods(makeMethods(builder.builtClass))
        builder.addInterfaces(ENUM_VALUE_INTERFACE)
        builder.setEnums(builder.metaModel.allowedValues.collect { toItem(builder, it) })
    }

    EnumItemCm toItem(ClassCmBuildHelper<EnumType> builder, String value) {
        new EnumItemCm(
                name: toEnumItemName(value),
                value: ValueCm.escaped(value),
                description: builder.metaModel.getDescription(value)
        )
    }

    private static MethodCm[] makeMethods(ClassCd enumClass) {
        [new MethodCm(
                name: enumClass.className,
                accessModifier: ACCESS_PRIVATE,
                params: [new FieldCm(name: 'value', type: T_STRING)],
                implBody: 'this.value = value;'
        ),
         new MethodCm(
                 name: 'getValue',
                 resultType: T_STRING,
                 implBody: 'return value;'
         ),
         new MethodCm(
                 name: 'fromValue',
                 resultType: enumClass,
                 staticModifier: true,
                 params: [new FieldCm(name: 'value', type: T_STRING)],
                 implBody: toEnumMethodImpBody(enumClass.className)
         ),
        ] as MethodCm[]
    }

    private static String toEnumMethodImpBody(String enumType) {
        CodeBuffer buf = new CodeBuffer(JavaCodeFormatter.NEW_LINE, JavaCodeFormatter.TAB)
        buf.add("for ($enumType e : ${enumType}.values()) {")
        buf.indent(1).newLine('if (e.value.equals(value)) return e;')
        buf.indent().newLine('}')
        buf.newLine("throw new IllegalArgumentException(\"Unknonw $enumType: \" + value);")
        buf.toString()
    }

    private static String toEnumItemName(String value) {
        StringBuilder buf= new StringBuilder()
        Character prevC = null
        for (int idx = 0; idx < value.length(); idx++){
            def c = value.charAt(idx)
            if ((idx == 0 && !c.isLetter() && !ALLOWED_FIRST_LETTERS.contains(c))
                    || (c.isUpperCase() && prevC?.isLowerCase())){
                buf.append('_')
            }
            if (c.isLowerCase()) {
                buf.append(c.toUpperCase())
            } else if (c.isUpperCase() || c.isDigit()) {
                buf.append(c)
            } else {
                buf.append('_')
            }
            prevC = Character.valueOf(c)
        }
        buf.toString()
    }

}
