/*
 * Copyright (c) 2018,2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.java

import groovy.text.SimpleTemplateEngine
import pl.metaprogramming.codemodel.builder.java.method.MethodCmBuilder
import pl.metaprogramming.codemodel.model.java.*
import pl.metaprogramming.codemodel.model.java.index.ClassIndex
import pl.metaprogramming.metamodel.model.data.DataType

import java.util.function.Predicate

class ClassCmBuilderImpl<T> implements ClassCmBuilder<T>, ClassCmBuildHelper<T> {

    static List<AnnotationCm> GENERATED_ANNOTATIONS = [new AnnotationCm('javax.annotation.Generated', [value: ValueCm.escaped('pl.metaprogramming.codegen')])]

    ClassIndex classIndex

    T metaModel
    String metaModelName
    String packageName

    ClassBuilderConfig config

    ClassCm classCm
    List<MethodCmBuilder> methodBuilders = []

    ClassCmBuilder<T> make() {
        makeDeclaration()
        makeImplementation()
        this
    }

    ClassCmBuilder<T> makeDeclaration() {
        classCm = new ClassCm(
                packageName: makePackageName(),
                className: makeClassName()
        )
        addAnnotations(GENERATED_ANNOTATIONS)
        classIndex.put(classCm, metaModel, config.classType)
        config.strategies.each { it.makeDeclaration(this) }
        this
    }

    ClassCmBuilder<T> makeImplementation() {
        config.strategies.each { it.makeImplementation(this) }
        this
    }

    private String makePackageName() {
        if (packageName) {
            return packageName
        }
        if (config.packageName.contains('$')) {
            def metaModelMap = metaModel.properties
            return new SimpleTemplateEngine()
                    .createTemplate(config.packageName)
                    .make(metaModelMap).toString()
        }
        config.packageName
    }

    private String makeClassName() {
        (config.classNamePrefix ?: '') + metaModelName + (config.classNameSuffix ?: '')
    }

    def getClassType() {
        config.classType
    }

    ClassCd getBuiltClass() {
        classCm
    }

    void addImport(ClassCd classCd) {
        classCm.imports.add(classCd.getCanonicalName())
    }

    void addImportStatic(ClassCd classCd) {
        classCm.imports.add("static ${classCd.getCanonicalName()}.*".toString())
    }

    void addImports(List<String> imports) {
        classCm.imports.addAll(imports)
    }

    void addImports(Object...imports) {
        imports.each {
            if (it instanceof String) {
                classCm.imports.add(it as String)
            } else if (it instanceof ClassCd) {
                classCm.imports.add((it as ClassCd).canonicalName)
            }
        }
    }

    void setComment(String comment) {
        classCm.description = comment
    }

    void addAnnotation(AnnotationCm annotation) {
        classCm.annotations.add(annotation)
    }

    void addAnnotations(List<AnnotationCm> annotations) {
        classCm.annotations.addAll(annotations)
    }

    void setInterface() {
        classCm.isInterface = true
    }

    void setEnums(List<EnumItemCm> enums) {
        classCm.isEnum = true
        classCm.enumItems = enums
    }

    void setGenericParams(ClassCd...genericParams) {
        classCm.genericParams = Arrays.asList(genericParams)
    }

    ClassCd getSuperClass() {
        return classCm.extend
    }

    void setSuperClass(Object classType, def metaModel = null) {
        def superClass = getClass(classType, metaModel)
        check(classCm.extend == null, "Can't set super class '$superClass', super class already set to '$classCm.extend'")
        classCm.extend = superClass
    }

    void addMethods(MethodCm...methods) {
        methods.each { it.ownerClass = classCm }
        classCm.methods.addAll(methods)
    }

    List<MethodCm> getMethods() {
        classCm.methods
    }

    void addInheritMapper(List fromClassTypes, Object toClassType, String methodName) {
        classIndex.putMappers(new MethodCm(
                ownerClass: classCm,
                name: methodName,
                resultType: getClass(toClassType),
                params: fromClassTypes.collect { new FieldCm(type: getClass(it)) }
        ))
    }

    void addMapper(MethodCmBuilder methodCmBuilder) {
        methodBuilders.add(methodCmBuilder)
        classIndex.putMappers(methodCmBuilder.makeDeclaration())
    }

    void addMappers(MethodCm...methods) {
        addMethods(methods)
        classIndex.putMappers(methods)
    }


    void addInterfaces(Object...interfaces) {
        interfaces.each {
            classCm.interfaces.add(getClass(it))
        }
    }

    List<ClassCd> getInterfaces() {
        classCm.interfaces
    }

    void addFields(FieldCm...fields) {
        fields.each { classCm.addField(it) }
    }

    List<FieldCm> findFields(Predicate<FieldCm> predicate) {
        classCm.fields.findAll { predicate.test(it) }
    }

    ClassCd getClass(Object classType, def metaModel = metaModel, boolean optional = false) {
        if (classType instanceof String) {
            return new ClassCd(classType)
        }
        if (classType instanceof ClassCd) {
            return classType
        }
        if (classType instanceof DataType) {
            return config.dataTypeMapper.map(classType)
        }
        if (metaModel instanceof DataType) {
            return findDataSchemaClass(metaModel, classType)
        }
        def classCd = classIndex.getClass(classType, metaModel)
        check(classCd != null || optional, "Can't find Class ($classType) for $metaModel")
        classCd
    }

    Optional<ClassCd> getClassOptional(Object classType, def metaModel = metaModel) {
        Optional.ofNullable(getClass(classType, metaModel, true))
    }

    ClassCd getGenericClass(Object classType, List genericClassTypes) {
        new ClassCd(getClass(classType), genericClassTypes.collect { getClass(it)} )
    }

    ClassCd findDataSchemaClass(DataType dataType, def classType) {
        assert dataType != null
        def classCd
        try {
            classCd = classIndex.getClassForDataType(dataType, classType, config.dataTypeMapper)
        } catch (Exception e) {
            panic("Can't find class for $dataType (of type $classType)", e)
        }
        check(classCd != null, "Can't find class for $dataType (of type $classType)")
        classCd
    }

    MethodCm findMapper(ClassCd resultType, List<ClassCd> params) {
        def result = classIndex.getMapper(resultType, params)
        check(result != null, "Can't find mapper $params -> $resultType")
        result
    }


    FieldCm injectDependency(ClassCd toInject) {
        check(toInject != null, "toInject can't be null")
        FieldCm injectField = classCm.fields.find { it.type == toInject }
        if (injectField == null) {
            injectField = new FieldCm(type: toInject , name: toInject.className.uncapitalize())
            classCm.fields.add(injectField)
        }
        injectField
    }


    ClassCm extractInterface(ClassBuilderConfig config) {
        def interfaceCm = new ClassCm(
                packageName: config.packageName,
                className: metaModelName + config.classNameSuffix,
                annotations: GENERATED_ANNOTATIONS.collect() as List<AnnotationCm>,
                isInterface: true,
        )
        interfaceCm.methods = classCm.methods.findAll { it.accessModifier == 'public' }.collect {
            new MethodCm(
                    name: it.name,
                    resultType: it.resultType,
                    params: it.params,
                    annotations: it.annotations,
                    ownerClass: interfaceCm,
            )
        }
        def interfaceBuilder = new ClassCmBuilderImpl<T>(
                classCm: interfaceCm,
                config: config,
                classIndex: classIndex,
                metaModel: metaModel,
        )
        config.strategies?.each { it.makeImplementation(interfaceBuilder) }
        classCm.interfaces.add(interfaceCm)
        classIndex.put(interfaceCm, metaModel, config.classType)
        classIndex.putMappers(interfaceCm.methods as MethodCm[])
        interfaceCm
    }

    private void check(boolean isOk, String errorMessage) {
        if (!isOk) {
            panic(errorMessage)
        }
    }

    private void panic(String message, Exception cause = null) {
        classIndex.printIndex()
        classIndex.printMapperIndex()
        throw new RuntimeException("Can't build $classCm ($config.classType)\n$message", cause)
    }

    @Override
    String toString() {
        "Builder of $config.classType $metaModelName"
    }
}
