/*
 * Copyright (c) 2018,2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codemodel.builder.groovy.spock

import pl.metaprogramming.codemodel.builder.CodeGenerationTask
import pl.metaprogramming.codemodel.builder.CustomModuleBuilder
import pl.metaprogramming.codemodel.model.groovy.SpockRestSpecCm
import pl.metaprogramming.metamodel.model.data.*
import pl.metaprogramming.metamodel.model.rest.Operation
import pl.metaprogramming.metamodel.model.rest.Parameter
import pl.metaprogramming.metamodel.model.rest.RestApi
import pl.metaprogramming.metamodel.model.rest.enums.ParamLocation

import static pl.metaprogramming.metamodel.model.data.DataTypeCode.*
import static pl.metaprogramming.metamodel.model.rest.enums.ParamLocation.*

class SpockRestServicesTestsMapper extends CustomModuleBuilder<RestApi> {

    SpockRestServicesTestsMapperConfig cfg

    SpockRestServicesTestsMapper(Map<String, String> params) {
        cfg = new SpockRestServicesTestsMapperConfig(params)
    }

    List<CodeGenerationTask> make(RestApi model) {
        List<CodeGenerationTask> result = []
        def specModelList = mapSpec(model)
        result.addAll(make(specModelList, cfg.destCodeDir, cfg.specTemplate))
        result
    }

    List<SpockRestSpecCm> mapSpec(RestApi model) {
        List<SpockRestSpecCm> result = []
        model.operations.each { resourceCode, operations ->
            operations.each { operationCode, operationModel ->
                result.add(new SpockRestSpecCm(
                        restClientClass: "${cfg.testPackage}.utils.${cfg.clientClassName}",
                        className: "${operationCode.capitalize()}Spec",
                        packageName: "${cfg.testPackage}.${resourceCode.uncapitalize()}",
                        operationCode: operationCode,
                        operationType: operationModel.type.name().toLowerCase(),
                        path: operationModel.path,
                        headerParams: getParams(operationModel.parameters, HEADER),
                        pathParams: getParams(operationModel.parameters, PATH),
                        queryParams: getParams(operationModel.parameters, QUERY),
                        responseStatuses: operationModel.responses.collect { it.status },
                        requestTemplate: prepareRequestTemplate(operationModel),
                        hasBody: operationModel.type.hasBody(),
                ))
            }
        }
        result
    }

    static List<String> getParams(List<Parameter> params, ParamLocation location) {
        params.findAll { it.location == location }.collect { it.name }
    }

    Map prepareRequestTemplate(Operation model) {
        Map result = !model.parameters ? [:] : model.parameters.collectEntries {
            [it.name, getTemplateValue(it)]
        }
        if (model.requestBody != null) {
            def body = getTemplateValue('body', model.requestBody)
            if (body instanceof Map && result.keySet().disjoint(body.keySet())) {
                result.putAll(body)
            } else {
                result.put('requestBody', body)
            }
        }
        result
    }

    def getTemplateValue(String code, DataType dataType) {
        getTemplateValue(new DataSchema(code: code, dataType: dataType))
    }

    def getTemplateValue(DataSchema schema) {
        switch (schema.dataType.typeCode) {
            case OBJECT:
                return schema.objectType.fields.collectEntries {
                    [(it.code): getTemplateValue(it)]
                }
            case ARRAY:
                return [getTemplateValue(schema.arrayType.itemsSchema)]
            case STRING:
                return "'_${schema.code}_'"
            case ENUM:
                EnumType s = (EnumType) schema.dataType
                return s.allowedValues?.size() > 0 ? "'${s.allowedValues[0]}'" : "'_${s.dictionaryRef}_'"
            case DATE_TIME:
                return "'${new Date().format(schema.format)}'"
            case BINARY:
                return null
            case DECIMAL:
            case FLOAT:
            case DOUBLE:
                return "'1.1'"
            case INT32:
            case INT64:
                return "'1'"
            default:
                throw new RuntimeException("Unknown java type for $schema")
        }

    }
}
