/*
 * Copyright (c) 2018,2019 Dawid Walczak.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package pl.metaprogramming.codegen


import pl.metaprogramming.codemodel.builder.java.config.CommonsModuleConfigurator
import pl.metaprogramming.codemodel.builder.java.config.JavaModuleConfig
import pl.metaprogramming.codemodel.builder.java.config.JavaModuleConfigurator
import pl.metaprogramming.codemodel.builder.java.config.SpringRestInAdapterConfigurator
import pl.metaprogramming.codemodel.builder.java.config.SpringWsOutAdapterConfigurator
import pl.metaprogramming.codemodel.builder.java.module.JavaModuleBuilder
import pl.metaprogramming.codemodel.builder.java.module.ModuleWithoutModelBuilder
import pl.metaprogramming.codemodel.builder.java.module.SpringRestInAdapterBuilder
import pl.metaprogramming.codemodel.builder.java.module.SpringWsOutAdapterBuilder
import pl.metaprogramming.codemodel.model.java.index.ClassIndex
import pl.metaprogramming.metamodel.model.rest.RestApi
import pl.metaprogramming.metamodel.model.wsdl.WsdlApi

import java.util.function.Consumer

class CodeGenerationConfigurator {

    CodeGenerationConfig cfg = new CodeGenerationConfig()

    List<ClassIndex> commonDependencies = []
    Map<String, ModuleConfiguration> modules = [:]

    Consumer<JavaModuleConfigurator> mainSetter = {} as Consumer

    private <T extends JavaModuleConfigurator> JavaModuleConfig getModuleConfig(T configurator, String rootPackage, String projectDir = null) {
        mainSetter.accept(configurator)
        configurator.init()
                .setRootPackage(rootPackage)
                .setProjectDir(projectDir)
                .getModuleConfig()
    }

    def newSpringRestInAdapterConfigurator() {
        def result = new SpringRestInAdapterConfigurator()
        mainSetter.accept(result)
        result
    }

    def setBaseDir(File baseDir) {
        cfg.baseDir = baseDir
        this
    }

    def setIndexFile(File indexFile) {
        cfg.indexFile = indexFile
        this
    }

    def javaModule(String moduleName, Object metaModel, JavaModuleBuilder builder, boolean isCommon = false) {
        assert moduleName, 'moduleName should not be empty'
        assert !modules.containsKey(moduleName), "moduleName '$moduleName' is already used"

        builder.dependsOn(collectDependencies(metaModel))
        def module = new ModuleConfiguration(moduleName, metaModel, builder)
        modules.put(moduleName, module)
        cfg.modules.add(module)
        if (isCommon) {
            commonDependencies.add(builder.classIndex)
        }
        this
    }

    def commonsModule(String moduleName, String rootPackage = moduleName, String projectDir = null) {
        def builder = new ModuleWithoutModelBuilder(getModuleConfig(new CommonsModuleConfigurator(), rootPackage, projectDir))
        javaModule(rootPackage, null, builder, true)
        this
    }

    def springRestInAdapter(String moduleName, RestApi d, String rootPackage) {
        javaModule(moduleName, d, new SpringRestInAdapterBuilder(
                getModuleConfig(new SpringRestInAdapterConfigurator(), rootPackage)))
    }

    def springRestInAdapter(String moduleName, RestApi d, SpringRestInAdapterConfigurator configurator) {
        javaModule(moduleName, d, new SpringRestInAdapterBuilder(configurator.moduleConfig))
    }

    def springWsOutAdapter(String component, WsdlApi d, SpringWsOutAdapterConfigurator configurator) {
        javaModule(component, d, new SpringWsOutAdapterBuilder(configurator.moduleConfig))
    }

    ClassIndex[] collectDependencies(def metaModel) {
        ClassIndex[] dependencies = []
        if (metaModel instanceof RestApi) {
            dependencies = metaModel.dependsOn.collect { otherComponent ->
                def c = cfg.modules.find { it.metaModel == otherComponent }
                assert c, "Nieznany komponent $otherComponent"
                ((JavaModuleBuilder) c.moduleBuilder).classIndex
            }
        }
        dependencies + commonDependencies
    }
}
