package pl.codewise.commons.aws.cqrs.operations.s3;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.util.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import pl.codewise.commons.aws.cqrs.discovery.S3Discovery;
import pl.codewise.commons.aws.cqrs.utils.FileSystem;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.util.List;

public class S3Operations {

    private static final String APPLICATION_OCTET_STREAM = "application/octet-stream";
    private static final String TEXT_PLAIN = "text/plain";

    private static final Logger logger = LoggerFactory.getLogger(S3Operations.class);

    private final AmazonS3 amazonS3;
    private final S3Discovery s3Discovery;
    private final FileSystem fileSystem;
    private final int s3OperationsPutFileRetryCount;
    private final int s3OperationsGetFileRetryCount;

    public S3Operations(AmazonS3 amazonS3, S3Discovery s3Discovery, FileSystem fileSystem,
            int s3OperationsPutFileRetryCount,
            int s3OperationsGetFileRetryCount) {
        this.amazonS3 = amazonS3;
        this.s3Discovery = s3Discovery;
        this.fileSystem = fileSystem;
        this.s3OperationsPutFileRetryCount = s3OperationsPutFileRetryCount;
        this.s3OperationsGetFileRetryCount = s3OperationsGetFileRetryCount;
    }

    public PutResult putText(List<String> bucketNames, String key, String text) {
        Path content = null;
        try {
            content = fileSystem.createTempFileWithText(key, text);
            return putFile(bucketNames, key, content.toFile(), TEXT_PLAIN);
        } catch (IOException e) {
            throw new IllegalArgumentException(e);
        } finally {
            fileSystem.deleteFileIfExists(content);
        }
    }

    public String getText(String bucketName, String key) {
        for (int i = 0; i < s3OperationsGetFileRetryCount; i++) {
            String content = tryGetFile(key, bucketName);
            if (content != null) {
                return content;
            }
        }
        return null;
    }

    public InputStream getStream(String bucketName, String key) {
        for (int i = 0; i < s3OperationsGetFileRetryCount; i++) {
            InputStream contentStream = tryGetStream(key, bucketName);
            if (contentStream != null) {
                return contentStream;
            }
        }
        return null;
    }

    public PutResult putFile(List<String> bucketNames, String key, File file) {
        return putFile(bucketNames, key, file, APPLICATION_OCTET_STREAM);
    }

    public void deleteObject(String bucket, String key) {
        amazonS3.deleteObject(bucket, key);
        logger.info("Object deleted! Bucket <{}> | Object-key <{}>", bucket, key);
    }

    public void deleteRecursively(String bucket, String prefix) {
        String[] keys = s3Discovery.listObjects(bucket, prefix).toArray(new String[0]);
        DeleteObjectsRequest request = new DeleteObjectsRequest(bucket)
                .withKeys(keys);
        amazonS3.deleteObjects(request);
    }

    private PutResult putFile(List<String> bucketNames, String key, File file, String contentType) {
        PutResult putResult = new PutResult();

        for (String bucketName : bucketNames) {
            ResultWithMessage result = putFileWithRetry(key, file, bucketName, contentType);
            putResult.collect(result.successful, bucketName, result.message);
        }

        return putResult;
    }

    private ResultWithMessage putFileWithRetry(String key, File file, String bucketName, String contentType) {
        ResultWithMessage result = new ResultWithMessage(false, null);
        for (int i = 0; i < s3OperationsPutFileRetryCount; i++) {
            result = tryPutFile(key, file, bucketName, contentType);
            if (result.successful) {
                break;
            }
        }
        return result;
    }

    private ResultWithMessage tryPutFile(String key, File file, String bucketName, String contentType) {
        try {
            PutObjectRequest putObjectRequest = new PutObjectRequest(bucketName, key, file);
            ObjectMetadata metadata = new ObjectMetadata();
            metadata.setContentType(contentType);
            putObjectRequest.withMetadata(metadata);
            amazonS3.putObject(putObjectRequest);
            return new ResultWithMessage(true, null);
        } catch (Exception e) {
            logger.error(String.format("Could not put object to %s/%s", bucketName, key), e);
            return new ResultWithMessage(false, e.getMessage());
        }
    }

    private String tryGetFile(String key, String bucketName) {
        try {
            GetObjectRequest getObjectRequest = new GetObjectRequest(bucketName, key);
            S3Object s3Object = amazonS3.getObject(getObjectRequest);
            try (InputStream content = s3Object.getObjectContent()) {
                return IOUtils.toString(content);
            }
        } catch (Exception e) {
            logger.error(String.format("Could not get object to %s/%s", bucketName, key), e);
        }
        return null;
    }

    private InputStream tryGetStream(String key, String bucketName) {
        try {
            GetObjectRequest getObjectRequest = new GetObjectRequest(bucketName, key);
            S3Object s3Object = amazonS3.getObject(getObjectRequest);
            return s3Object.getObjectContent();
        } catch (Exception e) {
            logger.error(String.format("Could not get stream to %s/%s", bucketName, key), e);
        }
        return null;
    }

    private static class ResultWithMessage {

        private final boolean successful;
        private final String message;

        public ResultWithMessage(boolean successful, String message) {
            this.successful = successful;
            this.message = message;
        }
    }
}
