package pl.codewise.commons.aws.cqrs.operations;

import com.amazonaws.services.elasticloadbalancing.AmazonElasticLoadBalancing;
import com.amazonaws.services.elasticloadbalancing.model.AddTagsRequest;
import com.amazonaws.services.elasticloadbalancing.model.ConfigureHealthCheckRequest;
import com.amazonaws.services.elasticloadbalancing.model.CreateLoadBalancerRequest;
import com.amazonaws.services.elasticloadbalancing.model.DeleteLoadBalancerRequest;
import com.amazonaws.services.elasticloadbalancing.model.DeregisterInstancesFromLoadBalancerRequest;
import com.amazonaws.services.elasticloadbalancing.model.HealthCheck;
import com.amazonaws.services.elasticloadbalancing.model.Instance;
import com.amazonaws.services.elasticloadbalancing.model.Listener;
import com.amazonaws.services.elasticloadbalancing.model.RegisterInstancesWithLoadBalancerRequest;
import com.amazonaws.services.elasticloadbalancing.model.Tag;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import pl.codewise.commons.aws.cqrs.model.AwsInstance;
import pl.codewise.commons.aws.cqrs.model.ec2.AwsElasticLoadBalancer;
import pl.codewise.commons.aws.cqrs.model.ec2.AwsLoadBalancerHealthCheckParameters;
import pl.codewise.commons.aws.cqrs.model.ec2.AwsLoadBalancerListener;
import pl.codewise.commons.aws.cqrs.model.ec2.AwsResourceTag;
import pl.codewise.commons.aws.cqrs.model.ec2.LoadBalancerParameters;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.util.Arrays.stream;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Stream.concat;

public class ClassicLoadBalancingOperations {

    private static final Logger log = LoggerFactory.getLogger(ClassicLoadBalancingOperations.class);

    private final AmazonElasticLoadBalancing amazonElasticLoadBalancing;

    public ClassicLoadBalancingOperations(AmazonElasticLoadBalancing amazonElasticLoadBalancing) {
        this.amazonElasticLoadBalancing = amazonElasticLoadBalancing;
    }

    public void tag(String loadBalancerName, List<AwsResourceTag> tags) {
        AddTagsRequest request = new AddTagsRequest()
                .withLoadBalancerNames(loadBalancerName)
                .withTags(toTags(tags));
        amazonElasticLoadBalancing.addTags(request);
    }

    public AwsElasticLoadBalancer createLoadBalancer(LoadBalancerParameters parameters) {
        CreateLoadBalancerRequest request = prepareCreateLoadBalancerRequest(parameters);
        amazonElasticLoadBalancing.createLoadBalancer(request);
        log.info("Load balancer <{}> created!", parameters.getName());
        return new AwsElasticLoadBalancer(parameters.getName());
    }

    public void deleteLoadBalancer(String loadBalancerName) {
        DeleteLoadBalancerRequest request = new DeleteLoadBalancerRequest(loadBalancerName);
        amazonElasticLoadBalancing.deleteLoadBalancer(request);
        log.info("Elastic load balancer <{}> deleted!", request.getLoadBalancerName());
    }

    public void configureHealthCheckForLoadBalancer(String name, AwsLoadBalancerHealthCheckParameters parameters) {
        ConfigureHealthCheckRequest request = prepareConfigureHealthCheckRequest(name, parameters);
        amazonElasticLoadBalancing.configureHealthCheck(request);
        log.info("Health check configured! Load balancer <{}> | Parameters <{}>", name, parameters);
    }

    public void deregisterInstances(String loadBalancerName, AwsInstance... instances) {
        amazonElasticLoadBalancing.deregisterInstancesFromLoadBalancer(
                new DeregisterInstancesFromLoadBalancerRequest(loadBalancerName, toInstances(instances)));
    }

    public void registerInstances(String loadBalancerName, AwsInstance... instances) {
        amazonElasticLoadBalancing.registerInstancesWithLoadBalancer(
                new RegisterInstancesWithLoadBalancerRequest(loadBalancerName, toInstances(instances)));
    }

    private ConfigureHealthCheckRequest prepareConfigureHealthCheckRequest(String loadBalancer,
            AwsLoadBalancerHealthCheckParameters parameters) {
        return new ConfigureHealthCheckRequest()
                .withLoadBalancerName(loadBalancer)
                .withHealthCheck(prepareHealthCheck(parameters));
    }

    private HealthCheck prepareHealthCheck(AwsLoadBalancerHealthCheckParameters parameters) {
        return new HealthCheck()
                .withTarget(parameters.getTarget())
                .withTimeout(parameters.getTimeout())
                .withInterval(parameters.getInterval())
                .withUnhealthyThreshold(parameters.getUnhealthyThreshold())
                .withHealthyThreshold(parameters.getHealthyThreshold());
    }

    private CreateLoadBalancerRequest prepareCreateLoadBalancerRequest(LoadBalancerParameters parameters) {
        CreateLoadBalancerRequest request = new CreateLoadBalancerRequest()
                .withLoadBalancerName(parameters.getName())
                .withListeners(toRawListeners(parameters.getListeners()))
                .withSubnets(parameters.getSubnets())
                .withSecurityGroups(parameters.getSecurityGroups())
                .withTags(toTags(parameters));
        if (parameters.isInternal()) {
            request.withScheme("internal");
        }
        return request;
    }

    private Collection<Listener> toRawListeners(Collection<AwsLoadBalancerListener> awsLoadBalancerListeners) {
        return awsLoadBalancerListeners.stream()
                .map(listener -> new Listener(listener.protocol, listener.loadBalancerPort, listener.instancePort)
                        .withSSLCertificateId(listener.sslCertificateArn))
                .collect(Collectors.toList());
    }

    private List<Instance> toInstances(AwsInstance... instances) {
        return stream(instances).map(this::toInstance).collect(toList());
    }

    private Instance toInstance(AwsInstance awsInstance) {
        return new Instance(awsInstance.getInstanceId());
    }

    private Collection<Tag> toTags(LoadBalancerParameters parameters) {
        return concat(
                parameters.getTags().stream().map(this::toTag),
                Stream.of(nameTag(parameters.getName())))
                .collect(toList());
    }

    private List<Tag> toTags(List<AwsResourceTag> tags) {
        return tags.stream().map(this::toTag).collect(toList());
    }

    private Tag toTag(AwsResourceTag tag) {
        return new Tag().withKey(tag.key()).withValue(tag.value());
    }

    private Tag nameTag(String name) {
        return new Tag().withKey("Name").withValue(name);
    }
}
