package pl.codewise.commons.aws.cqrs.operations;

import com.amazonaws.services.autoscaling.AmazonAutoScaling;
import com.amazonaws.services.autoscaling.model.Alarm;
import com.amazonaws.services.autoscaling.model.AutoScalingGroup;
import com.amazonaws.services.autoscaling.model.CreateAutoScalingGroupRequest;
import com.amazonaws.services.autoscaling.model.CreateOrUpdateTagsRequest;
import com.amazonaws.services.autoscaling.model.DeleteAutoScalingGroupRequest;
import com.amazonaws.services.autoscaling.model.DeleteTagsRequest;
import com.amazonaws.services.autoscaling.model.DescribeAutoScalingGroupsRequest;
import com.amazonaws.services.autoscaling.model.DescribeAutoScalingGroupsResult;
import com.amazonaws.services.autoscaling.model.DetachInstancesRequest;
import com.amazonaws.services.autoscaling.model.DetachLoadBalancersRequest;
import com.amazonaws.services.autoscaling.model.PutScalingPolicyRequest;
import com.amazonaws.services.autoscaling.model.PutScalingPolicyResult;
import com.amazonaws.services.autoscaling.model.ResumeProcessesRequest;
import com.amazonaws.services.autoscaling.model.ScalingPolicy;
import com.amazonaws.services.autoscaling.model.SetInstanceProtectionRequest;
import com.amazonaws.services.autoscaling.model.SuspendProcessesRequest;
import com.amazonaws.services.autoscaling.model.Tag;
import com.amazonaws.services.autoscaling.model.UpdateAutoScalingGroupRequest;
import com.amazonaws.services.cloudwatch.AmazonCloudWatch;
import com.amazonaws.services.cloudwatch.model.DescribeAlarmsRequest;
import com.amazonaws.services.cloudwatch.model.DescribeAlarmsResult;
import com.amazonaws.services.cloudwatch.model.Dimension;
import com.amazonaws.services.cloudwatch.model.MetricAlarm;
import com.amazonaws.services.cloudwatch.model.PutMetricAlarmRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import pl.codewise.commons.aws.cqrs.discovery.AutoScalingDiscovery;
import pl.codewise.commons.aws.cqrs.discovery.ClassicLoadBalancingDiscovery;
import pl.codewise.commons.aws.cqrs.model.AwsAutoScalingGroup;
import pl.codewise.commons.aws.cqrs.model.AwsInstance;
import pl.codewise.commons.aws.cqrs.model.ec2.AwsAutoScalingTag;
import pl.codewise.commons.aws.cqrs.utils.Awaitilities;

import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import static java.lang.String.format;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static pl.codewise.commons.aws.cqrs.utils.Functions.overrideIfNotEmpty;
import static pl.codewise.commons.aws.cqrs.utils.Functions.overrideIfNotNull;

public class AutoScalingOperations {

    private static final Logger log = LoggerFactory.getLogger(AutoScalingOperations.class);
    private static final String ALARM_NOTIFICATION = "AlarmNotification";

    private final String region;
    private final AmazonAutoScaling autoScaling;
    private final AmazonCloudWatch cloudWatch;
    private final AutoScalingDiscovery autoScalingDiscovery;
    private final ClassicLoadBalancingDiscovery classicLoadBalancingDiscovery;
    private final Awaitilities awaitilities;
    private final long waitForNoInService;
    private final long waitForAutoScalingShutdown;
    private final long waitForInstanceStartup;
    private final Duration pollInterval;

    public AutoScalingOperations(String region, AmazonAutoScaling autoScaling, AmazonCloudWatch cloudWatch,
            AutoScalingDiscovery autoScalingDiscovery,
            ClassicLoadBalancingDiscovery classicLoadBalancingDiscovery, Awaitilities awaitilities,
            long waitForNoInService, long waitForAutoScalingShutdown, long waitForInstanceStartup, long pollInterval) {
        this.region = region;
        this.autoScaling = autoScaling;
        this.autoScalingDiscovery = autoScalingDiscovery;
        this.awaitilities = awaitilities;
        this.classicLoadBalancingDiscovery = classicLoadBalancingDiscovery;
        this.waitForNoInService = waitForNoInService;
        this.waitForAutoScalingShutdown = waitForAutoScalingShutdown;
        this.waitForInstanceStartup = waitForInstanceStartup;
        this.pollInterval = Duration.ofMillis(pollInterval);
        this.cloudWatch = cloudWatch;
    }

    public AwsAutoScalingGroup create(AwsAutoScalingGroup awsAutoScalingGroupTemplate) {
        autoScaling.createAutoScalingGroup(newCreateAutoScalingGroupRequest(awsAutoScalingGroupTemplate));
        log.info("Auto scaling group <{}> created!", awsAutoScalingGroupTemplate.getLaunchConfigurationName());
        return autoScalingDiscovery.getAutoScalingGroupByName(awsAutoScalingGroupTemplate.getAutoScalingGroupName());
    }

    /**
     * copies ASG from given name to target name. right now this method is configured by {@link AwsAutoScalingGroup}.
     *
     * @param sourceAutoScalingGroupName lookup name
     * @param newAutoScalingGroupTemplate meta to copy basic configuration from
     *
     * @return created ASG
     */
    public AwsAutoScalingGroup copy(String sourceAutoScalingGroupName,
            AwsAutoScalingGroup newAutoScalingGroupTemplate) {
        AwsAutoScalingGroup sourceAutoScalingGroup =
                autoScalingDiscovery.getAutoScalingGroupByName(sourceAutoScalingGroupName);

        autoScaling.createAutoScalingGroup(
                newCreateAutoScalingGroupRequest(newAutoScalingGroupTemplate, sourceAutoScalingGroup));

        return autoScalingDiscovery.getAutoScalingGroupByName(newAutoScalingGroupTemplate.getAutoScalingGroupName());
    }

    public void copyScalingPolicies(String sourceAutoScalingGroupName,
            String targetAutoScalingGroupName, String newAlarmSuffix) {
        List<ScalingPolicy> sourceScalingPolicies =
                autoScalingDiscovery.getScalingPolicies(sourceAutoScalingGroupName).getPolicies();

        sourceScalingPolicies.forEach(policy ->
                copyScalingPolicyWithAlarms(policy, targetAutoScalingGroupName, newAlarmSuffix));
    }

    public void deleteForcefully(String autoScalingGroupName) {
        DeleteAutoScalingGroupRequest request = new DeleteAutoScalingGroupRequest()
                .withAutoScalingGroupName(autoScalingGroupName)
                .withForceDelete(true);
        autoScaling.deleteAutoScalingGroup(request);
        log.info("Auto scaling group <{}> deleted!", request.getAutoScalingGroupName());
    }

    public void setSizes(String sourceAutoScalingGroupName, int minCapacity, int desiredCapacity,
            int maxCapacity) {
        UpdateAutoScalingGroupRequest request = new UpdateAutoScalingGroupRequest()
                .withAutoScalingGroupName(sourceAutoScalingGroupName)
                .withMinSize(minCapacity)
                .withDesiredCapacity(desiredCapacity)
                .withMaxSize(maxCapacity);
        updateAutoScalingGroup(request);
    }

    public void setLaunchConfiguration(String autoScalingGroupName, String newLaunchConfigurationName) {
        UpdateAutoScalingGroupRequest request = new UpdateAutoScalingGroupRequest()
                .withAutoScalingGroupName(autoScalingGroupName)
                .withLaunchConfigurationName(newLaunchConfigurationName);
        updateAutoScalingGroup(request);
    }

    public void setTerminationPolicies(String autoScalingGroupName, Collection<String> terminationPolicies) {
        UpdateAutoScalingGroupRequest request = new UpdateAutoScalingGroupRequest()
                .withAutoScalingGroupName(autoScalingGroupName)
                .withTerminationPolicies(terminationPolicies);
        updateAutoScalingGroup(request);
    }

    public void detachInstances(String autoScalingGroupName, String... instanceIds) {
        detachInstances(autoScalingGroupName, false, instanceIds);
    }

    public void detachInstancesAndDecrementDesired(String autoScalingGroupName, String... instanceIds) {
        detachInstances(autoScalingGroupName, true, instanceIds);
    }

    public void removeAutoScalingGroup(String sourceAutoScalingGroupName) {
        setSizes(sourceAutoScalingGroupName, 0, 0, 0);
        awaitilities.awaitTillActionSucceed(waitForNoInService, pollInterval.toMillis(),
                format("%s to have no instances", sourceAutoScalingGroupName),
                () -> autoScalingDiscovery.getAutoScalingGroupRaw(sourceAutoScalingGroupName).getInstances().isEmpty());

        awaitilities.awaitTillActionSucceed(waitForNoInService, pollInterval.toMillis(),
                format("%s to have no scaling activities in progress", sourceAutoScalingGroupName),
                () -> autoScalingDiscovery.getAutoScalingActivitiesInProgress(sourceAutoScalingGroupName)
                        .isEmpty());

        autoScaling.deleteAutoScalingGroup(new DeleteAutoScalingGroupRequest()
                .withAutoScalingGroupName(sourceAutoScalingGroupName));

        awaitilities.awaitTillActionSucceed(waitForAutoScalingShutdown, pollInterval.toMillis(),
                format("%s to be completely deleted", sourceAutoScalingGroupName), () -> {
                    DescribeAutoScalingGroupsResult describeAutoScalingGroupsResult = autoScaling
                            .describeAutoScalingGroups(new DescribeAutoScalingGroupsRequest()
                                    .withAutoScalingGroupNames(sourceAutoScalingGroupName));
                    List<AutoScalingGroup> autoScalingGroups = describeAutoScalingGroupsResult
                            .getAutoScalingGroups();
                    return autoScalingGroups.stream()
                            .filter(autoScalingGroup -> !"Delete in progress".equals(autoScalingGroup.getStatus()))
                            .count() == 0;
                }
        );
    }

    public void detachLoadBalancers(String targetAutoScalingGroupName) {
        AwsAutoScalingGroup autoScalingGroup =
                autoScalingDiscovery.getAutoScalingGroupByName(targetAutoScalingGroupName);
        List<String> loadBalancerNames = autoScalingGroup.getLoadBalancerNames();
        if (loadBalancerNames.isEmpty()) {
            log.info("No load balancer associated with the Auto Scaling group {} in region {}.",
                    targetAutoScalingGroupName, region);
            return;
        }

        autoScaling
                .detachLoadBalancers(new DetachLoadBalancersRequest()
                        .withAutoScalingGroupName(targetAutoScalingGroupName)
                        .withLoadBalancerNames(loadBalancerNames));
    }

    public void waitForAsgToChangeSize(String newAutoScalingGroupName, int newAsgSize) {
        awaitilities.awaitTillActionSucceed(newAsgSize * waitForInstanceStartup, pollInterval.toMillis(),
                format("%s to have %d instances in service", newAutoScalingGroupName, newAsgSize), () -> {
                    AwsAutoScalingGroup asg = autoScalingDiscovery.getAutoScalingGroupByName(newAutoScalingGroupName);
                    List<AwsInstance> instances = asg.getAwsInstances();
                    long runningInstancesCount = instances.stream().filter(AwsInstance::isRunning).count();
                    return runningInstancesCount == newAsgSize && allInstancesRegisteredInLoadBalancers(asg);
                });
    }

    public void createOrUpdateTag(AwsAutoScalingTag tag) {
        createOrUpdateTags(singletonList(tag));
    }

    public void createOrUpdateTags(List<AwsAutoScalingTag> tags) {
        CreateOrUpdateTagsRequest createOrUpdateTagsRequest = new CreateOrUpdateTagsRequest()
                .withTags(mapAwsTags(tags));
        autoScaling.createOrUpdateTags(createOrUpdateTagsRequest);
    }

    public void deleteTag(AwsAutoScalingTag tag) {
        DeleteTagsRequest request = new DeleteTagsRequest()
                .withTags(mapAwsTags(singletonList(tag)));
        autoScaling.deleteTags(request);
    }

    public void setScaleInProtection(String autoScalingGroupName, String instanceId, boolean isProtected) {
        autoScaling.setInstanceProtection(new SetInstanceProtectionRequest()
                .withInstanceIds(instanceId)
                .withAutoScalingGroupName(autoScalingGroupName)
                .withProtectedFromScaleIn(isProtected));
    }

    public void freeze(String autoScalingGroupName) {
        autoScaling.suspendProcesses(new SuspendProcessesRequest()
                .withAutoScalingGroupName(autoScalingGroupName)
                .withScalingProcesses(ALARM_NOTIFICATION));
    }

    public void unfreeze(String autoScalingGroupName) {
        autoScaling.resumeProcesses(new ResumeProcessesRequest()
                .withAutoScalingGroupName(autoScalingGroupName)
                .withScalingProcesses(ALARM_NOTIFICATION));
    }

    private void updateAutoScalingGroup(UpdateAutoScalingGroupRequest request) {
        autoScaling.updateAutoScalingGroup(request);
    }

    private void detachInstances(String autoScalingGroupName, boolean shouldDecrementDesired, String... instanceIds) {
        autoScaling
                .detachInstances(new DetachInstancesRequest()
                        .withAutoScalingGroupName(autoScalingGroupName)
                        .withInstanceIds(instanceIds)
                        .withShouldDecrementDesiredCapacity(shouldDecrementDesired));
    }

    private void copyScalingPolicyWithAlarms(ScalingPolicy sourceScalingPolicy, String targetAutoScalingGroupName,
            String newAlarmSuffix) {
        PutScalingPolicyResult putScalingPolicyResult = autoScaling
                .putScalingPolicy(newPutScalingPolicyRequest(targetAutoScalingGroupName, sourceScalingPolicy));
        String newPolicyArn = putScalingPolicyResult.getPolicyARN();

        DescribeAlarmsResult describeAlarmsResult =
                cloudWatch.describeAlarms(newDescribeAlarmsRequest(sourceScalingPolicy.getAlarms()));

        describeAlarmsResult.getMetricAlarms().stream()
                .map(metricAlarm ->
                        newPutMetricAlarmRequest(metricAlarm, targetAutoScalingGroupName, newPolicyArn, newAlarmSuffix))
                .forEach(cloudWatch::putMetricAlarm);
    }

    private CreateAutoScalingGroupRequest newCreateAutoScalingGroupRequest(AwsAutoScalingGroup newAutoScalingGroup,
            AwsAutoScalingGroup sourceAutoScalingGroup) {
        return new CreateAutoScalingGroupRequest()
                .withAutoScalingGroupName(newAutoScalingGroup.getAutoScalingGroupName())
                .withMinSize(
                        overrideIfNotNull(newAutoScalingGroup, sourceAutoScalingGroup, AwsAutoScalingGroup::getMinSize))
                .withMaxSize(
                        overrideIfNotNull(newAutoScalingGroup, sourceAutoScalingGroup, AwsAutoScalingGroup::getMaxSize))
                .withDesiredCapacity(overrideIfNotNull(newAutoScalingGroup, sourceAutoScalingGroup,
                        AwsAutoScalingGroup::getDesiredCapacity))
                .withLaunchConfigurationName(overrideIfNotNull(newAutoScalingGroup, sourceAutoScalingGroup,
                        AwsAutoScalingGroup::getLaunchConfigurationName))
                .withDefaultCooldown(overrideIfNotNull(newAutoScalingGroup, sourceAutoScalingGroup,
                        AwsAutoScalingGroup::getDefaultCooldown))
                .withAvailabilityZones(overrideIfNotEmpty(newAutoScalingGroup, sourceAutoScalingGroup,
                        AwsAutoScalingGroup::getAvailabilityZones))
                .withLoadBalancerNames(overrideIfNotEmpty(newAutoScalingGroup, sourceAutoScalingGroup,
                        AwsAutoScalingGroup::getLoadBalancerNames))
                .withHealthCheckType(overrideIfNotNull(newAutoScalingGroup, sourceAutoScalingGroup,
                        AwsAutoScalingGroup::getHealthCheckType))
                .withHealthCheckGracePeriod(overrideIfNotNull(newAutoScalingGroup, sourceAutoScalingGroup,
                        AwsAutoScalingGroup::getHealthCheckGracePeriod))
                .withPlacementGroup(overrideIfNotNull(newAutoScalingGroup, sourceAutoScalingGroup,
                        AwsAutoScalingGroup::getPlacementGroup))
                .withVPCZoneIdentifier(overrideIfNotNull(newAutoScalingGroup, sourceAutoScalingGroup,
                        AwsAutoScalingGroup::getVPCZoneIdentifier))
                .withNewInstancesProtectedFromScaleIn(overrideIfNotNull(newAutoScalingGroup, sourceAutoScalingGroup,
                        AwsAutoScalingGroup::isNewInstancesProtectedFromScaleIn))
                .withTags(mapAwsTags(
                        overrideIfNotEmpty(newAutoScalingGroup, sourceAutoScalingGroup, AwsAutoScalingGroup::getTags),
                        newAutoScalingGroup.getAutoScalingGroupName()))
                .withTerminationPolicies(overrideIfNotEmpty(newAutoScalingGroup, sourceAutoScalingGroup,
                        AwsAutoScalingGroup::getTerminationPolicies));
    }

    private CreateAutoScalingGroupRequest newCreateAutoScalingGroupRequest(AwsAutoScalingGroup sourceAutoScalingGroup) {
        return new CreateAutoScalingGroupRequest()
                .withAutoScalingGroupName(sourceAutoScalingGroup.getAutoScalingGroupName())
                .withMinSize(sourceAutoScalingGroup.getMinSize())
                .withMaxSize(sourceAutoScalingGroup.getMaxSize())
                .withDesiredCapacity(sourceAutoScalingGroup.getDesiredCapacity())
                .withLaunchConfigurationName(sourceAutoScalingGroup.getLaunchConfigurationName())
                .withDefaultCooldown(sourceAutoScalingGroup.getDefaultCooldown())
                .withAvailabilityZones(sourceAutoScalingGroup.getAvailabilityZones())
                .withLoadBalancerNames(sourceAutoScalingGroup.getLoadBalancerNames())
                .withHealthCheckType(sourceAutoScalingGroup.getHealthCheckType())
                .withHealthCheckGracePeriod(sourceAutoScalingGroup.getHealthCheckGracePeriod())
                .withPlacementGroup(sourceAutoScalingGroup.getPlacementGroup())
                .withVPCZoneIdentifier(sourceAutoScalingGroup.getVPCZoneIdentifier())
                .withNewInstancesProtectedFromScaleIn(
                        sourceAutoScalingGroup.isNewInstancesProtectedFromScaleIn())
                .withTags(mapAwsTags(sourceAutoScalingGroup.getTags()))
                .withTerminationPolicies(sourceAutoScalingGroup.getTerminationPolicies());
    }

    private PutScalingPolicyRequest newPutScalingPolicyRequest(String newAutoScalingGroupName,
            ScalingPolicy policy) {
        return new PutScalingPolicyRequest()
                .withAutoScalingGroupName(newAutoScalingGroupName)
                .withPolicyName(policy.getPolicyName())
                .withPolicyType(policy.getPolicyType())
                .withCooldown(policy.getCooldown())
                .withEstimatedInstanceWarmup(policy.getEstimatedInstanceWarmup())
                .withMetricAggregationType(policy.getMetricAggregationType())
                .withAdjustmentType(policy.getAdjustmentType())
                .withMinAdjustmentMagnitude(policy.getMinAdjustmentMagnitude())
                .withMinAdjustmentStep(policy.getMinAdjustmentStep())
                .withScalingAdjustment(policy.getScalingAdjustment())
                .withStepAdjustments(policy.getStepAdjustments());
    }

    private PutMetricAlarmRequest newPutMetricAlarmRequest(MetricAlarm sourceMetricAlarm,
            String targetAutoScalingGroupName, String newPolicyArn, String newAlarmSuffix) {

        return new PutMetricAlarmRequest()
                .withAlarmName(sourceMetricAlarm.getAlarmName() + newAlarmSuffix)
                .withAlarmDescription(sourceMetricAlarm.getAlarmDescription())
                .withActionsEnabled(sourceMetricAlarm.getActionsEnabled())
                .withAlarmActions(newPolicyArn)
                .withMetricName(sourceMetricAlarm.getMetricName())
                .withNamespace(sourceMetricAlarm.getNamespace())
                .withStatistic(sourceMetricAlarm.getStatistic())
                .withExtendedStatistic(sourceMetricAlarm.getExtendedStatistic())
                .withDimensions(mapSourceDimensions(sourceMetricAlarm.getDimensions(), targetAutoScalingGroupName))
                .withPeriod(sourceMetricAlarm.getPeriod())
                .withUnit(sourceMetricAlarm.getUnit())
                .withEvaluationPeriods(sourceMetricAlarm.getEvaluationPeriods())
                .withThreshold(sourceMetricAlarm.getThreshold())
                .withComparisonOperator(sourceMetricAlarm.getComparisonOperator());
    }

    private DescribeAlarmsRequest newDescribeAlarmsRequest(List<Alarm> alarms) {
        return new DescribeAlarmsRequest()
                .withAlarmNames(alarms.stream().map(Alarm::getAlarmName).collect(toList()));
    }

    private boolean allInstancesRegisteredInLoadBalancers(AwsAutoScalingGroup asg) {
        List<AwsInstance> awsInstances = asg.getAwsInstances();
        return asg.getLoadBalancerNames().stream()
                .allMatch(elbName -> classicLoadBalancingDiscovery.areInstancesInService(elbName, awsInstances));
    }

    private List<Tag> mapAwsTags(List<AwsAutoScalingTag> tags) {
        return tags.stream()
                .map(tag -> new Tag()
                        .withResourceId(tag.getResourceId())
                        .withKey(tag.getKey())
                        .withValue(tag.getValue())
                        .withPropagateAtLaunch(tag.getPropagateAtLaunch())
                        .withResourceType(tag.resourceType())
                )
                .collect(toList());
    }

    private List<Tag> mapAwsTags(List<AwsAutoScalingTag> tags, String resourceId) {
        return tags.stream()
                .map(tag -> new Tag()
                        .withResourceId(resourceId)
                        .withKey(tag.getKey())
                        .withValue(tag.getValue())
                        .withPropagateAtLaunch(tag.getPropagateAtLaunch())
                        .withResourceType(tag.resourceType())
                )
                .collect(toList());
    }

    private List<Dimension> mapSourceDimensions(List<Dimension> sourceDimensions, String targetAutoScalingGroupName) {
        return Optional.ofNullable(sourceDimensions).orElse(Collections.emptyList())
                .stream().map(dimension -> {
                    if (dimension.getName().equals("AutoScalingGroupName")) {
                        return new Dimension()
                                .withName("AutoScalingGroupName")
                                .withValue(targetAutoScalingGroupName);
                    } else {
                        return dimension;
                    }
                }).collect(toList());
    }
}
