package pl.codewise.commons.aws.cqrs.model.ec2;

import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import pl.codewise.commons.aws.cqrs.model.AwsNameBasedResource;
import pl.codewise.commons.aws.cqrs.model.AwsResourceVisitor;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

public class AwsLaunchConfiguration extends AwsNameBasedResource {

    private final String imageId;
    private final String iamInstanceProfile;
    private final String keyName;
    private final String instanceType;
    private final String spotPrice;
    private final List<String> securityGroupIds;
    private final String userData;
    private final Boolean associatePublicIpAddress;
    private final Boolean instanceMonitoring;
    private final List<AwsBlockDeviceMapping> blockDeviceMappings;
    private final LocalDateTime creationTime;

    private AwsLaunchConfiguration(String name, String imageId, String iamInstanceProfile, String keyName,
            String instanceType, String spotPrice, List<String> securityGroupIds, String userData,
            Boolean associatePublicIpAddress, Boolean instanceMonitoring,
            List<AwsBlockDeviceMapping> blockDeviceMappings, LocalDateTime creationTime) {
        super(name);
        this.imageId = imageId;
        this.iamInstanceProfile = iamInstanceProfile;
        this.keyName = keyName;
        this.instanceType = instanceType;
        this.spotPrice = spotPrice;
        this.securityGroupIds = securityGroupIds;
        this.userData = userData;
        this.associatePublicIpAddress = associatePublicIpAddress;
        this.instanceMonitoring = instanceMonitoring;
        this.blockDeviceMappings = blockDeviceMappings;
        this.creationTime = creationTime;
    }

    public String getImageId() {
        return imageId;
    }

    public String getIamInstanceProfile() {
        return iamInstanceProfile;
    }

    public String getKeyName() {
        return keyName;
    }

    public String getInstanceType() {
        return instanceType;
    }

    public String getSpotPrice() {
        return spotPrice;
    }

    public List<String> getSecurityGroupIds() {
        return securityGroupIds;
    }

    public String getUserData() {
        return userData;
    }

    public Boolean getAssociatePublicIpAddress() {
        return associatePublicIpAddress;
    }

    public Boolean getInstanceMonitoring() {
        return instanceMonitoring;
    }

    public List<AwsBlockDeviceMapping> getBlockDeviceMappings() {
        return blockDeviceMappings;
    }

    public LocalDateTime getCreationTime() {
        return creationTime;
    }

    @Override
    public void accept(AwsResourceVisitor visitor) {
        visitor.visit(this);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }
        AwsLaunchConfiguration that = (AwsLaunchConfiguration) o;
        return Objects.equals(imageId, that.imageId) &&
                Objects.equals(iamInstanceProfile, that.iamInstanceProfile) &&
                Objects.equals(keyName, that.keyName) &&
                Objects.equals(instanceType, that.instanceType) &&
                Objects.equals(spotPrice, that.spotPrice) &&
                Objects.equals(securityGroupIds, that.securityGroupIds) &&
                Objects.equals(userData, that.userData) &&
                Objects.equals(associatePublicIpAddress, that.associatePublicIpAddress) &&
                Objects.equals(instanceMonitoring, that.instanceMonitoring) &&
                Objects.equals(blockDeviceMappings, that.blockDeviceMappings) &&
                Objects.equals(creationTime, that.creationTime);
    }

    @Override
    public int hashCode() {
        return Objects
                .hash(super.hashCode(), imageId, iamInstanceProfile, keyName, instanceType, spotPrice, securityGroupIds,
                        userData, associatePublicIpAddress, instanceMonitoring, blockDeviceMappings, creationTime);
    }

    @Override
    public String toString() {
        return new ReflectionToStringBuilder(this).toString();
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {

        private String name;
        private String imageId;
        private String iamInstanceProfile;
        private String keyName;
        private String instanceType;
        private String spotPrice;
        private List<String> securityGroupIds = new ArrayList<>();
        private String userData;
        private Boolean associatePublicIpAddress = true;
        private Boolean instanceMonitoring;
        private List<AwsBlockDeviceMapping> blockDeviceMappings = new ArrayList<>();
        private LocalDateTime creationTime;

        public Builder withName(String name) {
            this.name = name;
            return this;
        }

        public Builder withImageId(String imageId) {
            this.imageId = imageId;
            return this;
        }

        public Builder withIamInstanceProfile(String iamInstanceProfile) {
            this.iamInstanceProfile = iamInstanceProfile;
            return this;
        }

        public Builder withKeyName(String keyName) {
            this.keyName = keyName;
            return this;
        }

        public Builder withInstanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public Builder withSpotPrice(String spotPrice) {
            this.spotPrice = spotPrice;
            return this;
        }

        public Builder withSecurityGroupIds(List<String> securityGroupIds) {
            this.securityGroupIds = securityGroupIds;
            return this;
        }

        public Builder withUserData(String userData) {
            this.userData = userData;
            return this;
        }

        public Builder withAssociatePublicIpAddress(Boolean associatePublicIpAddress) {
            this.associatePublicIpAddress = associatePublicIpAddress != null ? associatePublicIpAddress : true;
            return this;
        }

        public Builder withInstanceMonitoring(Boolean instanceMonitoring) {
            this.instanceMonitoring = instanceMonitoring;
            return this;
        }

        public Builder withBlockDeviceMappings(List<AwsBlockDeviceMapping> blockDeviceMappings) {
            this.blockDeviceMappings = blockDeviceMappings;
            return this;
        }

        public Builder withCreationTime(LocalDateTime creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public AwsLaunchConfiguration build() {
            return new AwsLaunchConfiguration(this.name, this.imageId, this.iamInstanceProfile, this.keyName,
                    this.instanceType, spotPrice, this.securityGroupIds, this.userData, this.associatePublicIpAddress,
                    instanceMonitoring, blockDeviceMappings, creationTime);
        }
    }
}
