package pl.codewise.commons.aws.cqrs.model;

import com.amazonaws.services.ec2.model.InstanceStateName;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import pl.codewise.commons.aws.cqrs.model.ec2.AwsInstanceBlockDeviceMapping;
import pl.codewise.commons.aws.cqrs.model.ec2.AwsResourceTag;
import pl.codewise.commons.aws.cqrs.model.ec2.autoscaling.AwsAutoScalingDetails;
import pl.codewise.commons.aws.cqrs.model.ec2.sg.AwsSecurityGroup;
import pl.codewise.commons.aws.cqrs.model.ec2.spot.AwsSpotRequestDetails;

import java.util.Date;
import java.util.List;
import java.util.Objects;

public class AwsInstance implements AwsResource {

    private final String instanceId;
    private final String publicDnsName;
    private final String privateIpAddress;
    private final String publicIpAddress;
    private final String iamInstanceProfileArn;
    private final String imageId;
    private final String region;
    private final String state;
    private final String instanceType;
    private final String keyName;
    private final List<AwsNetworkInterface> networkInterfaces;
    private final String subnetId;
    private final String lifecycle;
    private final Date launchTime;
    private final AwsAutoScalingDetails autoScalingDetails;
    private final AwsSpotRequestDetails spotRequestDetails;
    private final List<AwsResourceTag> tags;
    private final List<AwsInstanceBlockDeviceMapping> blockDeviceMappings;
    private final List<AwsSecurityGroup> securityGroups;

    private AwsInstance(String instanceId, String publicDnsName, String privateIpAddress, String publicIpAddress,
            String iamInstanceProfileArn, String imageId, String region, String state, String instanceType,
            String keyName, List<AwsNetworkInterface> networkInterfaces, String subnetId, String lifecycle,
            Date launchTime, AwsAutoScalingDetails autoScalingDetails, AwsSpotRequestDetails spotRequestDetails,
            List<AwsResourceTag> tags,
            List<AwsInstanceBlockDeviceMapping> blockDeviceMappings,
            List<AwsSecurityGroup> securityGroups) {
        this.instanceId = instanceId;
        this.publicDnsName = publicDnsName;
        this.privateIpAddress = privateIpAddress;
        this.publicIpAddress = publicIpAddress;
        this.iamInstanceProfileArn = iamInstanceProfileArn;
        this.imageId = imageId;
        this.region = region;
        this.state = state;
        this.instanceType = instanceType;
        this.keyName = keyName;
        this.networkInterfaces = networkInterfaces;
        this.subnetId = subnetId;
        this.lifecycle = lifecycle;
        this.launchTime = launchTime;
        this.autoScalingDetails = autoScalingDetails;
        this.spotRequestDetails = spotRequestDetails;
        this.tags = tags;
        this.blockDeviceMappings = blockDeviceMappings;
        this.securityGroups = securityGroups;
    }

    @Override
    public String getId() {
        return getInstanceId();
    }

    @Override
    public void accept(AwsResourceVisitor visitor) {
        visitor.visit(this);
    }

    public String getHost() {
        if (StringUtils.isNotBlank(publicDnsName)) {
            return publicDnsName;
        }
        return privateIpAddress;
    }

    public boolean isRunning() {
        return InstanceStateName.Running.toString().equals(state);
    }

    public String getInstanceId() {
        return instanceId;
    }

    public String getPublicDnsName() {
        return publicDnsName;
    }

    public String getPrivateIpAddress() {
        return privateIpAddress;
    }

    public String getPublicIpAddress() {
        return publicIpAddress;
    }

    public String getIamInstanceProfileArn() {
        return iamInstanceProfileArn;
    }

    public String getImageId() {
        return imageId;
    }

    public String getRegion() {
        return region;
    }

    public String getState() {
        return state;
    }

    public String getInstanceType() {
        return instanceType;
    }

    public String getKeyName() {
        return keyName;
    }

    public List<AwsNetworkInterface> getNetworkInterfaces() {
        return networkInterfaces;
    }

    public String getSubnetId() {
        return subnetId;
    }

    public boolean isSpotInstance() {
        return Objects.equals(lifecycle, "spot");
    }

    public String getLifecycle() {
        return lifecycle;
    }

    public Date getLaunchTime() {
        return launchTime;
    }

    public AwsAutoScalingDetails getAutoScalingDetails() {
        return autoScalingDetails;
    }

    public AwsSpotRequestDetails getSpotRequestDetails() {
        return spotRequestDetails;
    }

    public List<AwsResourceTag> getTags() {
        return tags;
    }

    public List<AwsInstanceBlockDeviceMapping> getBlockDeviceMappings() {
        return blockDeviceMappings;
    }

    public List<AwsSecurityGroup> getSecurityGroups() {
        return securityGroups;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AwsInstance that = (AwsInstance) o;
        return Objects.equals(instanceId, that.instanceId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(instanceId);
    }

    @Override
    public String toString() {
        return new ReflectionToStringBuilder(this).toString();
    }

    public Builder toBuilder() {
        return new Builder()
                .withInstanceId(instanceId)
                .withPublicDnsName(publicDnsName)
                .withPublicIpAddress(publicIpAddress)
                .withPrivateIpAddress(privateIpAddress)
                .withIamInstanceProfileArn(iamInstanceProfileArn)
                .withImageId(imageId)
                .withRegion(region)
                .withState(state)
                .withInstanceType(instanceType)
                .withKeyName(keyName)
                .withNetworkInterfaces(networkInterfaces)
                .withSubnetId(subnetId)
                .withAutoScalingDetails(autoScalingDetails)
                .withLifecycle(lifecycle)
                .withLaunchTime(launchTime)
                .withSpotRequestDetails(spotRequestDetails)
                .withTags(tags)
                .withBlockDeviceMapping(blockDeviceMappings)
                .withSecurityGroups(securityGroups);
    }

    public static class Builder {

        private String instanceId;
        private String publicDnsName;
        private String privateIpAddress;
        private String publicIpAddress;
        private String iamInstanceProfileArn;
        private String imageId;
        private String region;
        private String state;
        private String instanceType;
        private String keyName;
        private List<AwsNetworkInterface> networkInterfaces;
        private String subnetId;
        private String lifecycle;
        private Date launchTime;
        private AwsAutoScalingDetails autoScalingDetails;
        private AwsSpotRequestDetails spotRequestDetails;
        private List<AwsResourceTag> tags;
        private List<AwsInstanceBlockDeviceMapping> blockDeviceMappings;
        private List<AwsSecurityGroup> securityGroups;

        public Builder withInstanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public Builder withPublicDnsName(String publicDnsName) {
            this.publicDnsName = publicDnsName;
            return this;
        }

        public Builder withPublicIpAddress(String publicIpAddress) {
            this.publicIpAddress = publicIpAddress;
            return this;
        }

        public Builder withPrivateIpAddress(String privateIpAddress) {
            this.privateIpAddress = privateIpAddress;
            return this;
        }

        public Builder withIamInstanceProfileArn(String iamInstanceProfileArn) {
            this.iamInstanceProfileArn = iamInstanceProfileArn;
            return this;
        }

        public Builder withImageId(String amiId) {
            this.imageId = amiId;
            return this;
        }

        public Builder withRegion(String region) {
            this.region = region;
            return this;
        }

        public Builder withState(String state) {
            this.state = state;
            return this;
        }

        public Builder withInstanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public Builder withKeyName(String keyName) {
            this.keyName = keyName;
            return this;
        }

        public Builder withNetworkInterfaces(List<AwsNetworkInterface> networkInterfaces) {
            this.networkInterfaces = networkInterfaces;
            return this;
        }

        public Builder withSubnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public Builder withAutoScalingDetails(AwsAutoScalingDetails autoScalingDetails) {
            this.autoScalingDetails = autoScalingDetails;
            return this;
        }

        public Builder withLifecycle(String lifecycle) {
            if (lifecycle == null) {
                this.lifecycle = "normal";
            } else {
                this.lifecycle = lifecycle;
            }
            return this;
        }

        public Builder withLaunchTime(Date launchTime) {
            this.launchTime = launchTime;
            return this;
        }

        public Builder withSpotRequestDetails(AwsSpotRequestDetails spotRequestDetails) {
            this.spotRequestDetails = spotRequestDetails;
            return this;
        }

        public Builder withTags(List<AwsResourceTag> tags) {
            this.tags = tags;
            return this;
        }

        public Builder withBlockDeviceMapping(List<AwsInstanceBlockDeviceMapping> blockDeviceMappings) {
            this.blockDeviceMappings = blockDeviceMappings;
            return this;
        }

        public Builder withSecurityGroups(List<AwsSecurityGroup> securityGroups) {
            this.securityGroups = securityGroups;
            return this;
        }

        public AwsInstance build() {
            return new AwsInstance(instanceId, publicDnsName, privateIpAddress, publicIpAddress, iamInstanceProfileArn,
                    imageId, region, state, instanceType, keyName, networkInterfaces, subnetId, lifecycle, launchTime,
                    autoScalingDetails, spotRequestDetails, tags, blockDeviceMappings, securityGroups);
        }
    }
}
