package pl.codewise.commons.aws.cqrs.discovery;

import com.amazonaws.services.route53domains.AmazonRoute53Domains;
import com.amazonaws.services.route53domains.model.CheckDomainAvailabilityRequest;
import com.amazonaws.services.route53domains.model.DomainAvailability;
import com.amazonaws.services.route53domains.model.DomainSummary;
import com.amazonaws.services.route53domains.model.GetOperationDetailRequest;
import com.amazonaws.services.route53domains.model.GetOperationDetailResult;
import com.amazonaws.services.route53domains.model.ListDomainsRequest;
import com.amazonaws.services.route53domains.model.ListDomainsResult;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.RateLimiter;
import pl.codewise.commons.aws.cqrs.discovery.route53.domains.Route53DomainsSettings;
import pl.codewise.commons.aws.cqrs.model.route53.AwsBasicDomain;
import pl.codewise.commons.aws.cqrs.model.route53.AwsDomainRegistrationStatus;
import pl.codewise.commons.aws.cqrs.utils.Awaitilities;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Predicate;

import static org.hamcrest.Matchers.both;
import static org.hamcrest.Matchers.isEmptyOrNullString;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.hamcrest.core.IsNot.not;

public class Route53DomainsDiscovery {

    private static final int LIST_DOMAINS_RESULT_MAX_ITEMS = 100;

    private static final String DEFAULT_EXCEPTION_MESSAGE = "Unable to check domain availability";

    private static final String AVAILABILITY_STATUS_AVAILABLE = DomainAvailability.AVAILABLE.toString();
    /**
     * PENDING
     * <p>
     * The TLD registry didn't return a response in the expected amount of time. When the response is delayed, it
     * usually takes just a few extra seconds. You can resubmit the request immediately.
     * </p>
     */
    @VisibleForTesting
    static final String AVAILABILITY_STATUS_PENDING = "PENDING";

    private final int timeoutMillis;
    private final int pollIntervalMillis;
    private final RateLimiter rateLimiter;

    private final AmazonRoute53Domains amazonRoute53Domains;
    private final Awaitilities awaitilities;

    public Route53DomainsDiscovery(
            AmazonRoute53Domains amazonRoute53Domains,
            Route53DomainsSettings route53DomainsSettings,
            Awaitilities awaitilities) {

        this.amazonRoute53Domains = amazonRoute53Domains;
        this.awaitilities = awaitilities;
        this.pollIntervalMillis = route53DomainsSettings.getPollIntervalMillis();
        this.timeoutMillis = route53DomainsSettings.getTimeoutMillis();
        this.rateLimiter = route53DomainsSettings.getRateLimiter();
    }

    /**
     * Check whether domain address is available for registration
     */
    public boolean isDomainAvailableForRegistration(final String domainName) {

        String domainAvailabilityStatus = awaitilities
                .awaitForValueOrReturnLastValue(timeoutMillis,
                        pollIntervalMillis,
                        () -> checkDomainAvailability(domainName),
                        both(not(isEmptyOrNullString())).and(not(equalTo(AVAILABILITY_STATUS_PENDING))),
                        DEFAULT_EXCEPTION_MESSAGE);

        return AVAILABILITY_STATUS_AVAILABLE.equals(domainAvailabilityStatus);
    }

    public AwsDomainRegistrationStatus getDomainRegistrationStatus(String operationId) {

        GetOperationDetailRequest request = new GetOperationDetailRequest().withOperationId(operationId);
        GetOperationDetailResult operationDetail = amazonRoute53Domains.getOperationDetail(request);
        return toAwsDomainRegistrationStatus(operationDetail);
    }

    public List<AwsBasicDomain> getAllDomains(Predicate<AwsBasicDomain> filter) {

        ListDomainsRequest listDomainsRequest = new ListDomainsRequest()
                .withMaxItems(LIST_DOMAINS_RESULT_MAX_ITEMS);

        ListDomainsResult listDomainsResult = amazonRoute53Domains.listDomains(listDomainsRequest);
        List<AwsBasicDomain> filteredDomains = new ArrayList<>();
        addDomainsMatching(listDomainsResult, filteredDomains, filter);

        while (listDomainsResult.getNextPageMarker() != null) {
            listDomainsRequest.setMarker(listDomainsResult.getNextPageMarker());
            listDomainsResult = amazonRoute53Domains.listDomains(listDomainsRequest);
            addDomainsMatching(listDomainsResult, filteredDomains, filter);
        }
        return filteredDomains;
    }

    public int getDomainsCount() {
        ListDomainsRequest listDomainsRequest = new ListDomainsRequest();
        listDomainsRequest.setMaxItems(LIST_DOMAINS_RESULT_MAX_ITEMS);
        ListDomainsResult listDomainsResult = amazonRoute53Domains.listDomains(listDomainsRequest);

        int domainsCount = listDomainsResult.getDomains().size();

        while (listDomainsResult.getNextPageMarker() != null) {
            listDomainsRequest.setMarker(listDomainsResult.getNextPageMarker());
            listDomainsResult = amazonRoute53Domains.listDomains(listDomainsRequest);
            domainsCount += listDomainsResult.getDomains().size();
        }
        return domainsCount;
    }

    private void addDomainsMatching(ListDomainsResult listDomainsResult, List<AwsBasicDomain> basicDomains,
            Predicate<AwsBasicDomain> filter) {
        listDomainsResult.getDomains().stream()
                .map(this::toBasicDomain)
                .filter(filter)
                .forEach(basicDomains::add);
    }

    private AwsBasicDomain toBasicDomain(DomainSummary domainSummary) {
        return new AwsBasicDomain(
                domainSummary.getDomainName(),
                domainSummary.isAutoRenew(),
                domainSummary.getExpiry().toInstant()
        );
    }

    private AwsDomainRegistrationStatus toAwsDomainRegistrationStatus(GetOperationDetailResult result) {
        return new AwsDomainRegistrationStatus(
                result.getOperationId(),
                result.getStatus(),
                result.getMessage(),
                result.getDomainName(),
                result.getType(),
                result.getSubmittedDate()
        );
    }

    private String checkDomainAvailability(String domainName) {
        CheckDomainAvailabilityRequest request = new CheckDomainAvailabilityRequest().withDomainName(domainName);

        Optional.ofNullable(rateLimiter).ifPresent(RateLimiter::acquire);

        return amazonRoute53Domains
                .checkDomainAvailability(request)
                .getAvailability();
    }
}
