package pl.codewise.commons.aws.cqrs.discovery;

import com.amazonaws.services.route53.AmazonRoute53;
import com.amazonaws.services.route53.model.AliasTarget;
import com.amazonaws.services.route53.model.ChangeInfo;
import com.amazonaws.services.route53.model.GetChangeRequest;
import com.amazonaws.services.route53.model.GetChangeResult;
import com.amazonaws.services.route53.model.HostedZone;
import com.amazonaws.services.route53.model.ListHostedZonesRequest;
import com.amazonaws.services.route53.model.ListHostedZonesResult;
import com.amazonaws.services.route53.model.ListResourceRecordSetsRequest;
import com.amazonaws.services.route53.model.ListResourceRecordSetsResult;
import com.amazonaws.services.route53.model.ResourceRecord;
import com.amazonaws.services.route53.model.ResourceRecordSet;
import pl.codewise.commons.aws.cqrs.model.route53.AwsAliasTarget;
import pl.codewise.commons.aws.cqrs.model.route53.AwsChangeInfo;
import pl.codewise.commons.aws.cqrs.model.route53.AwsHostedZone;
import pl.codewise.commons.aws.cqrs.model.route53.AwsRecordSet;
import pl.codewise.commons.aws.cqrs.model.route53.AwsResourceRecord;
import pl.codewise.commons.aws.cqrs.utils.Awaitilities;

import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Stream;

import static java.lang.Boolean.TRUE;
import static java.lang.String.format;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Stream.concat;

public class Route53Discovery {

    private static final String OPERATION_STATUS_INSYNC = "INSYNC";

    private final AmazonRoute53 amazonRoute53;
    private final Awaitilities awaitilities;
    private final long pollInterval;

    public Route53Discovery(AmazonRoute53 amazonRoute53, Awaitilities awaitilities, long pollInterval) {
        this.amazonRoute53 = amazonRoute53;
        this.awaitilities = awaitilities;
        this.pollInterval = pollInterval;
    }

    public List<AwsHostedZone> listHostedZones() {
        return hostedZoneStream()
                .map(this::toAwsHostedZone)
                .collect(toList());
    }

    public int getHostedZonesCount() {
        Long hostedZoneCount = amazonRoute53.getHostedZoneCount().getHostedZoneCount();
        // not possible to have hosted zones number > max integer;
        return Math.toIntExact(hostedZoneCount);
    }

    public int getHealthChecksCount() {
        Long healthCheckCount = amazonRoute53.getHealthCheckCount().getHealthCheckCount();
        // not possible to have health checks number > max integer;
        return Math.toIntExact(healthCheckCount);
    }

    public List<AwsRecordSet> listRecordSets(String hostedZoneId) {
        return recordSetsStream(hostedZoneId)
                .map(this::toAwsRecordSet)
                .collect(toList());
    }

    /**
     * Wait till given change is in state PENDING
     *
     * @param changeInfo Change status
     *
     * @return Final change status
     */
    public AwsChangeInfo waitForChange(AwsChangeInfo changeInfo, long maxWaitTimeMs) {
        if (OPERATION_STATUS_INSYNC.equals(changeInfo.getStatus())) {
            return changeInfo;
        } else {
            AtomicReference<GetChangeResult> change = new AtomicReference<>();
            awaitilities.awaitTillActionSucceed(maxWaitTimeMs, pollInterval,
                    format("change %s to be applied", changeInfo.getId()),
                    () -> {
                        change.set(amazonRoute53.getChange(new GetChangeRequest(changeInfo.getId())));
                        return OPERATION_STATUS_INSYNC.equals(change.get().getChangeInfo().getStatus());
                    });
            return toAwsChangeInfo(change.get().getChangeInfo());
        }
    }

    private AwsChangeInfo toAwsChangeInfo(ChangeInfo changeInfo) {
        return new AwsChangeInfo.Builder()
                .withId(changeInfo.getId())
                .withStatus(changeInfo.getStatus())
                .build();
    }

    private AwsRecordSet toAwsRecordSet(ResourceRecordSet resourceRecordSet) {
        List<AwsResourceRecord> resourceRecords = resourceRecordSet.getResourceRecords().stream()
                .map(this::toResourceRecord)
                .collect(toList());

        return new AwsRecordSet.Builder()
                .withName(resourceRecordSet.getName())
                .withType(resourceRecordSet.getType())
                .withRegion(resourceRecordSet.getRegion())
                .withTtl(resourceRecordSet.getTTL())
                .withSetIdentifier(resourceRecordSet.getSetIdentifier())
                .withAliasTarget(toAliasTarget(resourceRecordSet.getAliasTarget()))
                .withMultiValueAnswer(resourceRecordSet.getMultiValueAnswer())
                .withHealthCheckId(resourceRecordSet.getHealthCheckId())
                .withResourceRecords(resourceRecords)
                .build();
    }

    private AwsAliasTarget toAliasTarget(AliasTarget aliasTarget) {
        return ofNullable(aliasTarget)
                .map(alias -> new AwsAliasTarget.Builder()
                        .withHostedZoneId(alias.getHostedZoneId())
                        .withDnsName(alias.getDNSName())
                        .withEvaluateTargetHealth(alias.getEvaluateTargetHealth())
                        .build())
                .orElse(null);
    }

    private AwsResourceRecord toResourceRecord(ResourceRecord resourceRecord) {
        return new AwsResourceRecord.Builder()
                .withValue(resourceRecord.getValue())
                .build();
    }

    private Stream<ResourceRecordSet> recordSetsStream(String hostedZoneId) {
        ListResourceRecordSetsRequest request = new ListResourceRecordSetsRequest(hostedZoneId);
        ListResourceRecordSetsResult recordSets = amazonRoute53.listResourceRecordSets(request);

        Stream<ResourceRecordSet> result = recordSets.getResourceRecordSets().stream();

        while (TRUE.equals(recordSets.getIsTruncated())) {
            // request for subsequent pages
            ListResourceRecordSetsRequest nextRequest = new ListResourceRecordSetsRequest()
                    .withHostedZoneId(hostedZoneId)
                    .withStartRecordName(recordSets.getNextRecordName())
                    .withStartRecordType(recordSets.getNextRecordType())
                    .withStartRecordIdentifier(recordSets.getNextRecordIdentifier());
            recordSets = amazonRoute53.listResourceRecordSets(nextRequest);

            // append zones to the result
            result = concat(result, recordSets.getResourceRecordSets().stream());
        }

        return result;
    }

    private Stream<HostedZone> hostedZoneStream() {
        ListHostedZonesResult hostedZones = amazonRoute53.listHostedZones();

        Stream<HostedZone> result = hostedZones.getHostedZones().stream();

        while (TRUE.equals(hostedZones.getIsTruncated())) {
            // request for subsequent pages
            ListHostedZonesRequest nextRequest = new ListHostedZonesRequest().withMarker(hostedZones.getNextMarker());
            hostedZones = amazonRoute53.listHostedZones(nextRequest);

            // append zones to the result
            result = concat(result, hostedZones.getHostedZones().stream());
        }

        return result;
    }

    private AwsHostedZone toAwsHostedZone(HostedZone hostedZone) {
        return new AwsHostedZone.Builder()
                .withId(hostedZone.getId())
                .withName(hostedZone.getName())
                .withType(toType(hostedZone))
                .build();
    }

    private AwsHostedZone.Type toType(HostedZone hostedZone) {
        return hostedZone.getConfig().isPrivateZone() ? AwsHostedZone.Type.PRIVATE : AwsHostedZone.Type.PUBLIC;
    }
}
