package pl.codewise.commons.aws;

import com.amazonaws.services.autoscaling.AmazonAutoScaling;
import com.amazonaws.services.cloudwatch.AmazonCloudWatch;
import com.amazonaws.services.ec2.AmazonEC2;
import com.amazonaws.services.elasticloadbalancing.AmazonElasticLoadBalancing;
import com.amazonaws.services.identitymanagement.AmazonIdentityManagement;
import com.amazonaws.services.lambda.AWSLambda;
import com.amazonaws.services.route53.AmazonRoute53;
import com.amazonaws.services.route53domains.AmazonRoute53Domains;
import com.amazonaws.services.s3.AmazonS3;
import pl.codewise.commons.aws.cqrs.discovery.ApplicationLoadBalancingDiscovery;
import pl.codewise.commons.aws.cqrs.discovery.AutoScalingDiscovery;
import pl.codewise.commons.aws.cqrs.discovery.ClassicLoadBalancingDiscovery;
import pl.codewise.commons.aws.cqrs.discovery.Ec2Discovery;
import pl.codewise.commons.aws.cqrs.discovery.IamDiscovery;
import pl.codewise.commons.aws.cqrs.discovery.LaunchConfigurationDiscovery;
import pl.codewise.commons.aws.cqrs.discovery.Route53Discovery;
import pl.codewise.commons.aws.cqrs.discovery.Route53DomainsDiscovery;
import pl.codewise.commons.aws.cqrs.discovery.S3Discovery;
import pl.codewise.commons.aws.cqrs.discovery.SecurityGroupDiscovery;
import pl.codewise.commons.aws.cqrs.discovery.ec2.AddressDiscovery;
import pl.codewise.commons.aws.cqrs.discovery.ec2.AmiDiscovery;
import pl.codewise.commons.aws.cqrs.discovery.route53.domains.Route53DomainsSettings;
import pl.codewise.commons.aws.cqrs.factories.AssumeRoleParameters;
import pl.codewise.commons.aws.cqrs.operations.ApplicationLoadBalancingOperations;
import pl.codewise.commons.aws.cqrs.operations.AutoScalingOperations;
import pl.codewise.commons.aws.cqrs.operations.ClassicLoadBalancingOperations;
import pl.codewise.commons.aws.cqrs.operations.CloudWatchOperations;
import pl.codewise.commons.aws.cqrs.operations.IamOperations;
import pl.codewise.commons.aws.cqrs.operations.LambdaOperations;
import pl.codewise.commons.aws.cqrs.operations.Route53DomainsOperations;
import pl.codewise.commons.aws.cqrs.operations.Route53Operations;
import pl.codewise.commons.aws.cqrs.operations.ec2.AddressOperations;
import pl.codewise.commons.aws.cqrs.operations.ec2.InstanceOperations;
import pl.codewise.commons.aws.cqrs.operations.ec2.LaunchConfigurationOperations;
import pl.codewise.commons.aws.cqrs.operations.ec2.SecurityGroupOperations;
import pl.codewise.commons.aws.cqrs.operations.s3.S3Operations;
import pl.codewise.commons.aws.cqrs.utils.Awaitilities;
import pl.codewise.commons.aws.cqrs.utils.FileSystem;

public class AwsCloudFactory {

    private final AwsClientsCache awsClientsCache;

    private final FileSystem fileSystem;
    private final Awaitilities awaitilities;
    private final int s3OperationsPutFileRetryCount;
    private final int s3OperationsGetFileRetryCount;
    private final long waitForNoInService;
    private final long waitForAutoScalingShutdown;
    private final long waitForInstanceStartup;
    private final long autoScalingPollInterval;
    private final long defaultPollInterval;

    public AwsCloudFactory(AwsClientsCache awsClientsCache,
            FileSystem fileSystem,
            Awaitilities awaitilities,
            int s3OperationsPutFileRetryCount,
            int s3OperationsGetFileRetryCount,
            long waitForNoInService,
            long waitForAutoScalingShutdown,
            long waitForInstanceStartup,
            long autoScalingPollInterval,
            long defaultPollInterval) {
        this.awsClientsCache = awsClientsCache;
        this.fileSystem = fileSystem;
        this.awaitilities = awaitilities;
        this.s3OperationsPutFileRetryCount = s3OperationsPutFileRetryCount;
        this.s3OperationsGetFileRetryCount = s3OperationsGetFileRetryCount;
        this.waitForNoInService = waitForNoInService;
        this.waitForAutoScalingShutdown = waitForAutoScalingShutdown;
        this.waitForInstanceStartup = waitForInstanceStartup;
        this.autoScalingPollInterval = autoScalingPollInterval;
        this.defaultPollInterval = defaultPollInterval;
    }

    public S3Discovery s3Discovery() {
        AmazonS3 client = awsClientsCache.newAmazonS3();
        return new S3Discovery(client);
    }

    public S3Discovery s3Discovery(AssumeRoleParameters parameters) {
        AmazonS3 client = awsClientsCache.newAmazonS3(parameters);
        return new S3Discovery(client);
    }

    public S3Operations s3Operations() {
        AmazonS3 client = awsClientsCache.newAmazonS3();
        S3Discovery s3Discovery = s3Discovery();
        return new S3Operations(client, s3Discovery, fileSystem, s3OperationsPutFileRetryCount,
                s3OperationsGetFileRetryCount);
    }

    public S3Operations s3Operations(AssumeRoleParameters parameters) {
        AmazonS3 client = awsClientsCache.newAmazonS3(parameters);
        S3Discovery s3Discovery = s3Discovery(parameters);
        return new S3Operations(client, s3Discovery, fileSystem, s3OperationsPutFileRetryCount,
                s3OperationsGetFileRetryCount);
    }

    public IamDiscovery iamDiscovery() {
        AmazonIdentityManagement client = awsClientsCache.newAmazonIdentityManagement();
        return new IamDiscovery(client);
    }

    public IamDiscovery iamDiscovery(AssumeRoleParameters parameters) {
        AmazonIdentityManagement client = awsClientsCache.newAmazonIdentityManagement(parameters);
        return new IamDiscovery(client);
    }

    public IamOperations iamOperations() {
        AmazonIdentityManagement client = awsClientsCache.newAmazonIdentityManagement();
        IamDiscovery iamDiscovery = iamDiscovery();
        return new IamOperations(iamDiscovery, client);
    }

    public IamOperations iamOperations(AssumeRoleParameters parameters) {
        AmazonIdentityManagement client = awsClientsCache.newAmazonIdentityManagement(parameters);
        IamDiscovery iamDiscovery = iamDiscovery(parameters);
        return new IamOperations(iamDiscovery, client);
    }

    public Ec2Discovery ec2Discovery(String region) {
        AmazonEC2 amazonEC2 = awsClientsCache.newAmazonEC2(region);
        AmazonAutoScaling amazonAutoScaling = awsClientsCache.newAmazonAutoScaling(region);
        return new Ec2Discovery(region, amazonEC2, amazonAutoScaling, awaitilities, defaultPollInterval);
    }

    public Ec2Discovery ec2Discovery(String region, AssumeRoleParameters parameters) {
        AmazonEC2 amazonEC2 = awsClientsCache.newAmazonEC2(region, parameters);
        AmazonAutoScaling amazonAutoScaling = awsClientsCache.newAmazonAutoScaling(region, parameters);
        return new Ec2Discovery(region, amazonEC2, amazonAutoScaling, awaitilities, defaultPollInterval);
    }

    public AutoScalingDiscovery autoScalingDiscovery(String region) {
        AmazonAutoScaling client = awsClientsCache.newAmazonAutoScaling(region);
        Ec2Discovery ec2Discovery = ec2Discovery(region);
        return new AutoScalingDiscovery(client, ec2Discovery, awaitilities);
    }

    public AutoScalingDiscovery autoScalingDiscovery(String region, AssumeRoleParameters parameters) {
        AmazonAutoScaling client = awsClientsCache.newAmazonAutoScaling(region, parameters);
        Ec2Discovery ec2Discovery = ec2Discovery(region, parameters);
        return new AutoScalingDiscovery(client, ec2Discovery, awaitilities);
    }

    public AmiDiscovery amiDiscovery(String region) {
        AmazonEC2 amazonEC2 = awsClientsCache.newAmazonEC2(region);
        return new AmiDiscovery(amazonEC2);
    }

    public AmiDiscovery amiDiscovery(String region, AssumeRoleParameters parameters) {
        AmazonEC2 amazonEC2 = awsClientsCache.newAmazonEC2(region, parameters);
        return new AmiDiscovery(amazonEC2);
    }

    public AddressDiscovery addressDiscovery(String region) {
        AmazonEC2 amazonEC2 = awsClientsCache.newAmazonEC2(region);
        return new AddressDiscovery(region, amazonEC2);
    }

    public AddressDiscovery addressDiscovery(String region, AssumeRoleParameters parameters) {
        AmazonEC2 amazonEC2 = awsClientsCache.newAmazonEC2(region, parameters);
        return new AddressDiscovery(region, amazonEC2);
    }

    public ClassicLoadBalancingDiscovery elasticLoadBalancingDiscovery(String region) {
        AmazonElasticLoadBalancing amazonElasticLoadBalancing = awsClientsCache.newAmazonElasticLoadBalancing(region);
        return new ClassicLoadBalancingDiscovery(amazonElasticLoadBalancing);
    }

    public ClassicLoadBalancingDiscovery elasticLoadBalancingDiscovery(String region, AssumeRoleParameters parameters) {
        AmazonElasticLoadBalancing amazonElasticLoadBalancing =
                awsClientsCache.newAmazonElasticLoadBalancing(region, parameters);
        return new ClassicLoadBalancingDiscovery(amazonElasticLoadBalancing);
    }

    public ApplicationLoadBalancingDiscovery applicationLoadBalancingDiscovery(String region) {
        com.amazonaws.services.elasticloadbalancingv2.AmazonElasticLoadBalancing amazonElasticLoadBalancing =
                awsClientsCache.newApplicationLoadBalancing(region);
        return new ApplicationLoadBalancingDiscovery(amazonElasticLoadBalancing);
    }

    public ApplicationLoadBalancingDiscovery applicationLoadBalancingDiscovery(String region,
            AssumeRoleParameters parameters) {
        com.amazonaws.services.elasticloadbalancingv2.AmazonElasticLoadBalancing amazonElasticLoadBalancing =
                awsClientsCache.newApplicationLoadBalancing(region, parameters);
        return new ApplicationLoadBalancingDiscovery(amazonElasticLoadBalancing);
    }

    public LaunchConfigurationDiscovery launchConfigurationDiscovery(String region) {
        AmazonAutoScaling client = awsClientsCache.newAmazonAutoScaling(region);
        return new LaunchConfigurationDiscovery(client);
    }

    public LaunchConfigurationDiscovery launchConfigurationDiscovery(String region, AssumeRoleParameters parameters) {
        AmazonAutoScaling amazonAutoScaling = awsClientsCache.newAmazonAutoScaling(region, parameters);
        return new LaunchConfigurationDiscovery(amazonAutoScaling);
    }

    public SecurityGroupDiscovery securityGroupDiscovery(String region) {
        AmazonEC2 amazonEC2 = awsClientsCache.newAmazonEC2(region);
        return new SecurityGroupDiscovery(amazonEC2);
    }

    public SecurityGroupDiscovery securityGroupDiscovery(String region, AssumeRoleParameters parameters) {
        AmazonEC2 amazonEC2 = awsClientsCache.newAmazonEC2(region, parameters);
        return new SecurityGroupDiscovery(amazonEC2);
    }

    public Route53Discovery route53Discovery() {
        AmazonRoute53 amazonRoute53 = awsClientsCache.newAmazonRoute53();
        return new Route53Discovery(amazonRoute53, awaitilities, defaultPollInterval);
    }

    public Route53Discovery route53Discovery(AssumeRoleParameters parameters) {
        AmazonRoute53 amazonRoute53 = awsClientsCache.newAmazonRoute53(parameters);
        return new Route53Discovery(amazonRoute53, awaitilities, defaultPollInterval);
    }

    public AutoScalingOperations autoScalingOperations(String region) {
        AmazonAutoScaling autoScaling = awsClientsCache.newAmazonAutoScaling(region);
        AmazonCloudWatch cloudWatch = awsClientsCache.newAmazonCloudWatch(region);
        AutoScalingDiscovery autoScalingDiscovery = autoScalingDiscovery(region);
        ClassicLoadBalancingDiscovery classicLoadBalancingDiscovery = elasticLoadBalancingDiscovery(region);
        return new AutoScalingOperations(region, autoScaling, cloudWatch, autoScalingDiscovery,
                classicLoadBalancingDiscovery, awaitilities, waitForNoInService, waitForAutoScalingShutdown,
                waitForInstanceStartup, autoScalingPollInterval);
    }

    public AutoScalingOperations autoScalingOperations(String region, AssumeRoleParameters parameters) {
        AmazonAutoScaling autoScaling = awsClientsCache.newAmazonAutoScaling(region, parameters);
        AmazonCloudWatch cloudWatch = awsClientsCache.newAmazonCloudWatch(region, parameters);
        AutoScalingDiscovery autoScalingDiscovery = autoScalingDiscovery(region, parameters);
        ClassicLoadBalancingDiscovery classicLoadBalancingDiscovery = elasticLoadBalancingDiscovery(region, parameters);
        return new AutoScalingOperations(region, autoScaling, cloudWatch, autoScalingDiscovery,
                classicLoadBalancingDiscovery, awaitilities, waitForNoInService, waitForAutoScalingShutdown,
                waitForInstanceStartup, autoScalingPollInterval);
    }

    public AddressOperations addressOperations(String region) {
        AmazonEC2 amazonEC2 = awsClientsCache.newAmazonEC2(region);
        AddressDiscovery addressDiscovery = addressDiscovery(region);
        return new AddressOperations(region, amazonEC2, addressDiscovery);
    }

    public AddressOperations addressOperations(String region, AssumeRoleParameters parameters) {
        AmazonEC2 amazonEC2 = awsClientsCache.newAmazonEC2(region, parameters);
        AddressDiscovery addressDiscovery = addressDiscovery(region, parameters);
        return new AddressOperations(region, amazonEC2, addressDiscovery);
    }

    public ClassicLoadBalancingOperations elasticLoadBalancingOperations(String region) {
        AmazonElasticLoadBalancing amazonElasticLoadBalancing = awsClientsCache.newAmazonElasticLoadBalancing(region);
        return new ClassicLoadBalancingOperations(amazonElasticLoadBalancing);
    }

    public ClassicLoadBalancingOperations elasticLoadBalancingOperations(String region,
            AssumeRoleParameters parameters) {
        AmazonElasticLoadBalancing amazonElasticLoadBalancing =
                awsClientsCache.newAmazonElasticLoadBalancing(region, parameters);
        return new ClassicLoadBalancingOperations(amazonElasticLoadBalancing);
    }

    public ApplicationLoadBalancingOperations applicationLoadBalancingOperations(String region) {
        com.amazonaws.services.elasticloadbalancingv2.AmazonElasticLoadBalancing amazonElasticLoadBalancing =
                awsClientsCache.newApplicationLoadBalancing(region);
        return new ApplicationLoadBalancingOperations(amazonElasticLoadBalancing);
    }

    public ApplicationLoadBalancingOperations applicationLoadBalancingOperations(String region,
            AssumeRoleParameters parameters) {
        com.amazonaws.services.elasticloadbalancingv2.AmazonElasticLoadBalancing amazonElasticLoadBalancing =
                awsClientsCache.newApplicationLoadBalancing(region, parameters);
        return new ApplicationLoadBalancingOperations(amazonElasticLoadBalancing);
    }

    public LaunchConfigurationOperations launchConfigurationOperations(String region) {
        AmazonAutoScaling amazonAutoScaling = awsClientsCache.newAmazonAutoScaling(region);
        LaunchConfigurationDiscovery configurationDiscovery = launchConfigurationDiscovery(region);
        return new LaunchConfigurationOperations(amazonAutoScaling, configurationDiscovery);
    }

    public LaunchConfigurationOperations launchConfigurationOperations(String region, AssumeRoleParameters parameters) {
        AmazonAutoScaling amazonAutoScaling = awsClientsCache.newAmazonAutoScaling(region, parameters);
        LaunchConfigurationDiscovery configurationDiscovery = launchConfigurationDiscovery(region, parameters);
        return new LaunchConfigurationOperations(amazonAutoScaling, configurationDiscovery);
    }

    public SecurityGroupOperations securityGroupOperations(String region) {
        AmazonEC2 amazonEC2 = awsClientsCache.newAmazonEC2(region);
        return new SecurityGroupOperations(amazonEC2);
    }

    public SecurityGroupOperations securityGroupOperations(String region, AssumeRoleParameters parameters) {
        AmazonEC2 amazonEC2 = awsClientsCache.newAmazonEC2(region, parameters);
        return new SecurityGroupOperations(amazonEC2);
    }

    public Route53Operations route53Operations() {
        return new Route53Operations(awsClientsCache.newAmazonRoute53());
    }

    public Route53Operations route53Operations(AssumeRoleParameters parameters) {
        AmazonRoute53 amazonRoute53 = awsClientsCache.newAmazonRoute53(parameters);
        return new Route53Operations(amazonRoute53);
    }

    public Route53DomainsDiscovery route53DomainsDiscovery(Route53DomainsSettings route53DomainsSettings) {
        return new Route53DomainsDiscovery(
                awsClientsCache.newAmazonRoute53Domains(),
                route53DomainsSettings,
                awaitilities);
    }

    public Route53DomainsDiscovery route53DomainsDiscovery(
            Route53DomainsSettings route53DomainsSettings,
            AssumeRoleParameters parameters) {
        AmazonRoute53Domains amazonRoute53Domains = awsClientsCache.newAmazonRoute53Domains(parameters);
        return new Route53DomainsDiscovery(amazonRoute53Domains, route53DomainsSettings, awaitilities);
    }

    public Route53DomainsOperations route53DomainsOperations() {
        return new Route53DomainsOperations(awsClientsCache.newAmazonRoute53Domains());
    }

    public Route53DomainsOperations route53DomainsOperations(AssumeRoleParameters parameters) {
        AmazonRoute53Domains amazonRoute53Domains = awsClientsCache.newAmazonRoute53Domains(parameters);
        return new Route53DomainsOperations(amazonRoute53Domains);
    }

    public InstanceOperations instanceOperations(String region) {
        AmazonEC2 amazonEC2 = awsClientsCache.newAmazonEC2(region);
        Ec2Discovery ec2Discovery = ec2Discovery(region);
        return new InstanceOperations(amazonEC2, ec2Discovery);
    }

    public InstanceOperations instanceOperations(String region, AssumeRoleParameters parameters) {
        AmazonEC2 amazonEC2 = awsClientsCache.newAmazonEC2(region, parameters);
        Ec2Discovery ec2Discovery = ec2Discovery(region, parameters);
        return new InstanceOperations(amazonEC2, ec2Discovery);
    }

    public LambdaOperations lambdaOperations(String region) {
        AWSLambda awsLambda = awsClientsCache.newAwsLambda(region);
        return new LambdaOperations(awsLambda);
    }

    public LambdaOperations lambdaOperations(String region, AssumeRoleParameters parameters) {
        AWSLambda awsLambda = awsClientsCache.newAwsLambda(region, parameters);
        return new LambdaOperations(awsLambda);
    }

    public CloudWatchOperations cloudWatchOperations(String region) {
        AmazonCloudWatch amazonCloudWatch = awsClientsCache.newAmazonCloudWatch(region);
        return new CloudWatchOperations(amazonCloudWatch);
    }

    public CloudWatchOperations cloudWatchOperations(String region, AssumeRoleParameters parameters) {
        AmazonCloudWatch amazonCloudWatch = awsClientsCache.newAmazonCloudWatch(region, parameters);
        return new CloudWatchOperations(amazonCloudWatch);
    }
}
