/*
 * Hibernate Search, full-text search for your domain model
 *
 * License: GNU Lesser General Public License (LGPL), version 2.1 or later
 * See the lgpl.txt file in the root directory or <http://www.gnu.org/licenses/lgpl-2.1.html>.
 */
package org.hibernate.search.backend.elasticsearch.cfg;

import org.hibernate.search.backend.elasticsearch.analysis.ElasticsearchAnalysisConfigurer;
import org.hibernate.search.backend.elasticsearch.index.DynamicMapping;
import org.hibernate.search.backend.elasticsearch.index.IndexStatus;
import org.hibernate.search.util.common.impl.HibernateSearchConfiguration;

/**
 * Configuration properties for Elasticsearch indexes.
 * <p>
 * Constants in this class are to be appended to a prefix to form a property key;
 * see {@link org.hibernate.search.engine.cfg.IndexSettings} for details.
 */
@HibernateSearchConfiguration(
		prefix = {
				"hibernate.search.backend.",
				"hibernate.search.backend.index.<index name>.",
				"hibernate.search.backends.<backend name>.",
				"hibernate.search.backends.<backend name>.index.<index name>."
		},
		title = "Hibernate Search Backend - Elasticsearch",
		anchorPrefix = "hibernate-search-backend-elasticsearch-"
)
public final class ElasticsearchIndexSettings {

	private ElasticsearchIndexSettings() {
	}

	/**
	 * The analysis configurer applied to this index.
	 * <p>
	 * Expects a single-valued or multi-valued reference to beans of type {@link ElasticsearchAnalysisConfigurer}.
	 * <p>
	 * Defaults to no value.
	 *
	 * @see org.hibernate.search.engine.cfg The core documentation of configuration properties,
	 * which includes a description of the "multi-valued bean reference" properties and accepted values.
	 */
	public static final String ANALYSIS_CONFIGURER = "analysis.configurer";

	/**
	 * The minimal required status of an index on startup, before Hibernate Search can start using it.
	 * <p>
	 * Expects an {@link IndexStatus} value, or a String representation of such value.
	 * <p>
	 * Defaults to {@link Defaults#SCHEMA_MANAGEMENT_MINIMAL_REQUIRED_STATUS}.
	 */
	public static final String SCHEMA_MANAGEMENT_MINIMAL_REQUIRED_STATUS = "schema_management.minimal_required_status";

	/**
	 * The timeout when waiting for the {@link #SCHEMA_MANAGEMENT_MINIMAL_REQUIRED_STATUS required status}.
	 * <p>
	 * Expects a positive Integer value in milliseconds, such as {@code 60000},
	 * or a String that can be parsed into such Integer value.
	 * <p>
	 * Defaults to {@link Defaults#SCHEMA_MANAGEMENT_MINIMAL_REQUIRED_STATUS_WAIT_TIMEOUT}.
	 */
	public static final String SCHEMA_MANAGEMENT_MINIMAL_REQUIRED_STATUS_WAIT_TIMEOUT = "schema_management.minimal_required_status_wait_timeout";

	/**
	 * The path to a settings file, allowing to add custom settings to indexes created by Hibernate Search as part of schema management.
	 * <p>
	 * Expects a string representing the path to a file in the classpath, encoded in UTF-8. The file must contain index settings expressed in JSON format, with the exact same syntax as expected by the Elasticsearch server under the "settings" property <a href="https://www.elastic.co/guide/en/elasticsearch/reference/current/indices-create-index.html#create-index-settings">when creating an index</a>.
	 * For example, if the file content is <code>{"index.codec": "best_compression"}</code>, it will set {@literal  index.codec} to {@literal best_compression}.
	 * <p>
	 * Note that the settings generated by Hibernate Search will be overridden in case of conflict of some definitions.
 	 * For instance, if an analyzer "myAnalyzer" is defined by the {@link #ANALYSIS_CONFIGURER} <strong>and</strong> this settings file, the definition from the settings file will win. If it is only defined in either the analysis configurer or the settings file, but not both, it will be preserved as-is.
	 * <p>
	 * Defaults to no value. Because by default index settings are generated by Hibernate Search.
	 */
	public static final String SCHEMA_MANAGEMENT_SETTINGS_FILE = "schema_management.settings_file";

	/**
	 * The path to a mappings file, allowing to add custom mappings to indexes created by Hibernate Search as part of schema management.
	 * <p>
	 * Expects a string representing the path to a file in the classpath, encoded in UTF-8. The file must contain index settings expressed in JSON format, with the exact same syntax as expected by the Elasticsearch server under the "mappings" property <a href="https://www.elastic.co/guide/en/elasticsearch/reference/current/mapping.html">when defining the mapping for an index</a>.
	 * <p>
	 * The file does not need to contain the full mapping: Hibernate Search will automatically inject
	 * missing properties (index fields) in the given mapping.
	 * <p>
	 * Conflicts between the given mapping and the mapping generated by Hibernate Search will be handled as follows:
	 * <ul>
	 * <li>Mapping parameters other than {@code properties} at the mapping root will be
	 * those from the given mapping; those generated by Hibernate Search will be ignored.</li>
	 * <li>{@code properties} will be merged, using properties defined in both the
	 * given mapping and the mapping generated by Hibernate Search.
	 * If a property is defined on both sides, mapping parameters from the given mapping
	 * will be used, except for {@code properties},
	 * which will be merged recursively in the same way.
	 * </ul>
	 * <p>
	 * Defaults to no value. Because by default index mappings are generated by Hibernate Search.
	 */
	public static final String SCHEMA_MANAGEMENT_MAPPING_FILE = "schema_management.mapping_file";

	/**
	 * Specify the default behavior to handle dynamically-mapped fields in the Elasticsearch mapping.
	 * <p>
	 * Defaults to {@link Defaults#DYNAMIC_MAPPING}.
	 * <p>
	 * In case of dynamic fields with field templates, the value will be ignored,
	 * since this feature requires a {@link DynamicMapping#TRUE} to operate.
	 *
	 * @see DynamicMapping
	 */
	public static final String DYNAMIC_MAPPING = "dynamic_mapping";

	/**
	 * The prefix for indexing-related property keys.
	 */
	public static final String INDEXING_PREFIX = "indexing.";

	/**
	 * The number of indexing queues assigned to each index.
	 * <p>
	 * Expects a strictly positive integer value,
	 * or a string that can be parsed to such integer value.
	 * <p>
	 * Defaults to {@link Defaults#INDEXING_QUEUE_COUNT}.
	 * <p>
	 * See the reference documentation, section "Elasticsearch backend - Indexing",
	 * for more information about this setting and its implications.
	 */
	public static final String INDEXING_QUEUE_COUNT = INDEXING_PREFIX + IndexingRadicals.QUEUE_COUNT;

	/**
	 * The size of indexing queues.
	 * <p>
	 * Expects a strictly positive integer value,
	 * or a string that can be parsed to such integer value.
	 * <p>
	 * Defaults to {@link Defaults#INDEXING_QUEUE_SIZE}.
	 * <p>
	 * See the reference documentation, section "Elasticsearch backend - Indexing",
	 * for more information about this setting and its implications.
	 */
	public static final String INDEXING_QUEUE_SIZE = INDEXING_PREFIX + IndexingRadicals.QUEUE_SIZE;

	/**
	 * The maximum size of bulk requests created when processing indexing queues.
	 * <p>
	 * Expects a strictly positive integer value,
	 * or a string that can be parsed to such integer value.
	 * <p>
	 * Defaults to {@link Defaults#INDEXING_MAX_BULK_SIZE}.
	 * <p>
	 * See the reference documentation, section "Elasticsearch backend - Indexing",
	 * for more information about this setting and its implications.
	 */
	public static final String INDEXING_MAX_BULK_SIZE = INDEXING_PREFIX + IndexingRadicals.MAX_BULK_SIZE;

	/**
	 * Configuration property keys for indexing, without the {@link #INDEXING_PREFIX prefix}.
	 */
	public static final class IndexingRadicals {

		private IndexingRadicals() {
		}

		public static final String QUEUE_COUNT = "queue_count";
		public static final String QUEUE_SIZE = "queue_size";
		public static final String MAX_BULK_SIZE = "max_bulk_size";
	}

	/**
	 * Default values for the different settings if no values are given.
	 */
	public static final class Defaults {

		private Defaults() {
		}

		public static final IndexStatus SCHEMA_MANAGEMENT_MINIMAL_REQUIRED_STATUS = IndexStatus.YELLOW;
		public static final int SCHEMA_MANAGEMENT_MINIMAL_REQUIRED_STATUS_WAIT_TIMEOUT = 10_000;
		public static final int INDEXING_QUEUE_COUNT = 10;
		public static final int INDEXING_QUEUE_SIZE = 1000;
		public static final int INDEXING_MAX_BULK_SIZE = 100;
		public static final DynamicMapping DYNAMIC_MAPPING = DynamicMapping.STRICT;
	}

}
