/*
 *     Licensed to the Apache Software Foundation (ASF) under one or more
 *     contributor license agreements.  See the NOTICE file distributed with
 *     this work for additional information regarding copyright ownership.
 *     The ASF licenses this file to You under the Apache License, Version 2.0
 *     (the "License"); you may not use this file except in compliance with
 *     the License.  You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *     Unless required by applicable law or agreed to in writing, software
 *     distributed under the License is distributed on an "AS IS" BASIS,
 *     WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *     See the License for the specific language governing permissions and
 *     limitations under the License.
 */
package org.apache.tomee.microprofile.jwt.config;

import org.jose4j.jwk.JsonWebKey;
import org.jose4j.lang.JoseException;

import java.security.Key;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import java.util.stream.Collectors;

/**
 * The public key and expected issuer needed to validate a token.
 */
public class JWTAuthConfiguration {
    private static final Logger logger = Logger.getLogger(JWTAuthConfiguration.class.getName());
    public static final String DEFAULT_KEY = "DEFAULT";

    private Map<String, Key> publicKeys;
    private String issuer;
    private int expGracePeriodSecs = 60;

    private JWTAuthConfiguration(final Key publicKey, final String issuer) {
        this.publicKeys = Collections.singletonMap(DEFAULT_KEY, publicKey);
        this.issuer = issuer;
    }

    private JWTAuthConfiguration(final Map<String, Key> publicKeys, final String issuer) {
        if (publicKeys.size() == 1) {
            final Key singleKey = publicKeys.values().iterator().next();
            this.publicKeys = Collections.singletonMap(DEFAULT_KEY, singleKey);
        } else {
            this.publicKeys = Collections.unmodifiableMap(publicKeys);
        }
        this.issuer = issuer;
    }

    public static JWTAuthConfiguration authContextInfo(final Key publicKey, final String issuer) {
        return new JWTAuthConfiguration(publicKey, issuer);
    }

    public static JWTAuthConfiguration authContextInfo(final Map<String, Key> publicKeys, final String issuer) {
        return new JWTAuthConfiguration(publicKeys, issuer);
    }

    public boolean isSingleKey() {
        return publicKeys.size() == 1;
    }

    public Key getPublicKey() {
        return publicKeys.get(DEFAULT_KEY);
    }

    public List<JsonWebKey> getPublicKeys() {
        return publicKeys.entrySet().stream().map(key -> {
            try {
                final JsonWebKey jsonWebKey = JsonWebKey.Factory.newJwk(key.getValue());
                jsonWebKey.setKeyId(key.getKey());
                return jsonWebKey;
            } catch (final JoseException e) {
                logger.warning(e.getMessage());
                return null;
            }
        }).collect(Collectors.toList());
    }

    public String getIssuer() {
        return issuer;
    }

    public int getExpGracePeriodSecs() {
        return expGracePeriodSecs;
    }
}
