/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.tamaya.inject.internal;

import org.apache.tamaya.event.PropertyChangeSet;

import java.lang.reflect.Method;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * This class holds a method object that is annotated to be a callback method to be called on configuration
 * changes.
 */
public final class ConfigChangeCallbackMethod {

    private static final Logger LOG = Logger.getLogger(ConfigChangeCallbackMethod.class.getName());

    private Method callbackMethod;

    public ConfigChangeCallbackMethod(Method methodCandidate) {
        if (void.class.equals(methodCandidate.getReturnType()) &&
                methodCandidate.getParameterCount() == 1 &&
                methodCandidate.getParameterTypes()[0].equals(PropertyChangeSet.class)) {
            this.callbackMethod = methodCandidate;
        }
    }

    public PropertyChangeSetListener createConsumer(final Object instance) {
        return new PropertyChangeSetListener() {
            @Override
            public void propertyChange(PropertyChangeSet propertyChangeSet) {
                call(instance, propertyChangeSet);
            }
        };
    }

    public void call(Object instance, PropertyChangeSet configChangeEvent) {
        try {
            callbackMethod.setAccessible(true);
            callbackMethod.invoke(instance, configChangeEvent);
        } catch (Exception e) {
            LOG.log(Level.SEVERE, "Error calling ConfigChange callback method " +
                            callbackMethod.getDeclaringClass().getName() + '.' + callbackMethod.getName() + " on " + instance,
                    e);
        }
    }
}
