/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.geo;

import java.util.Locale;
import org.apache.lucene.util.Bits;
import org.apache.lucene.util.SloppyMath;
import org.elasticsearch.common.geo.GeoPoint;
import org.elasticsearch.common.geo.GeoUtils;
import org.elasticsearch.common.unit.DistanceUnit;
import org.elasticsearch.index.fielddata.FieldData;
import org.elasticsearch.index.fielddata.GeoPointValues;
import org.elasticsearch.index.fielddata.MultiGeoPointValues;
import org.elasticsearch.index.fielddata.NumericDoubleValues;
import org.elasticsearch.index.fielddata.SortedNumericDoubleValues;
import org.elasticsearch.index.fielddata.SortingNumericDoubleValues;

public enum GeoDistance {
    PLANE{

        @Override
        public double calculate(double sourceLatitude, double sourceLongitude, double targetLatitude, double targetLongitude, DistanceUnit unit) {
            double px = targetLongitude - sourceLongitude;
            double py = targetLatitude - sourceLatitude;
            return Math.sqrt(px * px + py * py) * unit.getDistancePerDegree();
        }

        @Override
        public double normalize(double distance, DistanceUnit unit) {
            return distance;
        }

        @Override
        public FixedSourceDistance fixedSourceDistance(double sourceLatitude, double sourceLongitude, DistanceUnit unit) {
            return new PlaneFixedSourceDistance(sourceLatitude, sourceLongitude, unit);
        }
    }
    ,
    FACTOR{

        @Override
        public double calculate(double sourceLatitude, double sourceLongitude, double targetLatitude, double targetLongitude, DistanceUnit unit) {
            double longitudeDifference = targetLongitude - sourceLongitude;
            double a = Math.toRadians(90.0 - sourceLatitude);
            double c = Math.toRadians(90.0 - targetLatitude);
            return Math.cos(a) * Math.cos(c) + Math.sin(a) * Math.sin(c) * Math.cos(Math.toRadians(longitudeDifference));
        }

        @Override
        public double normalize(double distance, DistanceUnit unit) {
            return Math.cos(distance / unit.getEarthRadius());
        }

        @Override
        public FixedSourceDistance fixedSourceDistance(double sourceLatitude, double sourceLongitude, DistanceUnit unit) {
            return new FactorFixedSourceDistance(sourceLatitude, sourceLongitude, unit);
        }
    }
    ,
    ARC{

        @Override
        public double calculate(double sourceLatitude, double sourceLongitude, double targetLatitude, double targetLongitude, DistanceUnit unit) {
            double x1 = sourceLatitude * Math.PI / 180.0;
            double x2 = targetLatitude * Math.PI / 180.0;
            double h1 = 1.0 - Math.cos(x1 - x2);
            double h2 = 1.0 - Math.cos((sourceLongitude - targetLongitude) * Math.PI / 180.0);
            double h3 = (h1 + Math.cos(x1) * Math.cos(x2) * h2) / 2.0;
            double averageLatitude = (x1 + x2) / 2.0;
            double diameter = GeoUtils.earthDiameter(averageLatitude);
            return unit.fromMeters(diameter * Math.asin(Math.min(1.0, Math.sqrt(h3))));
        }

        @Override
        public double normalize(double distance, DistanceUnit unit) {
            return distance;
        }

        @Override
        public FixedSourceDistance fixedSourceDistance(double sourceLatitude, double sourceLongitude, DistanceUnit unit) {
            return new ArcFixedSourceDistance(sourceLatitude, sourceLongitude, unit);
        }
    }
    ,
    SLOPPY_ARC{

        @Override
        public double normalize(double distance, DistanceUnit unit) {
            return distance;
        }

        @Override
        public double calculate(double sourceLatitude, double sourceLongitude, double targetLatitude, double targetLongitude, DistanceUnit unit) {
            return unit.fromMeters(SloppyMath.haversin(sourceLatitude, sourceLongitude, targetLatitude, targetLongitude) * 1000.0);
        }

        @Override
        public FixedSourceDistance fixedSourceDistance(double sourceLatitude, double sourceLongitude, DistanceUnit unit) {
            return new SloppyArcFixedSourceDistance(sourceLatitude, sourceLongitude, unit);
        }
    };

    public static final GeoDistance DEFAULT;
    private static final double MIN_LAT;
    private static final double MAX_LAT;
    private static final double MIN_LON;
    private static final double MAX_LON;
    public static final AlwaysDistanceBoundingCheck ALWAYS_INSTANCE;

    public abstract double normalize(double var1, DistanceUnit var3);

    public abstract double calculate(double var1, double var3, double var5, double var7, DistanceUnit var9);

    public abstract FixedSourceDistance fixedSourceDistance(double var1, double var3, DistanceUnit var5);

    public static DistanceBoundingCheck distanceBoundingCheck(double sourceLatitude, double sourceLongitude, double distance, DistanceUnit unit) {
        double maxLon;
        double minLon;
        double radDist = unit.toMeters(distance) / 6356752.314245;
        double radLat = Math.toRadians(sourceLatitude);
        double radLon = Math.toRadians(sourceLongitude);
        double minLat = radLat - radDist;
        double maxLat = radLat + radDist;
        if (minLat > MIN_LAT && maxLat < MAX_LAT) {
            double deltaLon = Math.asin(Math.sin(radDist) / Math.cos(radLat));
            minLon = radLon - deltaLon;
            if (minLon < MIN_LON) {
                minLon += Math.PI * 2;
            }
            if ((maxLon = radLon + deltaLon) > MAX_LON) {
                maxLon -= Math.PI * 2;
            }
        } else {
            minLat = Math.max(minLat, MIN_LAT);
            maxLat = Math.min(maxLat, MAX_LAT);
            minLon = MIN_LON;
            maxLon = MAX_LON;
        }
        GeoPoint topLeft = new GeoPoint(Math.toDegrees(maxLat), Math.toDegrees(minLon));
        GeoPoint bottomRight = new GeoPoint(Math.toDegrees(minLat), Math.toDegrees(maxLon));
        if (minLon > maxLon) {
            return new Meridian180DistanceBoundingCheck(topLeft, bottomRight);
        }
        return new SimpleDistanceBoundingCheck(topLeft, bottomRight);
    }

    public static GeoDistance fromString(String name) {
        if ("plane".equals(name = name.toLowerCase(Locale.ROOT))) {
            return PLANE;
        }
        if ("arc".equals(name)) {
            return ARC;
        }
        if ("sloppy_arc".equals(name)) {
            return SLOPPY_ARC;
        }
        if ("factor".equals(name)) {
            return FACTOR;
        }
        throw new IllegalArgumentException("No geo distance for [" + name + "]");
    }

    public static SortedNumericDoubleValues distanceValues(final MultiGeoPointValues geoPointValues, final FixedSourceDistance ... distances) {
        final GeoPointValues singleValues = FieldData.unwrapSingleton(geoPointValues);
        if (singleValues != null && distances.length == 1) {
            final Bits docsWithField = FieldData.unwrapSingletonBits(geoPointValues);
            return FieldData.singleton(new NumericDoubleValues(){

                @Override
                public double get(int docID) {
                    if (docsWithField != null && !docsWithField.get(docID)) {
                        return 0.0;
                    }
                    GeoPoint point = singleValues.get(docID);
                    return distances[0].calculate(point.lat(), point.lon());
                }
            }, docsWithField);
        }
        return new SortingNumericDoubleValues(){

            @Override
            public void setDocument(int doc) {
                geoPointValues.setDocument(doc);
                this.resize(geoPointValues.count() * distances.length);
                int valueCounter = 0;
                for (FixedSourceDistance distance : distances) {
                    for (int i = 0; i < geoPointValues.count(); ++i) {
                        GeoPoint point = geoPointValues.valueAt(i);
                        this.values[valueCounter] = distance.calculate(point.lat(), point.lon());
                        ++valueCounter;
                    }
                }
                this.sort();
            }
        };
    }

    static {
        DEFAULT = SLOPPY_ARC;
        MIN_LAT = Math.toRadians(-90.0);
        MAX_LAT = Math.toRadians(90.0);
        MIN_LON = Math.toRadians(-180.0);
        MAX_LON = Math.toRadians(180.0);
        ALWAYS_INSTANCE = new AlwaysDistanceBoundingCheck();
    }

    public static class SloppyArcFixedSourceDistance
    extends FixedSourceDistanceBase {
        public SloppyArcFixedSourceDistance(double sourceLatitude, double sourceLongitude, DistanceUnit unit) {
            super(sourceLatitude, sourceLongitude, unit);
        }

        @Override
        public double calculate(double targetLatitude, double targetLongitude) {
            return SLOPPY_ARC.calculate(this.sourceLatitude, this.sourceLongitude, targetLatitude, targetLongitude, this.unit);
        }
    }

    public static class ArcFixedSourceDistance
    extends FixedSourceDistanceBase {
        public ArcFixedSourceDistance(double sourceLatitude, double sourceLongitude, DistanceUnit unit) {
            super(sourceLatitude, sourceLongitude, unit);
        }

        @Override
        public double calculate(double targetLatitude, double targetLongitude) {
            return ARC.calculate(this.sourceLatitude, this.sourceLongitude, targetLatitude, targetLongitude, this.unit);
        }
    }

    public static abstract class FixedSourceDistanceBase
    implements FixedSourceDistance {
        protected final double sourceLatitude;
        protected final double sourceLongitude;
        protected final DistanceUnit unit;

        public FixedSourceDistanceBase(double sourceLatitude, double sourceLongitude, DistanceUnit unit) {
            this.sourceLatitude = sourceLatitude;
            this.sourceLongitude = sourceLongitude;
            this.unit = unit;
        }
    }

    public static class FactorFixedSourceDistance
    implements FixedSourceDistance {
        private final double sourceLongitude;
        private final double a;
        private final double sinA;
        private final double cosA;

        public FactorFixedSourceDistance(double sourceLatitude, double sourceLongitude, DistanceUnit unit) {
            this.sourceLongitude = sourceLongitude;
            this.a = Math.toRadians(90.0 - sourceLatitude);
            this.sinA = Math.sin(this.a);
            this.cosA = Math.cos(this.a);
        }

        @Override
        public double calculate(double targetLatitude, double targetLongitude) {
            double longitudeDifference = targetLongitude - this.sourceLongitude;
            double c = Math.toRadians(90.0 - targetLatitude);
            return this.cosA * Math.cos(c) + this.sinA * Math.sin(c) * Math.cos(Math.toRadians(longitudeDifference));
        }
    }

    public static class PlaneFixedSourceDistance
    implements FixedSourceDistance {
        private final double sourceLatitude;
        private final double sourceLongitude;
        private final double distancePerDegree;

        public PlaneFixedSourceDistance(double sourceLatitude, double sourceLongitude, DistanceUnit unit) {
            this.sourceLatitude = sourceLatitude;
            this.sourceLongitude = sourceLongitude;
            this.distancePerDegree = unit.getDistancePerDegree();
        }

        @Override
        public double calculate(double targetLatitude, double targetLongitude) {
            double px = targetLongitude - this.sourceLongitude;
            double py = targetLatitude - this.sourceLatitude;
            return Math.sqrt(px * px + py * py) * this.distancePerDegree;
        }
    }

    public static class SimpleDistanceBoundingCheck
    implements DistanceBoundingCheck {
        private final GeoPoint topLeft;
        private final GeoPoint bottomRight;

        public SimpleDistanceBoundingCheck(GeoPoint topLeft, GeoPoint bottomRight) {
            this.topLeft = topLeft;
            this.bottomRight = bottomRight;
        }

        @Override
        public boolean isWithin(double targetLatitude, double targetLongitude) {
            return targetLatitude >= this.bottomRight.lat() && targetLatitude <= this.topLeft.lat() && targetLongitude >= this.topLeft.lon() && targetLongitude <= this.bottomRight.lon();
        }

        @Override
        public GeoPoint topLeft() {
            return this.topLeft;
        }

        @Override
        public GeoPoint bottomRight() {
            return this.bottomRight;
        }
    }

    public static class Meridian180DistanceBoundingCheck
    implements DistanceBoundingCheck {
        private final GeoPoint topLeft;
        private final GeoPoint bottomRight;

        public Meridian180DistanceBoundingCheck(GeoPoint topLeft, GeoPoint bottomRight) {
            this.topLeft = topLeft;
            this.bottomRight = bottomRight;
        }

        @Override
        public boolean isWithin(double targetLatitude, double targetLongitude) {
            return targetLatitude >= this.bottomRight.lat() && targetLatitude <= this.topLeft.lat() && (targetLongitude >= this.topLeft.lon() || targetLongitude <= this.bottomRight.lon());
        }

        @Override
        public GeoPoint topLeft() {
            return this.topLeft;
        }

        @Override
        public GeoPoint bottomRight() {
            return this.bottomRight;
        }
    }

    private static class AlwaysDistanceBoundingCheck
    implements DistanceBoundingCheck {
        private AlwaysDistanceBoundingCheck() {
        }

        @Override
        public boolean isWithin(double targetLatitude, double targetLongitude) {
            return true;
        }

        @Override
        public GeoPoint topLeft() {
            return null;
        }

        @Override
        public GeoPoint bottomRight() {
            return null;
        }
    }

    public static interface DistanceBoundingCheck {
        public boolean isWithin(double var1, double var3);

        public GeoPoint topLeft();

        public GeoPoint bottomRight();
    }

    public static interface FixedSourceDistance {
        public double calculate(double var1, double var3);
    }
}

