/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.clients.producer;

import java.net.InetSocketAddress;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.kafka.clients.ClientUtils;
import org.apache.kafka.clients.Metadata;
import org.apache.kafka.clients.NetworkClient;
import org.apache.kafka.clients.producer.BufferExhaustedException;
import org.apache.kafka.clients.producer.Callback;
import org.apache.kafka.clients.producer.Partitioner;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.clients.producer.ProducerInterceptor;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.clients.producer.internals.ProducerInterceptors;
import org.apache.kafka.clients.producer.internals.RecordAccumulator;
import org.apache.kafka.clients.producer.internals.Sender;
import org.apache.kafka.common.Cluster;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.Metric;
import org.apache.kafka.common.MetricName;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.errors.ApiException;
import org.apache.kafka.common.errors.InterruptException;
import org.apache.kafka.common.errors.RecordTooLargeException;
import org.apache.kafka.common.errors.SerializationException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.errors.TopicAuthorizationException;
import org.apache.kafka.common.metrics.JmxReporter;
import org.apache.kafka.common.metrics.MetricConfig;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.metrics.MetricsReporter;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.network.ChannelBuilder;
import org.apache.kafka.common.network.Selectable;
import org.apache.kafka.common.network.Selector;
import org.apache.kafka.common.record.CompressionType;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.serialization.Serializer;
import org.apache.kafka.common.utils.AppInfoParser;
import org.apache.kafka.common.utils.KafkaThread;
import org.apache.kafka.common.utils.SystemTime;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KafkaProducer<K, V>
implements Producer<K, V> {
    private static final Logger log = LoggerFactory.getLogger(KafkaProducer.class);
    private static final AtomicInteger PRODUCER_CLIENT_ID_SEQUENCE = new AtomicInteger(1);
    private static final String JMX_PREFIX = "kafka.producer";
    private String clientId;
    private final Partitioner partitioner;
    private final int maxRequestSize;
    private final long totalMemorySize;
    private final Metadata metadata;
    private final RecordAccumulator accumulator;
    private final Sender sender;
    private final Metrics metrics;
    private final Thread ioThread;
    private final CompressionType compressionType;
    private final Sensor errors;
    private final Time time;
    private final Serializer<K> keySerializer;
    private final Serializer<V> valueSerializer;
    private final ProducerConfig producerConfig;
    private final long maxBlockTimeMs;
    private final int requestTimeoutMs;
    private final ProducerInterceptors<K, V> interceptors;

    public KafkaProducer(Map<String, Object> configs) {
        this(new ProducerConfig((Map<?, ?>)configs), null, null);
    }

    public KafkaProducer(Map<String, Object> configs, Serializer<K> keySerializer, Serializer<V> valueSerializer) {
        this(new ProducerConfig((Map<?, ?>)ProducerConfig.addSerializerToConfig(configs, keySerializer, valueSerializer)), keySerializer, valueSerializer);
    }

    public KafkaProducer(Properties properties) {
        this(new ProducerConfig((Map<?, ?>)properties), null, null);
    }

    public KafkaProducer(Properties properties, Serializer<K> keySerializer, Serializer<V> valueSerializer) {
        this(new ProducerConfig((Map<?, ?>)ProducerConfig.addSerializerToConfig(properties, keySerializer, valueSerializer)), keySerializer, valueSerializer);
    }

    private KafkaProducer(ProducerConfig config, Serializer<K> keySerializer, Serializer<V> valueSerializer) {
        try {
            log.trace("Starting the Kafka producer");
            Map<String, Object> userProvidedConfigs = config.originals();
            this.producerConfig = config;
            this.time = new SystemTime();
            this.clientId = config.getString("client.id");
            if (this.clientId.length() <= 0) {
                this.clientId = "producer-" + PRODUCER_CLIENT_ID_SEQUENCE.getAndIncrement();
            }
            LinkedHashMap<String, String> metricTags = new LinkedHashMap<String, String>();
            metricTags.put("client-id", this.clientId);
            MetricConfig metricConfig = new MetricConfig().samples(config.getInt("metrics.num.samples")).timeWindow(config.getLong("metrics.sample.window.ms"), TimeUnit.MILLISECONDS).tags(metricTags);
            List<MetricsReporter> reporters = config.getConfiguredInstances("metric.reporters", MetricsReporter.class);
            reporters.add(new JmxReporter(JMX_PREFIX));
            this.metrics = new Metrics(metricConfig, reporters, this.time);
            this.partitioner = config.getConfiguredInstance("partitioner.class", Partitioner.class);
            long retryBackoffMs = config.getLong("retry.backoff.ms");
            this.metadata = new Metadata(retryBackoffMs, config.getLong("metadata.max.age.ms"));
            this.maxRequestSize = config.getInt("max.request.size");
            this.totalMemorySize = config.getLong("buffer.memory");
            this.compressionType = CompressionType.forName(config.getString("compression.type"));
            if (userProvidedConfigs.containsKey("block.on.buffer.full")) {
                log.warn("block.on.buffer.full config is deprecated and will be removed soon. Please use max.block.ms");
                boolean blockOnBufferFull = config.getBoolean("block.on.buffer.full");
                if (blockOnBufferFull) {
                    this.maxBlockTimeMs = Long.MAX_VALUE;
                } else if (userProvidedConfigs.containsKey("metadata.fetch.timeout.ms")) {
                    log.warn("metadata.fetch.timeout.ms config is deprecated and will be removed soon. Please use max.block.ms");
                    this.maxBlockTimeMs = config.getLong("metadata.fetch.timeout.ms");
                } else {
                    this.maxBlockTimeMs = config.getLong("max.block.ms");
                }
            } else if (userProvidedConfigs.containsKey("metadata.fetch.timeout.ms")) {
                log.warn("metadata.fetch.timeout.ms config is deprecated and will be removed soon. Please use max.block.ms");
                this.maxBlockTimeMs = config.getLong("metadata.fetch.timeout.ms");
            } else {
                this.maxBlockTimeMs = config.getLong("max.block.ms");
            }
            if (userProvidedConfigs.containsKey("timeout.ms")) {
                log.warn("timeout.ms config is deprecated and will be removed soon. Please use request.timeout.ms");
                this.requestTimeoutMs = config.getInt("timeout.ms");
            } else {
                this.requestTimeoutMs = config.getInt("request.timeout.ms");
            }
            this.accumulator = new RecordAccumulator(config.getInt("batch.size"), this.totalMemorySize, this.compressionType, config.getLong("linger.ms"), retryBackoffMs, this.metrics, this.time);
            List<InetSocketAddress> addresses = ClientUtils.parseAndValidateAddresses(config.getList("bootstrap.servers"));
            this.metadata.update(Cluster.bootstrap(addresses), this.time.milliseconds());
            ChannelBuilder channelBuilder = ClientUtils.createChannelBuilder(config.values());
            NetworkClient client = new NetworkClient((Selectable)new Selector(config.getLong("connections.max.idle.ms"), this.metrics, this.time, "producer", channelBuilder), this.metadata, this.clientId, (int)config.getInt("max.in.flight.requests.per.connection"), (long)config.getLong("reconnect.backoff.ms"), (int)config.getInt("send.buffer.bytes"), (int)config.getInt("receive.buffer.bytes"), this.requestTimeoutMs, this.time);
            this.sender = new Sender(client, this.metadata, this.accumulator, config.getInt("max.in.flight.requests.per.connection") == 1, config.getInt("max.request.size"), (short)KafkaProducer.parseAcks(config.getString("acks")), config.getInt("retries"), this.metrics, new SystemTime(), this.clientId, this.requestTimeoutMs);
            String ioThreadName = "kafka-producer-network-thread" + (this.clientId.length() > 0 ? " | " + this.clientId : "");
            this.ioThread = new KafkaThread(ioThreadName, (Runnable)this.sender, true);
            this.ioThread.start();
            this.errors = this.metrics.sensor("errors");
            if (keySerializer == null) {
                this.keySerializer = config.getConfiguredInstance("key.serializer", Serializer.class);
                this.keySerializer.configure(config.originals(), true);
            } else {
                config.ignore("key.serializer");
                this.keySerializer = keySerializer;
            }
            if (valueSerializer == null) {
                this.valueSerializer = config.getConfiguredInstance("value.serializer", Serializer.class);
                this.valueSerializer.configure(config.originals(), false);
            } else {
                config.ignore("value.serializer");
                this.valueSerializer = valueSerializer;
            }
            userProvidedConfigs.put("client.id", this.clientId);
            List interceptorList = new ProducerConfig((Map<?, ?>)userProvidedConfigs).getConfiguredInstances("interceptor.classes", ProducerInterceptor.class);
            this.interceptors = interceptorList.isEmpty() ? null : new ProducerInterceptors(interceptorList);
            config.logUnused();
            AppInfoParser.registerAppInfo(JMX_PREFIX, this.clientId);
            log.debug("Kafka producer started");
        }
        catch (Throwable t) {
            this.close(0L, TimeUnit.MILLISECONDS, true);
            throw new KafkaException("Failed to construct kafka producer", t);
        }
    }

    private static int parseAcks(String acksString) {
        try {
            return acksString.trim().equalsIgnoreCase("all") ? -1 : Integer.parseInt(acksString.trim());
        }
        catch (NumberFormatException e) {
            throw new ConfigException("Invalid configuration value for 'acks': " + acksString);
        }
    }

    @Override
    public Future<RecordMetadata> send(ProducerRecord<K, V> record) {
        return this.send(record, null);
    }

    @Override
    public Future<RecordMetadata> send(ProducerRecord<K, V> record, Callback callback) {
        ProducerRecord<K, V> interceptedRecord = this.interceptors == null ? record : this.interceptors.onSend(record);
        return this.doSend(interceptedRecord, callback);
    }

    private Future<RecordMetadata> doSend(ProducerRecord<K, V> record, Callback callback) {
        TopicPartition tp = null;
        try {
            byte[] serializedValue;
            byte[] serializedKey;
            long waitedOnMetadataMs = this.waitOnMetadata(record.topic(), this.maxBlockTimeMs);
            long remainingWaitMs = Math.max(0L, this.maxBlockTimeMs - waitedOnMetadataMs);
            try {
                serializedKey = this.keySerializer.serialize(record.topic(), record.key());
            }
            catch (ClassCastException cce) {
                throw new SerializationException("Can't convert key of class " + record.key().getClass().getName() + " to class " + this.producerConfig.getClass("key.serializer").getName() + " specified in key.serializer");
            }
            try {
                serializedValue = this.valueSerializer.serialize(record.topic(), record.value());
            }
            catch (ClassCastException cce) {
                throw new SerializationException("Can't convert value of class " + record.value().getClass().getName() + " to class " + this.producerConfig.getClass("value.serializer").getName() + " specified in value.serializer");
            }
            int partition = this.partition(record, serializedKey, serializedValue, this.metadata.fetch());
            int serializedSize = 12 + Record.recordSize(serializedKey, serializedValue);
            this.ensureValidRecordSize(serializedSize);
            tp = new TopicPartition(record.topic(), partition);
            long timestamp2 = record.timestamp() == null ? this.time.milliseconds() : record.timestamp().longValue();
            log.trace("Sending record {} with callback {} to topic {} partition {}", new Object[]{record, callback, record.topic(), partition});
            Callback interceptCallback = this.interceptors == null ? callback : new InterceptorCallback<K, V>(callback, this.interceptors, tp);
            RecordAccumulator.RecordAppendResult result = this.accumulator.append(tp, timestamp2, serializedKey, serializedValue, interceptCallback, remainingWaitMs);
            if (result.batchIsFull || result.newBatchCreated) {
                log.trace("Waking up the sender since topic {} partition {} is either full or getting a new batch", (Object)record.topic(), (Object)partition);
                this.sender.wakeup();
            }
            return result.future;
        }
        catch (ApiException e) {
            log.debug("Exception occurred during message send:", (Throwable)e);
            if (callback != null) {
                callback.onCompletion(null, e);
            }
            this.errors.record();
            if (this.interceptors != null) {
                this.interceptors.onSendError(record, tp, e);
            }
            return new FutureFailure(e);
        }
        catch (InterruptedException e) {
            this.errors.record();
            if (this.interceptors != null) {
                this.interceptors.onSendError(record, tp, e);
            }
            throw new InterruptException(e);
        }
        catch (BufferExhaustedException e) {
            this.errors.record();
            this.metrics.sensor("buffer-exhausted-records").record();
            if (this.interceptors != null) {
                this.interceptors.onSendError(record, tp, e);
            }
            throw e;
        }
        catch (KafkaException e) {
            this.errors.record();
            if (this.interceptors != null) {
                this.interceptors.onSendError(record, tp, e);
            }
            throw e;
        }
        catch (Exception e) {
            if (this.interceptors != null) {
                this.interceptors.onSendError(record, tp, e);
            }
            throw e;
        }
    }

    private long waitOnMetadata(String topic, long maxWaitMs) throws InterruptedException {
        if (!this.metadata.containsTopic(topic)) {
            this.metadata.add(topic);
        }
        if (this.metadata.fetch().partitionsForTopic(topic) != null) {
            return 0L;
        }
        long begin = this.time.milliseconds();
        long remainingWaitMs = maxWaitMs;
        while (this.metadata.fetch().partitionsForTopic(topic) == null) {
            log.trace("Requesting metadata update for topic {}.", (Object)topic);
            int version = this.metadata.requestUpdate();
            this.sender.wakeup();
            this.metadata.awaitUpdate(version, remainingWaitMs);
            long elapsed = this.time.milliseconds() - begin;
            if (elapsed >= maxWaitMs) {
                throw new TimeoutException("Failed to update metadata after " + maxWaitMs + " ms.");
            }
            if (this.metadata.fetch().unauthorizedTopics().contains(topic)) {
                throw new TopicAuthorizationException(topic);
            }
            remainingWaitMs = maxWaitMs - elapsed;
        }
        return this.time.milliseconds() - begin;
    }

    private void ensureValidRecordSize(int size2) {
        if (size2 > this.maxRequestSize) {
            throw new RecordTooLargeException("The message is " + size2 + " bytes when serialized which is larger than the maximum request size you have configured with the " + "max.request.size" + " configuration.");
        }
        if ((long)size2 > this.totalMemorySize) {
            throw new RecordTooLargeException("The message is " + size2 + " bytes when serialized which is larger than the total memory buffer you have configured with the " + "buffer.memory" + " configuration.");
        }
    }

    @Override
    public void flush() {
        log.trace("Flushing accumulated records in producer.");
        this.accumulator.beginFlush();
        this.sender.wakeup();
        try {
            this.accumulator.awaitFlushCompletion();
        }
        catch (InterruptedException e) {
            throw new InterruptException("Flush interrupted.", e);
        }
    }

    @Override
    public List<PartitionInfo> partitionsFor(String topic) {
        try {
            this.waitOnMetadata(topic, this.maxBlockTimeMs);
        }
        catch (InterruptedException e) {
            throw new InterruptException(e);
        }
        return this.metadata.fetch().partitionsForTopic(topic);
    }

    @Override
    public Map<MetricName, ? extends Metric> metrics() {
        return Collections.unmodifiableMap(this.metrics.metrics());
    }

    @Override
    public void close() {
        this.close(Long.MAX_VALUE, TimeUnit.MILLISECONDS);
    }

    @Override
    public void close(long timeout, TimeUnit timeUnit) {
        this.close(timeout, timeUnit, false);
    }

    private void close(long timeout, TimeUnit timeUnit, boolean swallowException) {
        boolean invokedFromCallback;
        if (timeout < 0L) {
            throw new IllegalArgumentException("The timeout cannot be negative.");
        }
        log.info("Closing the Kafka producer with timeoutMillis = {} ms.", (Object)timeUnit.toMillis(timeout));
        AtomicReference<Throwable> firstException = new AtomicReference<Throwable>();
        boolean bl = invokedFromCallback = Thread.currentThread() == this.ioThread;
        if (timeout > 0L) {
            if (invokedFromCallback) {
                log.warn("Overriding close timeout {} ms to 0 ms in order to prevent useless blocking due to self-join. This means you have incorrectly invoked close with a non-zero timeout from the producer call-back.", (Object)timeout);
            } else {
                if (this.sender != null) {
                    this.sender.initiateClose();
                }
                if (this.ioThread != null) {
                    try {
                        this.ioThread.join(timeUnit.toMillis(timeout));
                    }
                    catch (InterruptedException t) {
                        firstException.compareAndSet(null, t);
                        log.error("Interrupted while joining ioThread", (Throwable)t);
                    }
                }
            }
        }
        if (this.sender != null && this.ioThread != null && this.ioThread.isAlive()) {
            log.info("Proceeding to force close the producer since pending requests could not be completed within timeout {} ms.", (Object)timeout);
            this.sender.forceClose();
            if (!invokedFromCallback) {
                try {
                    this.ioThread.join();
                }
                catch (InterruptedException e) {
                    firstException.compareAndSet(null, e);
                }
            }
        }
        ClientUtils.closeQuietly(this.interceptors, "producer interceptors", firstException);
        ClientUtils.closeQuietly(this.metrics, "producer metrics", firstException);
        ClientUtils.closeQuietly(this.keySerializer, "producer keySerializer", firstException);
        ClientUtils.closeQuietly(this.valueSerializer, "producer valueSerializer", firstException);
        AppInfoParser.unregisterAppInfo(JMX_PREFIX, this.clientId);
        log.debug("The Kafka producer has closed.");
        if (firstException.get() != null && !swallowException) {
            throw new KafkaException("Failed to close kafka producer", firstException.get());
        }
    }

    private int partition(ProducerRecord<K, V> record, byte[] serializedKey, byte[] serializedValue, Cluster cluster) {
        Integer partition = record.partition();
        if (partition != null) {
            List<PartitionInfo> partitions = cluster.partitionsForTopic(record.topic());
            int lastPartition = partitions.size() - 1;
            if (partition < 0 || partition > lastPartition) {
                throw new IllegalArgumentException(String.format("Invalid partition given with record: %d is not in the range [0...%d].", partition, lastPartition));
            }
            return partition;
        }
        return this.partitioner.partition(record.topic(), record.key(), serializedKey, record.value(), serializedValue, cluster);
    }

    private static class InterceptorCallback<K, V>
    implements Callback {
        private final Callback userCallback;
        private final ProducerInterceptors<K, V> interceptors;
        private final TopicPartition tp;

        public InterceptorCallback(Callback userCallback, ProducerInterceptors<K, V> interceptors, TopicPartition tp) {
            this.userCallback = userCallback;
            this.interceptors = interceptors;
            this.tp = tp;
        }

        @Override
        public void onCompletion(RecordMetadata metadata, Exception exception) {
            if (this.interceptors != null) {
                if (metadata == null) {
                    this.interceptors.onAcknowledgement(new RecordMetadata(this.tp, -1L, -1L, -1L, -1L, -1, -1), exception);
                } else {
                    this.interceptors.onAcknowledgement(metadata, exception);
                }
            }
            if (this.userCallback != null) {
                this.userCallback.onCompletion(metadata, exception);
            }
        }
    }

    private static class FutureFailure
    implements Future<RecordMetadata> {
        private final ExecutionException exception;

        public FutureFailure(Exception exception) {
            this.exception = new ExecutionException(exception);
        }

        @Override
        public boolean cancel(boolean interrupt) {
            return false;
        }

        @Override
        public RecordMetadata get() throws ExecutionException {
            throw this.exception;
        }

        @Override
        public RecordMetadata get(long timeout, TimeUnit unit) throws ExecutionException {
            throw this.exception;
        }

        @Override
        public boolean isCancelled() {
            return false;
        }

        @Override
        public boolean isDone() {
            return true;
        }
    }
}

