/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.spark.sql

import org.apache.spark.annotation.{Experimental, InterfaceStability}

/**
 * :: Experimental ::
 * A class to consume data generated by a `StreamingQuery`. Typically this is used to send the
 * generated data to external systems. Each partition will use a new deserialized instance, so you
 * usually should do all the initialization (e.g. opening a connection or initiating a transaction)
 * in the `open` method.
 *
 * Scala example:
 * {{{
 *   datasetOfString.writeStream.foreach(new ForeachWriter[String] {
 *
 *     def open(partitionId: Long, version: Long): Boolean = {
 *       // open connection
 *     }
 *
 *     def process(record: String) = {
 *       // write string to connection
 *     }
 *
 *     def close(errorOrNull: Throwable): Unit = {
 *       // close the connection
 *     }
 *   })
 * }}}
 *
 * Java example:
 * {{{
 *  datasetOfString.writeStream().foreach(new ForeachWriter<String>() {
 *
 *    @Override
 *    public boolean open(long partitionId, long version) {
 *      // open connection
 *    }
 *
 *    @Override
 *    public void process(String value) {
 *      // write string to connection
 *    }
 *
 *    @Override
 *    public void close(Throwable errorOrNull) {
 *      // close the connection
 *    }
 *  });
 * }}}
 * @since 2.0.0
 */
@Experimental
@InterfaceStability.Evolving
abstract class ForeachWriter[T] extends Serializable {

  // TODO: Move this to org.apache.spark.sql.util or consolidate this with batch API.

  /**
   * Called when starting to process one partition of new data in the executor. The `version` is
   * for data deduplication when there are failures. When recovering from a failure, some data may
   * be generated multiple times but they will always have the same version.
   *
   * If this method finds using the `partitionId` and `version` that this partition has already been
   * processed, it can return `false` to skip the further data processing. However, `close` still
   * will be called for cleaning up resources.
   *
   * @param partitionId the partition id.
   * @param version a unique id for data deduplication.
   * @return `true` if the corresponding partition and version id should be processed. `false`
   *         indicates the partition should be skipped.
   */
  def open(partitionId: Long, version: Long): Boolean

  /**
   * Called to process the data in the executor side. This method will be called only when `open`
   * returns `true`.
   */
  def process(value: T): Unit

  /**
   * Called when stopping to process one partition of new data in the executor side. This is
   * guaranteed to be called either `open` returns `true` or `false`. However,
   * `close` won't be called in the following cases:
   *  - JVM crashes without throwing a `Throwable`
   *  - `open` throws a `Throwable`.
   *
   * @param errorOrNull the error thrown during processing data or null if there was no error.
   */
  def close(errorOrNull: Throwable): Unit
}
