/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sentry.service.thrift;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.jdo.JDODataStoreException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.metastore.api.NotificationEvent;
import org.apache.sentry.binding.hive.conf.HiveAuthzConf;
import org.apache.sentry.core.common.utils.PubSub;
import org.apache.sentry.provider.db.service.persistent.PathsImage;
import org.apache.sentry.provider.db.service.persistent.SentryStore;
import org.apache.sentry.service.thrift.CounterWait;
import org.apache.sentry.service.thrift.HMSFollowerState;
import org.apache.sentry.service.thrift.HiveConnectionFactory;
import org.apache.sentry.service.thrift.HiveNotificationFetcher;
import org.apache.sentry.service.thrift.LeaderStatusMonitor;
import org.apache.sentry.service.thrift.NotificationProcessor;
import org.apache.sentry.service.thrift.SentryHMSClient;
import org.apache.sentry.service.thrift.SentryServiceState;
import org.apache.sentry.service.thrift.SentryServiceUtil;
import org.apache.sentry.service.thrift.SentryStateBank;
import org.apache.thrift.TException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HMSFollower
implements Runnable,
AutoCloseable,
PubSub.Subscriber {
    private static final Logger LOGGER = LoggerFactory.getLogger(HMSFollower.class);
    private static final String FULL_UPDATE_TRIGGER = "FULL UPDATE TRIGGER: ";
    private static boolean connectedToHms = false;
    private SentryHMSClient client;
    private final Configuration authzConf;
    private final SentryStore sentryStore;
    private final NotificationProcessor notificationProcessor;
    private boolean readyToServe;
    private final HiveNotificationFetcher notificationFetcher;
    private final boolean hdfsSyncEnabled;
    private final AtomicBoolean fullUpdateHMS = new AtomicBoolean(false);
    private final LeaderStatusMonitor leaderMonitor;
    private long hmsImageId = 0L;

    HMSFollower(Configuration conf, SentryStore store, LeaderStatusMonitor leaderMonitor, HiveConnectionFactory hiveConnectionFactory) {
        this(conf, store, leaderMonitor, hiveConnectionFactory, null);
    }

    @VisibleForTesting
    public HMSFollower(Configuration conf, SentryStore store, LeaderStatusMonitor leaderMonitor, HiveConnectionFactory hiveConnectionFactory, String authServerName) {
        LOGGER.info("HMSFollower is being initialized");
        this.readyToServe = false;
        this.authzConf = conf;
        this.leaderMonitor = leaderMonitor;
        this.sentryStore = store;
        if (authServerName == null) {
            authServerName = conf.get(HiveAuthzConf.AuthzConfVars.AUTHZ_SERVER_NAME.getVar(), conf.get(HiveAuthzConf.AuthzConfVars.AUTHZ_SERVER_NAME_DEPRECATED.getVar(), HiveAuthzConf.AuthzConfVars.AUTHZ_SERVER_NAME_DEPRECATED.getDefault()));
        }
        this.notificationProcessor = new NotificationProcessor(this.sentryStore, authServerName, this.authzConf);
        this.client = new SentryHMSClient(this.authzConf, hiveConnectionFactory);
        this.hdfsSyncEnabled = SentryServiceUtil.isHDFSSyncEnabledNoCache(this.authzConf);
        this.notificationFetcher = new HiveNotificationFetcher(this.sentryStore, hiveConnectionFactory);
        if (conf.getBoolean("sentry.hdfs.sync.full-update-pubsub", false)) {
            LOGGER.info("FULL UPDATE TRIGGER: subscribing to topic " + PubSub.Topic.HDFS_SYNC_HMS.getName());
            PubSub.getInstance().subscribe(PubSub.Topic.HDFS_SYNC_HMS, (PubSub.Subscriber)this);
        }
    }

    @VisibleForTesting
    public static boolean isConnectedToHms() {
        return connectedToHms;
    }

    @VisibleForTesting
    void setSentryHmsClient(SentryHMSClient client) {
        this.client = client;
    }

    @Override
    public void close() {
        if (this.client != null) {
            try {
                this.client.disconnect();
                SentryStateBank.disableState("HMSFollower", HMSFollowerState.CONNECTED);
            }
            catch (Exception failure) {
                LOGGER.error("Failed to close the Sentry Hms Client", (Throwable)failure);
            }
        }
        this.notificationFetcher.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        SentryStateBank.enableState("HMSFollower", HMSFollowerState.STARTED);
        try {
            long lastProcessedNotificationId;
            try {
                lastProcessedNotificationId = this.sentryStore.getLastProcessedNotificationID();
            }
            catch (Exception e) {
                LOGGER.error("Failed to get the last processed notification id from sentry store, Skipping the processing", (Throwable)e);
                SentryStateBank.disableState("HMSFollower", HMSFollowerState.STARTED);
                return;
            }
            this.wakeUpWaitingClientsForSync(lastProcessedNotificationId);
            if (!this.isLeader()) {
                this.close();
                return;
            }
            this.syncupWithHms(lastProcessedNotificationId);
        }
        finally {
            SentryStateBank.disableState("HMSFollower", HMSFollowerState.STARTED);
        }
    }

    private boolean isLeader() {
        return this.leaderMonitor == null || this.leaderMonitor.isLeader();
    }

    @VisibleForTesting
    String getAuthServerName() {
        return this.notificationProcessor.getAuthServerName();
    }

    private void syncupWithHms(long notificationId) {
        try {
            this.client.connect();
            connectedToHms = true;
            SentryStateBank.enableState("HMSFollower", HMSFollowerState.CONNECTED);
        }
        catch (Throwable e) {
            LOGGER.error("HMSFollower cannot connect to HMS!!", e);
            return;
        }
        try {
            if (this.isFullSnapshotRequired(notificationId)) {
                this.createFullSnapshot();
                return;
            }
            List<NotificationEvent> notifications = this.notificationFetcher.fetchNotifications(notificationId);
            if (this.areNotificationsOutOfSync(notifications, notificationId)) {
                this.createFullSnapshot();
                return;
            }
            if (!this.readyToServe) {
                System.out.println("Sentry HMS support is ready");
                this.readyToServe = true;
            }
            this.processNotifications(notifications);
        }
        catch (TException e) {
            LOGGER.error("An error occurred while fetching HMS notifications: {}", (Object)e.getMessage());
            this.close();
        }
        catch (Throwable t) {
            LOGGER.error("Exception in HMSFollower! Caused by: " + t.getMessage(), t);
            this.close();
        }
    }

    private boolean isFullSnapshotRequired(long latestSentryNotificationId) throws Exception {
        if (this.sentryStore.isHmsNotificationEmpty()) {
            return true;
        }
        if (this.hdfsSyncEnabled && this.sentryStore.isAuthzPathsSnapshotEmpty()) {
            LOGGER.debug("HDFSSync is enabled and MAuthzPathsSnapshotId table is empty. Need to request a full snapshot");
            return true;
        }
        long currentHmsNotificationId = this.notificationFetcher.getCurrentNotificationId();
        if (currentHmsNotificationId < latestSentryNotificationId) {
            LOGGER.info("The latest notification ID on HMS is less than the latest notification ID processed by Sentry. Need to request a full HMS snapshot.");
            return true;
        }
        if (this.fullUpdateHMS.compareAndSet(true, false)) {
            LOGGER.info("FULL UPDATE TRIGGER: initiating full HMS snapshot request");
            return true;
        }
        return false;
    }

    private boolean areNotificationsOutOfSync(Collection<NotificationEvent> events, long latestProcessedId) {
        if (events.isEmpty()) {
            return false;
        }
        List eventList = (List)events;
        long firstNotificationId = ((NotificationEvent)eventList.get(0)).getEventId();
        if (firstNotificationId > latestProcessedId + 1L) {
            LOGGER.info("Current HMS notifications are out-of-sync with latest Sentry processednotifications. Need to request a full HMS snapshot.");
            return true;
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long createFullSnapshot() throws Exception {
        LOGGER.debug("Attempting to take full HMS snapshot");
        Preconditions.checkState((!SentryStateBank.isEnabled("SentryService", SentryServiceState.FULL_UPDATE_RUNNING) ? 1 : 0) != 0, (Object)"HMSFollower shown loading full snapshot when it should not be.");
        try {
            SentryStateBank.enableState("SentryService", SentryServiceState.FULL_UPDATE_RUNNING);
            PathsImage snapshotInfo = this.client.getFullSnapshot();
            if (snapshotInfo.getPathImage().isEmpty()) {
                long l = snapshotInfo.getId();
                return l;
            }
            if (!this.isLeader()) {
                long l = 0L;
                return l;
            }
            try {
                LOGGER.debug("Persisting HMS path full snapshot");
                if (this.hdfsSyncEnabled) {
                    this.sentryStore.persistFullPathsImage(snapshotInfo.getPathImage(), snapshotInfo.getId());
                } else {
                    this.sentryStore.setLastProcessedNotificationID(snapshotInfo.getId());
                }
            }
            catch (Exception failure) {
                LOGGER.error("Received exception while persisting HMS path full snapshot ");
                throw failure;
            }
            this.wakeUpWaitingClientsForSync(snapshotInfo.getId());
            LOGGER.info("Sentry HMS support is ready");
            long l = snapshotInfo.getId();
            return l;
        }
        finally {
            SentryStateBank.disableState("SentryService", SentryServiceState.FULL_UPDATE_RUNNING);
        }
    }

    public void processNotifications(Collection<NotificationEvent> events) throws Exception {
        if (events.isEmpty()) {
            return;
        }
        for (NotificationEvent event : events) {
            boolean isNotificationProcessed = false;
            try {
                if (!this.isLeader()) {
                    return;
                }
                isNotificationProcessed = this.notificationProcessor.processNotificationEvent(event);
            }
            catch (Exception e) {
                if (e.getCause() instanceof JDODataStoreException) {
                    LOGGER.info("Received JDO Storage Exception, Could be because of processing duplicate notification");
                    if (event.getEventId() <= this.sentryStore.getLastProcessedNotificationID()) {
                        LOGGER.error("Received event with Id: {} which is smaller then the ID persisted in store", (Object)event.getEventId());
                        break;
                    }
                }
                LOGGER.error("Processing the notification with ID:{} failed with exception {}", (Object)event.getEventId(), (Object)e);
            }
            if (!isNotificationProcessed) {
                try {
                    LOGGER.debug("Explicitly Persisting Notification ID:{}", (Object)event.getEventId());
                    this.sentryStore.persistLastProcessedNotificationID(event.getEventId());
                }
                catch (Exception failure) {
                    LOGGER.error("Received exception while persisting the notification ID " + event.getEventId());
                    throw failure;
                }
            }
            this.wakeUpWaitingClientsForSync(event.getEventId());
        }
    }

    private void wakeUpWaitingClientsForSync(long eventId) {
        CounterWait counterWait = this.sentryStore.getCounterWait();
        if (counterWait == null) {
            return;
        }
        long lastHMSSnapshotId = this.hmsImageId;
        try {
            lastHMSSnapshotId = this.sentryStore.getLastProcessedImageID();
        }
        catch (Exception e) {
            counterWait.update(eventId);
            LOGGER.error("Failed to get the last processed HMS image id from sentry store");
            return;
        }
        if (lastHMSSnapshotId > this.hmsImageId) {
            counterWait.reset(eventId);
            this.hmsImageId = lastHMSSnapshotId;
        }
        counterWait.update(eventId);
    }

    public void onMessage(PubSub.Topic topic, String message) {
        Preconditions.checkArgument((topic == PubSub.Topic.HDFS_SYNC_HMS ? 1 : 0) != 0, (String)"Unexpected topic %s instead of %s", (Object[])new Object[]{topic, PubSub.Topic.HDFS_SYNC_HMS});
        LOGGER.info("FULL UPDATE TRIGGER: Received [{}, {}] notification", (Object)topic, (Object)message);
        this.fullUpdateHMS.set(true);
    }
}

