/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sentry.provider.db.generic;

import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Table;
import java.util.HashSet;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.sentry.provider.common.TableCache;
import org.apache.sentry.provider.db.generic.service.thrift.SentryGenericServiceClient;
import org.apache.sentry.provider.db.generic.service.thrift.SentryGenericServiceClientFactory;
import org.apache.sentry.provider.db.generic.service.thrift.TSentryPrivilege;
import org.apache.sentry.provider.db.generic.service.thrift.TSentryRole;
import org.apache.sentry.provider.db.generic.tools.command.TSentryPrivilegeConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class UpdatableCache
implements TableCache,
AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(UpdatableCache.class);
    private final Timer timer = new Timer();
    private boolean initialized = false;
    private static Timer savedTimer;
    private final String componentType;
    private final String serviceName;
    private final long cacheTtlNs;
    private final int allowedUpdateFailuresCount;
    private final Configuration conf;
    private final TSentryPrivilegeConverter tSentryPrivilegeConverter;
    private volatile long lastRefreshedNs = 0L;
    private int consecutiveUpdateFailuresCount = 0;
    private volatile Table<String, String, Set<String>> table;

    UpdatableCache(Configuration conf, String componentType, String serviceName, TSentryPrivilegeConverter tSentryPrivilegeConverter) {
        this.conf = conf;
        this.componentType = componentType;
        this.serviceName = serviceName;
        this.tSentryPrivilegeConverter = tSentryPrivilegeConverter;
        this.cacheTtlNs = TimeUnit.MILLISECONDS.toNanos(conf.getLong("sentry.provider.backend.generic.cache.ttl.ms", 30000L));
        this.allowedUpdateFailuresCount = conf.getInt("sentry.provider.backend.generic.cache.update.failures.count", 3);
    }

    public Table<String, String, Set<String>> getCache() {
        return this.table;
    }

    private Table<String, String, Set<String>> loadFromRemote() throws Exception {
        HashBasedTable tempCache = HashBasedTable.create();
        String requestor = UserGroupInformation.getLoginUser().getShortUserName();
        try (SentryGenericServiceClient client = this.getClient();){
            Set<TSentryRole> tSentryRoles = client.listAllRoles(requestor, this.componentType);
            for (TSentryRole tSentryRole : tSentryRoles) {
                String roleName = tSentryRole.getRoleName();
                Set<TSentryPrivilege> tSentryPrivileges = client.listAllPrivilegesByRoleName(requestor, roleName, this.componentType, this.serviceName);
                for (String group : tSentryRole.getGroups()) {
                    HashSet<String> currentPrivileges = (HashSet<String>)tempCache.get((Object)group, (Object)roleName);
                    if (currentPrivileges == null) {
                        currentPrivileges = new HashSet<String>();
                        tempCache.put((Object)group, (Object)roleName, currentPrivileges);
                    }
                    for (TSentryPrivilege tSentryPrivilege : tSentryPrivileges) {
                        currentPrivileges.add(this.tSentryPrivilegeConverter.toString(tSentryPrivilege));
                    }
                }
            }
            HashBasedTable hashBasedTable = tempCache;
            return hashBasedTable;
        }
    }

    private SentryGenericServiceClient getClient() throws Exception {
        return SentryGenericServiceClientFactory.create(this.conf);
    }

    void startUpdateThread(boolean blockUntilFirstReload) throws Exception {
        if (blockUntilFirstReload) {
            this.reloadData();
        }
        if (this.initialized) {
            LOGGER.info("Already initialized");
            return;
        }
        this.initialized = true;
        if (savedTimer != null) {
            LOGGER.debug("Resetting saved timer");
            savedTimer.cancel();
        }
        savedTimer = this.timer;
        long refreshIntervalMs = TimeUnit.NANOSECONDS.toMillis(this.cacheTtlNs);
        this.timer.scheduleAtFixedRate(new TimerTask(){

            @Override
            public void run() {
                if (UpdatableCache.this.shouldRefresh()) {
                    try {
                        LOGGER.debug("Loading all data.");
                        UpdatableCache.this.reloadData();
                    }
                    catch (Exception e) {
                        LOGGER.warn("Exception while updating data from DB", (Throwable)e);
                        UpdatableCache.this.revokeAllPrivilegesIfRequired();
                    }
                }
            }
        }, blockUntilFirstReload ? refreshIntervalMs : 0L, refreshIntervalMs);
    }

    private void revokeAllPrivilegesIfRequired() {
        if (++this.consecutiveUpdateFailuresCount > this.allowedUpdateFailuresCount) {
            this.consecutiveUpdateFailuresCount = 0;
            this.table = HashBasedTable.create();
            LOGGER.error("Failed to update roles and privileges cache for " + this.consecutiveUpdateFailuresCount + " times. Revoking all privileges from cache, which will cause all authorization requests to fail.");
        }
    }

    private void reloadData() throws Exception {
        this.table = this.loadFromRemote();
        this.lastRefreshedNs = System.nanoTime();
    }

    private boolean shouldRefresh() {
        long currentTimeNs = System.nanoTime();
        return this.lastRefreshedNs + this.cacheTtlNs < currentTimeNs;
    }

    public static void disable() {
        if (savedTimer != null) {
            LOGGER.info("Disabling timer");
            savedTimer.cancel();
        }
    }

    @Override
    public void close() {
        this.timer.cancel();
        savedTimer = null;
        LOGGER.info("Closed Updatable Cache");
    }
}

