/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.percolator;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import org.apache.lucene.index.PrefixCodedTerms;
import org.apache.lucene.index.Term;
import org.apache.lucene.queries.BlendedTermQuery;
import org.apache.lucene.queries.CommonTermsQuery;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.BoostQuery;
import org.apache.lucene.search.ConstantScoreQuery;
import org.apache.lucene.search.DisjunctionMaxQuery;
import org.apache.lucene.search.IndexOrDocValuesQuery;
import org.apache.lucene.search.MatchAllDocsQuery;
import org.apache.lucene.search.MatchNoDocsQuery;
import org.apache.lucene.search.MultiPhraseQuery;
import org.apache.lucene.search.PhraseQuery;
import org.apache.lucene.search.PointRangeQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.SynonymQuery;
import org.apache.lucene.search.TermInSetQuery;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.spans.SpanFirstQuery;
import org.apache.lucene.search.spans.SpanNearQuery;
import org.apache.lucene.search.spans.SpanNotQuery;
import org.apache.lucene.search.spans.SpanOrQuery;
import org.apache.lucene.search.spans.SpanQuery;
import org.apache.lucene.search.spans.SpanTermQuery;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.NumericUtils;
import org.elasticsearch.Version;
import org.elasticsearch.common.logging.LoggerMessageFormat;
import org.elasticsearch.common.lucene.search.function.FunctionScoreQuery;
import org.elasticsearch.index.search.ESToParentBlockJoinQuery;

final class QueryAnalyzer {
    private static final Map<Class<? extends Query>, BiFunction<Query, Version, Result>> queryProcessors;

    private QueryAnalyzer() {
    }

    static Result analyze(Query query, Version indexVersion) {
        BiFunction<Query, Version, Result> queryProcessor;
        Class<?> queryClass = query.getClass();
        if (queryClass.isAnonymousClass()) {
            queryClass = queryClass.getSuperclass();
        }
        if ((queryProcessor = queryProcessors.get(queryClass)) != null) {
            return queryProcessor.apply(query, indexVersion);
        }
        throw new UnsupportedQueryException(query);
    }

    private static BiFunction<Query, Version, Result> matchNoDocsQuery() {
        return (query, version) -> new Result(true, Collections.emptySet(), 0);
    }

    private static BiFunction<Query, Version, Result> matchAllDocsQuery() {
        return (query, version) -> new Result(true, true);
    }

    private static BiFunction<Query, Version, Result> constantScoreQuery() {
        return (query, boosts) -> {
            Query wrappedQuery = ((ConstantScoreQuery)query).getQuery();
            return QueryAnalyzer.analyze(wrappedQuery, boosts);
        };
    }

    private static BiFunction<Query, Version, Result> boostQuery() {
        return (query, version) -> {
            Query wrappedQuery = ((BoostQuery)query).getQuery();
            return QueryAnalyzer.analyze(wrappedQuery, version);
        };
    }

    private static BiFunction<Query, Version, Result> termQuery() {
        return (query, version) -> {
            TermQuery termQuery = (TermQuery)query;
            return new Result(true, Collections.singleton(new QueryExtraction(termQuery.getTerm())), 1);
        };
    }

    private static BiFunction<Query, Version, Result> termInSetQuery() {
        return (query, version) -> {
            TermInSetQuery termInSetQuery = (TermInSetQuery)query;
            HashSet<QueryExtraction> terms = new HashSet<QueryExtraction>();
            PrefixCodedTerms.TermIterator iterator = termInSetQuery.getTermData().iterator();
            BytesRef term = iterator.next();
            while (term != null) {
                terms.add(new QueryExtraction(new Term(iterator.field(), term)));
                term = iterator.next();
            }
            return new Result(true, terms, Math.min(1, terms.size()));
        };
    }

    private static BiFunction<Query, Version, Result> synonymQuery() {
        return (query, version) -> {
            Set<QueryExtraction> terms = ((SynonymQuery)query).getTerms().stream().map(QueryExtraction::new).collect(Collectors.toSet());
            return new Result(true, terms, Math.min(1, terms.size()));
        };
    }

    private static BiFunction<Query, Version, Result> commonTermsQuery() {
        return (query, version) -> {
            Set<QueryExtraction> terms = ((CommonTermsQuery)query).getTerms().stream().map(QueryExtraction::new).collect(Collectors.toSet());
            return new Result(false, terms, Math.min(1, terms.size()));
        };
    }

    private static BiFunction<Query, Version, Result> blendedTermQuery() {
        return (query, version) -> {
            Set<QueryExtraction> terms = ((BlendedTermQuery)query).getTerms().stream().map(QueryExtraction::new).collect(Collectors.toSet());
            return new Result(true, terms, Math.min(1, terms.size()));
        };
    }

    private static BiFunction<Query, Version, Result> phraseQuery() {
        return (query, version) -> {
            Term[] terms = ((PhraseQuery)query).getTerms();
            if (terms.length == 0) {
                return new Result(true, Collections.emptySet(), 0);
            }
            if (version.onOrAfter(Version.V_6_1_0)) {
                Set<QueryExtraction> extractions = Arrays.stream(terms).map(QueryExtraction::new).collect(Collectors.toSet());
                return new Result(false, extractions, extractions.size());
            }
            Term longestTerm = terms[0];
            for (Term term : terms) {
                if (longestTerm.bytes().length >= term.bytes().length) continue;
                longestTerm = term;
            }
            return new Result(false, Collections.singleton(new QueryExtraction(longestTerm)), 1);
        };
    }

    private static BiFunction<Query, Version, Result> multiPhraseQuery() {
        return (query, version) -> {
            Term[][] terms = ((MultiPhraseQuery)query).getTermArrays();
            if (terms.length == 0) {
                return new Result(true, Collections.emptySet(), 0);
            }
            BooleanQuery.Builder builder = new BooleanQuery.Builder();
            for (Term[] termArr : terms) {
                BooleanQuery.Builder subBuilder = new BooleanQuery.Builder();
                for (Term term : termArr) {
                    subBuilder.add(new TermQuery(term), BooleanClause.Occur.SHOULD);
                }
                builder.add(subBuilder.build(), BooleanClause.Occur.FILTER);
            }
            return QueryAnalyzer.booleanQuery().apply(builder.build(), (Version)version).unverify();
        };
    }

    private static BiFunction<Query, Version, Result> spanTermQuery() {
        return (query, version) -> {
            Term term = ((SpanTermQuery)query).getTerm();
            return new Result(true, Collections.singleton(new QueryExtraction(term)), 1);
        };
    }

    private static BiFunction<Query, Version, Result> spanNearQuery() {
        return (query, version) -> {
            SpanNearQuery spanNearQuery = (SpanNearQuery)query;
            if (version.onOrAfter(Version.V_6_1_0)) {
                BooleanQuery.Builder builder = new BooleanQuery.Builder();
                for (SpanQuery clause : spanNearQuery.getClauses()) {
                    builder.add(clause, BooleanClause.Occur.FILTER);
                }
                return QueryAnalyzer.booleanQuery().apply(builder.build(), (Version)version).unverify();
            }
            Result bestClause = null;
            for (SpanQuery clause : spanNearQuery.getClauses()) {
                Result temp = QueryAnalyzer.analyze(clause, version);
                bestClause = QueryAnalyzer.selectBestResult(temp, bestClause);
            }
            return bestClause;
        };
    }

    private static BiFunction<Query, Version, Result> spanOrQuery() {
        return (query, version) -> {
            SpanOrQuery spanOrQuery = (SpanOrQuery)query;
            BooleanQuery.Builder builder = new BooleanQuery.Builder();
            for (SpanQuery clause : spanOrQuery.getClauses()) {
                builder.add(clause, BooleanClause.Occur.SHOULD);
            }
            return QueryAnalyzer.booleanQuery().apply(builder.build(), (Version)version);
        };
    }

    private static BiFunction<Query, Version, Result> spanNotQuery() {
        return (query, version) -> {
            Result result = QueryAnalyzer.analyze(((SpanNotQuery)query).getInclude(), version);
            return new Result(false, result.extractions, result.minimumShouldMatch);
        };
    }

    private static BiFunction<Query, Version, Result> spanFirstQuery() {
        return (query, version) -> {
            Result result = QueryAnalyzer.analyze(((SpanFirstQuery)query).getMatch(), version);
            return new Result(false, result.extractions, result.minimumShouldMatch);
        };
    }

    private static BiFunction<Query, Version, Result> booleanQuery() {
        return (query, version) -> {
            Result result;
            BooleanQuery bq = (BooleanQuery)query;
            int minimumShouldMatch = bq.getMinimumNumberShouldMatch();
            ArrayList<Query> requiredClauses = new ArrayList<Query>();
            ArrayList<Query> optionalClauses = new ArrayList<Query>();
            boolean hasProhibitedClauses = false;
            for (BooleanClause clause : bq.clauses()) {
                if (clause.isRequired()) {
                    requiredClauses.add(clause.getQuery());
                    continue;
                }
                if (clause.isProhibited()) {
                    hasProhibitedClauses = true;
                    continue;
                }
                assert (clause.getOccur() == BooleanClause.Occur.SHOULD);
                optionalClauses.add(clause.getQuery());
            }
            if (minimumShouldMatch > optionalClauses.size() || requiredClauses.isEmpty() && optionalClauses.isEmpty()) {
                return new Result(false, Collections.emptySet(), 0);
            }
            if (requiredClauses.size() > 0) {
                if (minimumShouldMatch > 0) {
                    BooleanQuery.Builder minShouldMatchQuery = new BooleanQuery.Builder();
                    minShouldMatchQuery.setMinimumNumberShouldMatch(minimumShouldMatch);
                    for (Query q : optionalClauses) {
                        minShouldMatchQuery.add(q, BooleanClause.Occur.SHOULD);
                    }
                    requiredClauses.add(minShouldMatchQuery.build());
                    optionalClauses.clear();
                    minimumShouldMatch = 0;
                } else {
                    optionalClauses.clear();
                }
            }
            if (requiredClauses.size() > 0) {
                assert (optionalClauses.isEmpty());
                assert (minimumShouldMatch == 0);
                result = QueryAnalyzer.handleConjunctionQuery(requiredClauses, version);
            } else {
                assert (requiredClauses.isEmpty());
                if (minimumShouldMatch == 0) {
                    minimumShouldMatch = 1;
                }
                result = QueryAnalyzer.handleDisjunctionQuery(optionalClauses, minimumShouldMatch, version);
            }
            if (hasProhibitedClauses) {
                result = result.unverify();
            }
            return result;
        };
    }

    private static BiFunction<Query, Version, Result> disjunctionMaxQuery() {
        return (query, version) -> {
            List<Query> disjuncts = ((DisjunctionMaxQuery)query).getDisjuncts();
            if (disjuncts.isEmpty()) {
                return new Result(false, Collections.emptySet(), 0);
            }
            return QueryAnalyzer.handleDisjunctionQuery(disjuncts, 1, version);
        };
    }

    private static BiFunction<Query, Version, Result> functionScoreQuery() {
        return (query, version) -> {
            boolean verified;
            FunctionScoreQuery functionScoreQuery = (FunctionScoreQuery)query;
            Result result = QueryAnalyzer.analyze(functionScoreQuery.getSubQuery(), version);
            boolean bl = verified = result.verified && functionScoreQuery.getMinScore() == null;
            if (result.matchAllDocs) {
                return new Result(result.matchAllDocs, verified);
            }
            return new Result(verified, result.extractions, result.minimumShouldMatch);
        };
    }

    private static BiFunction<Query, Version, Result> pointRangeQuery() {
        return (query, version) -> {
            byte[] upperPoint;
            PointRangeQuery pointRangeQuery = (PointRangeQuery)query;
            if (pointRangeQuery.getNumDims() != 1) {
                throw new UnsupportedQueryException((Query)query);
            }
            byte[] lowerPoint = pointRangeQuery.getLowerPoint();
            if (new BytesRef(lowerPoint).compareTo(new BytesRef(upperPoint = pointRangeQuery.getUpperPoint())) > 0) {
                return new Result(true, Collections.emptySet(), 0);
            }
            byte[] interval = new byte[16];
            NumericUtils.subtract(16, 0, QueryAnalyzer.prepad(upperPoint), QueryAnalyzer.prepad(lowerPoint), interval);
            return new Result(false, Collections.singleton(new QueryExtraction(new Range(pointRangeQuery.getField(), lowerPoint, upperPoint, interval))), 1);
        };
    }

    private static byte[] prepad(byte[] original) {
        int offset = 16 - original.length;
        byte[] result = new byte[16];
        System.arraycopy(original, 0, result, offset, original.length);
        return result;
    }

    private static BiFunction<Query, Version, Result> indexOrDocValuesQuery() {
        return (query, version) -> {
            IndexOrDocValuesQuery indexOrDocValuesQuery = (IndexOrDocValuesQuery)query;
            return QueryAnalyzer.analyze(indexOrDocValuesQuery.getIndexQuery(), version);
        };
    }

    private static BiFunction<Query, Version, Result> toParentBlockJoinQuery() {
        return (query, version) -> {
            ESToParentBlockJoinQuery toParentBlockJoinQuery = (ESToParentBlockJoinQuery)query;
            Result result = QueryAnalyzer.analyze(toParentBlockJoinQuery.getChildQuery(), version);
            return new Result(false, result.extractions, result.minimumShouldMatch);
        };
    }

    private static Result handleConjunctionQuery(List<Query> conjunctions, Version version) {
        UnsupportedQueryException uqe = null;
        ArrayList<Result> results = new ArrayList<Result>(conjunctions.size());
        boolean success = false;
        for (Query query : conjunctions) {
            try {
                Result subResult = QueryAnalyzer.analyze(query, version);
                if (subResult.isMatchNoDocs()) {
                    return subResult;
                }
                results.add(subResult);
                success = true;
            }
            catch (UnsupportedQueryException e) {
                uqe = e;
            }
        }
        if (!success) {
            if (uqe != null) {
                throw uqe;
            }
            return new Result(true, Collections.emptySet(), 0);
        }
        Result result = QueryAnalyzer.handleConjunction(results, version);
        if (uqe != null) {
            result = result.unverify();
        }
        return result;
    }

    private static Result handleConjunction(List<Result> conjunctions, Version version) {
        if (conjunctions.isEmpty()) {
            throw new IllegalArgumentException("Must have at least on conjunction sub result");
        }
        if (version.onOrAfter(Version.V_6_1_0)) {
            for (Result subResult : conjunctions) {
                if (!subResult.isMatchNoDocs()) continue;
                return subResult;
            }
            int msm = 0;
            boolean verified = true;
            boolean matchAllDocs = true;
            boolean hasDuplicateTerms = false;
            HashSet<QueryExtraction> extractions = new HashSet<QueryExtraction>();
            HashSet<String> seenRangeFields = new HashSet<String>();
            for (Result result : conjunctions) {
                int resultMsm = result.minimumShouldMatch;
                for (QueryExtraction queryExtraction : result.extractions) {
                    if (queryExtraction.range != null) {
                        resultMsm = seenRangeFields.add(queryExtraction.range.fieldName) ? 1 : 0;
                    }
                    if (!extractions.contains(queryExtraction)) continue;
                    resultMsm = 0;
                    verified = false;
                    break;
                }
                msm += resultMsm;
                if (!result.verified || result.minimumShouldMatch < result.extractions.size()) {
                    verified = false;
                }
                matchAllDocs &= result.matchAllDocs;
                extractions.addAll(result.extractions);
            }
            if (matchAllDocs) {
                return new Result(matchAllDocs, verified);
            }
            return new Result(verified, extractions, hasDuplicateTerms ? 1 : msm);
        }
        Result bestClause = null;
        for (Result result : conjunctions) {
            bestClause = QueryAnalyzer.selectBestResult(result, bestClause);
        }
        return bestClause;
    }

    private static Result handleDisjunctionQuery(List<Query> disjunctions, int requiredShouldClauses, Version version) {
        ArrayList<Result> subResults = new ArrayList<Result>();
        for (Query query : disjunctions) {
            Result subResult = QueryAnalyzer.analyze(query, version);
            subResults.add(subResult);
        }
        return QueryAnalyzer.handleDisjunction(subResults, requiredShouldClauses, version);
    }

    private static Result handleDisjunction(List<Result> disjunctions, int requiredShouldClauses, Version version) {
        List<Object> clauses = new ArrayList(disjunctions.size());
        boolean verified = version.before(Version.V_6_1_0) ? requiredShouldClauses <= 1 : true;
        int numMatchAllClauses = 0;
        boolean hasRangeExtractions = false;
        boolean hasDuplicateTerms = false;
        HashSet<QueryExtraction> terms = new HashSet<QueryExtraction>();
        for (int i = 0; i < disjunctions.size(); ++i) {
            Result subResult = disjunctions.get(i);
            if (!subResult.verified || subResult.minimumShouldMatch > 1 || subResult.extractions.size() > 1 && requiredShouldClauses > 1) {
                verified = false;
            }
            if (subResult.matchAllDocs) {
                ++numMatchAllClauses;
            }
            int resultMsm = subResult.minimumShouldMatch;
            for (QueryExtraction extraction : subResult.extractions) {
                if (terms.add(extraction)) continue;
                verified = false;
                hasDuplicateTerms = true;
            }
            if (!hasRangeExtractions) {
                hasRangeExtractions = subResult.extractions.stream().anyMatch(qe -> qe.range != null);
            }
            clauses.add(resultMsm);
        }
        boolean matchAllDocs = numMatchAllClauses > 0 && numMatchAllClauses >= requiredShouldClauses;
        int msm = 0;
        if (version.onOrAfter(Version.V_6_1_0) && !hasRangeExtractions) {
            clauses = clauses.stream().filter(val -> val > 0).sorted().collect(Collectors.toList());
            if (hasDuplicateTerms) {
                msm = (Integer)clauses.get(0);
            } else {
                int limit = Math.min(clauses.size(), Math.max(1, requiredShouldClauses));
                for (int i = 0; i < limit; ++i) {
                    msm += ((Integer)clauses.get(i)).intValue();
                }
            }
        } else {
            msm = 1;
        }
        if (matchAllDocs) {
            return new Result(matchAllDocs, verified);
        }
        return new Result(verified, terms, msm);
    }

    static Result selectBestResult(Result result1, Result result2) {
        int extraction2ShortestTerm;
        assert (result1 != null || result2 != null);
        if (result1 == null) {
            return result2;
        }
        if (result2 == null) {
            return result1;
        }
        if (result1.matchAllDocs) {
            Result result = result2;
            if (!result1.verified) {
                result = result.unverify();
            }
            return result;
        }
        if (result2.matchAllDocs) {
            Result result = result1;
            if (!result2.verified) {
                result = result.unverify();
            }
            return result;
        }
        boolean onlyRangeBasedExtractions = true;
        for (QueryExtraction clause : result1.extractions) {
            if (clause.term == null) continue;
            onlyRangeBasedExtractions = false;
            break;
        }
        for (QueryExtraction clause : result2.extractions) {
            if (clause.term == null) continue;
            onlyRangeBasedExtractions = false;
            break;
        }
        if (onlyRangeBasedExtractions) {
            BytesRef extraction1SmallestRange = QueryAnalyzer.smallestRange(result1.extractions);
            BytesRef extraction2SmallestRange = QueryAnalyzer.smallestRange(result2.extractions);
            if (extraction1SmallestRange == null) {
                return result2.unverify();
            }
            if (extraction2SmallestRange == null) {
                return result1.unverify();
            }
            if (extraction1SmallestRange.compareTo(extraction2SmallestRange) <= 0) {
                return result1.unverify();
            }
            return result2.unverify();
        }
        int extraction1ShortestTerm = QueryAnalyzer.minTermLength(result1.extractions);
        if (extraction1ShortestTerm >= (extraction2ShortestTerm = QueryAnalyzer.minTermLength(result2.extractions))) {
            return result1.unverify();
        }
        return result2.unverify();
    }

    private static int minTermLength(Set<QueryExtraction> extractions) {
        if (extractions.stream().filter(queryExtraction -> queryExtraction.term != null).count() == 0L && extractions.stream().filter(queryExtraction -> queryExtraction.range != null).count() > 0L) {
            return Integer.MIN_VALUE;
        }
        int min = Integer.MAX_VALUE;
        for (QueryExtraction qt : extractions) {
            if (qt.term == null) continue;
            min = Math.min(min, qt.bytes().length);
        }
        return min;
    }

    private static BytesRef smallestRange(Set<QueryExtraction> terms) {
        BytesRef min = null;
        for (QueryExtraction qt : terms) {
            if (qt.range == null || min != null && qt.range.interval.compareTo(min) >= 0) continue;
            min = qt.range.interval;
        }
        return min;
    }

    static {
        HashMap<Class, BiFunction<Query, Version, Result>> map = new HashMap<Class, BiFunction<Query, Version, Result>>();
        map.put(MatchNoDocsQuery.class, QueryAnalyzer.matchNoDocsQuery());
        map.put(MatchAllDocsQuery.class, QueryAnalyzer.matchAllDocsQuery());
        map.put(ConstantScoreQuery.class, QueryAnalyzer.constantScoreQuery());
        map.put(BoostQuery.class, QueryAnalyzer.boostQuery());
        map.put(TermQuery.class, QueryAnalyzer.termQuery());
        map.put(TermInSetQuery.class, QueryAnalyzer.termInSetQuery());
        map.put(CommonTermsQuery.class, QueryAnalyzer.commonTermsQuery());
        map.put(BlendedTermQuery.class, QueryAnalyzer.blendedTermQuery());
        map.put(PhraseQuery.class, QueryAnalyzer.phraseQuery());
        map.put(MultiPhraseQuery.class, QueryAnalyzer.multiPhraseQuery());
        map.put(SpanTermQuery.class, QueryAnalyzer.spanTermQuery());
        map.put(SpanNearQuery.class, QueryAnalyzer.spanNearQuery());
        map.put(SpanOrQuery.class, QueryAnalyzer.spanOrQuery());
        map.put(SpanFirstQuery.class, QueryAnalyzer.spanFirstQuery());
        map.put(SpanNotQuery.class, QueryAnalyzer.spanNotQuery());
        map.put(BooleanQuery.class, QueryAnalyzer.booleanQuery());
        map.put(DisjunctionMaxQuery.class, QueryAnalyzer.disjunctionMaxQuery());
        map.put(SynonymQuery.class, QueryAnalyzer.synonymQuery());
        map.put(FunctionScoreQuery.class, QueryAnalyzer.functionScoreQuery());
        map.put(PointRangeQuery.class, QueryAnalyzer.pointRangeQuery());
        map.put(IndexOrDocValuesQuery.class, QueryAnalyzer.indexOrDocValuesQuery());
        map.put(ESToParentBlockJoinQuery.class, QueryAnalyzer.toParentBlockJoinQuery());
        queryProcessors = Collections.unmodifiableMap(map);
    }

    static class Range {
        final String fieldName;
        final byte[] lowerPoint;
        final byte[] upperPoint;
        final BytesRef interval;

        Range(String fieldName, byte[] lowerPoint, byte[] upperPoint, byte[] interval) {
            this.fieldName = fieldName;
            this.lowerPoint = lowerPoint;
            this.upperPoint = upperPoint;
            this.interval = new BytesRef(interval);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Range range = (Range)o;
            return Objects.equals(this.fieldName, range.fieldName) && Arrays.equals(this.lowerPoint, range.lowerPoint) && Arrays.equals(this.upperPoint, range.upperPoint);
        }

        public int hashCode() {
            int result = 1;
            result += 31 * this.fieldName.hashCode();
            result += Arrays.hashCode(this.lowerPoint);
            return result += Arrays.hashCode(this.upperPoint);
        }

        public String toString() {
            return "Range{, fieldName='" + this.fieldName + '\'' + ", interval=" + this.interval + '}';
        }
    }

    static class UnsupportedQueryException
    extends RuntimeException {
        private final Query unsupportedQuery;

        UnsupportedQueryException(Query unsupportedQuery) {
            super(LoggerMessageFormat.format("no query terms can be extracted from query [{}]", unsupportedQuery));
            this.unsupportedQuery = unsupportedQuery;
        }

        Query getUnsupportedQuery() {
            return this.unsupportedQuery;
        }
    }

    static class QueryExtraction {
        final Term term;
        final Range range;

        QueryExtraction(Term term) {
            this.term = term;
            this.range = null;
        }

        QueryExtraction(Range range) {
            this.term = null;
            this.range = range;
        }

        String field() {
            return this.term != null ? this.term.field() : null;
        }

        BytesRef bytes() {
            return this.term != null ? this.term.bytes() : null;
        }

        String text() {
            return this.term != null ? this.term.text() : null;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            QueryExtraction queryExtraction = (QueryExtraction)o;
            return Objects.equals(this.term, queryExtraction.term) && Objects.equals(this.range, queryExtraction.range);
        }

        public int hashCode() {
            return Objects.hash(this.term, this.range);
        }

        public String toString() {
            return "QueryExtraction{term=" + this.term + ",range=" + this.range + '}';
        }
    }

    static class Result {
        final Set<QueryExtraction> extractions;
        final boolean verified;
        final int minimumShouldMatch;
        final boolean matchAllDocs;

        private Result(boolean matchAllDocs, boolean verified, Set<QueryExtraction> extractions, int minimumShouldMatch) {
            if (minimumShouldMatch > extractions.size()) {
                throw new IllegalArgumentException("minimumShouldMatch can't be greater than the number of extractions: " + minimumShouldMatch + " > " + extractions.size());
            }
            this.matchAllDocs = matchAllDocs;
            this.extractions = extractions;
            this.verified = verified;
            this.minimumShouldMatch = minimumShouldMatch;
        }

        Result(boolean verified, Set<QueryExtraction> extractions, int minimumShouldMatch) {
            this(false, verified, extractions, minimumShouldMatch);
        }

        Result(boolean matchAllDocs, boolean verified) {
            this(matchAllDocs, verified, Collections.emptySet(), 0);
        }

        Result unverify() {
            if (this.verified) {
                return new Result(this.matchAllDocs, false, this.extractions, this.minimumShouldMatch);
            }
            return this;
        }

        boolean isMatchNoDocs() {
            return !this.matchAllDocs && this.extractions.isEmpty();
        }
    }
}

