/*
 * Licensed to the Apache Software Foundation (ASF) under one
 *   or more contributor license agreements.  See the NOTICE file
 *   distributed with this work for additional information
 *   regarding copyright ownership.  The ASF licenses this file
 *   to you under the Apache License, Version 2.0 (the
 *   "License"); you may not use this file except in compliance
 *   with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 */

package org.apache.phoenix.pherf.util;

import static org.apache.phoenix.jdbc.PhoenixDatabaseMetaData.TABLE_NAME;
import static org.apache.phoenix.jdbc.PhoenixDatabaseMetaData.TABLE_SCHEM;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;

import org.apache.phoenix.pherf.PherfConstants;
import org.apache.phoenix.pherf.configuration.Column;
import org.apache.phoenix.pherf.configuration.DataTypeMapping;
import org.apache.phoenix.pherf.configuration.Query;
import org.apache.phoenix.pherf.configuration.QuerySet;
import org.apache.phoenix.pherf.configuration.Scenario;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

// TODO This class needs to be cleanup up a bit. I just wanted to get an initial placeholder in.
public class PhoenixUtil {
	private static final Logger logger = LoggerFactory.getLogger(PhoenixUtil.class);
	private static String zookeeper;
	private static int rowCountOverride = 0;
    private boolean testEnabled;
    private static PhoenixUtil instance;

    private PhoenixUtil() {
        this(false);
    }

    private PhoenixUtil(final boolean testEnabled) {
        this.testEnabled = testEnabled;
    }

    public static PhoenixUtil create() {
        return create(false);
    }

    public static PhoenixUtil create(final boolean testEnabled) {
        instance = instance != null ? instance : new PhoenixUtil(testEnabled);
        return instance;
    }

    public Connection getConnection() throws Exception{
    	return getConnection(null);
    }
	
    public Connection getConnection(String tenantId) throws Exception {
        return getConnection(tenantId, testEnabled);
    }

    private Connection getConnection(String tenantId, boolean testEnabled) throws Exception {
        if (null == zookeeper) {
            throw new IllegalArgumentException(
                    "Zookeeper must be set before initializing connection!");
        }
        Properties props = new Properties();
        if (null != tenantId) {
            props.setProperty("TenantId", tenantId);
            logger.debug("\nSetting tenantId to " + tenantId);
        }
        String url = "jdbc:phoenix:" + zookeeper + (testEnabled ? ";test=true" : "");
        return DriverManager.getConnection(url, props);
    }

    public boolean executeStatement(String sql, Scenario scenario) throws Exception {
        Connection connection = null;
        boolean result = false;
        try {
            connection = getConnection(scenario.getTenantId());
            result = executeStatement(sql, connection);
        } finally {
            if (connection != null) {
                connection.close();
            }
        }
        return result;
    }

    /**
     * Execute statement
     * @param sql
     * @param connection
     * @return
     * @throws SQLException
     */
    public boolean executeStatementThrowException(String sql, Connection connection) throws SQLException {
    	boolean result = false;
    	PreparedStatement preparedStatement = null;
    	try {
            preparedStatement = connection.prepareStatement(sql);
            result = preparedStatement.execute();
            connection.commit();
        } finally {
            preparedStatement.close();
        }
        return result;
    }
    
    public boolean executeStatement(String sql, Connection connection) {
    	boolean result = false;
        PreparedStatement preparedStatement = null;
        try {
            preparedStatement = connection.prepareStatement(sql);
            result = preparedStatement.execute();
            connection.commit();
        } catch (SQLException e) {
            e.printStackTrace();
        } finally {
            try {
                preparedStatement.close();
            } catch (SQLException e) {
                e.printStackTrace();
            }
        }
        return result;
    }

    @SuppressWarnings("unused")
    public boolean executeStatement(PreparedStatement preparedStatement, Connection connection) {
    	boolean result = false;
        try {
            result = preparedStatement.execute();
            connection.commit();
        } catch (SQLException e) {
            e.printStackTrace();
        }
        return result;
    }

    /**
     * Delete existing tables with schema name set as {@link PherfConstants#PHERF_SCHEMA_NAME} with regex comparison 
     * 
     * @param regexMatch
     * @throws SQLException
     * @throws Exception
     */
    public void deleteTables(String regexMatch) throws Exception {
    	regexMatch = regexMatch.toUpperCase().replace("ALL", ".*");
    	Connection conn = getConnection();
    	try {
        	ResultSet resultSet = getTableMetaData(PherfConstants.PHERF_SCHEMA_NAME, null, conn);
			while (resultSet.next()) {
				String tableName = resultSet.getString(TABLE_SCHEM) == null ? resultSet
						.getString(TABLE_NAME) : resultSet
						.getString(TABLE_SCHEM)
						+ "."
						+ resultSet.getString(TABLE_NAME);
				if (tableName.matches(regexMatch)) {
					logger.info("\nDropping " + tableName);
					try {
						executeStatementThrowException("DROP TABLE "
								+ tableName + " CASCADE", conn);
					} catch (org.apache.phoenix.schema.TableNotFoundException tnf) {
						logger.error("Table might be already be deleted via cascade. Schema: "
								+ tnf.getSchemaName()
								+ " Table: "
								+ tnf.getTableName());
					}
				}
			}
    	} finally {
    		conn.close();
    	}
    }
    
    public ResultSet getTableMetaData(String schemaName, String tableName, Connection connection) throws SQLException {
    	DatabaseMetaData dbmd = connection.getMetaData();
    	ResultSet resultSet = dbmd.getTables(null, schemaName, tableName, null);
    	return resultSet;
    }
    
    public ResultSet getColumnsMetaData(String schemaName, String tableName, Connection connection) throws SQLException {
    	DatabaseMetaData dbmd = connection.getMetaData();
    	ResultSet resultSet = dbmd.getColumns(null, schemaName, tableName, null);
    	return resultSet;
    }
    
    public synchronized List<Column> getColumnsFromPhoenix(String schemaName, String tableName, Connection connection) throws SQLException {
    	List<Column> columnList = new ArrayList<Column>();
    	ResultSet resultSet = null;
    	try {
    		resultSet = getColumnsMetaData(schemaName, tableName, connection);
    		while (resultSet.next()) {
    			Column column = new Column();
    	        column.setName(resultSet.getString("COLUMN_NAME"));
    	        column.setType(DataTypeMapping.valueOf(resultSet.getString("TYPE_NAME")));
    	        column.setLength(resultSet.getInt("COLUMN_SIZE"));
    	        columnList.add(column);
   	        }
    	} finally {
    		if (null != resultSet) { 
    			resultSet.close();
    		}
    	}
    	
    	return Collections.unmodifiableList(columnList);
    }

    /**
     * Execute all querySet DDLs first based on tenantId if specified. This is executed
     * first since we don't want to run DDLs in parallel to executing queries.
     *
     * @param querySet
     * @throws Exception
     */
    public void executeQuerySetDdls(QuerySet querySet) throws Exception {
        for (Query query : querySet.getQuery()) {
            if (null != query.getDdl()) {
                Connection conn = null;
                try {
                    logger.info("\nExecuting DDL:" + query.getDdl() + " on tenantId:" + query
                            .getTenantId());
                    executeStatement(query.getDdl(),
                            conn = getConnection(query.getTenantId()));
                } finally {
                    if (null != conn) {
                        conn.close();
                    }
                }
            }
        }
    }

    public static String getZookeeper() {
		return zookeeper;
	}

	public static void setZookeeper(String zookeeper) {
		logger.info("Setting zookeeper: " + zookeeper);
		PhoenixUtil.zookeeper = zookeeper;
	}
	
	public static int getRowCountOverride() {
		return rowCountOverride;
	}
	
	public static void setRowCountOverride(int rowCountOverride) {
		PhoenixUtil.rowCountOverride = rowCountOverride;
	}
	
    /**
     * Update Phoenix table stats
     *
     * @param tableName
     * @throws Exception
     */
    public void updatePhoenixStats(String tableName, Scenario scenario) throws Exception {
        logger.info("Updating stats for " + tableName);
        executeStatement("UPDATE STATISTICS " + tableName, scenario);
    }
}
