/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.olingo.client.core.communication.request;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.olingo.client.api.CommonODataClient;
import org.apache.olingo.client.api.ODataBatchConstants;
import org.apache.olingo.client.api.communication.request.ODataBasicRequest;
import org.apache.olingo.client.api.communication.request.ODataStreamer;
import org.apache.olingo.client.api.communication.request.batch.CommonODataBatchRequest;
import org.apache.olingo.client.api.communication.response.ODataResponse;
import org.apache.olingo.client.api.http.HttpMethod;
import org.apache.olingo.commons.api.format.ODataFormat;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;

/**
 * Basic request abstract implementation.
 *
 * @param <T> OData response type corresponding to the request implementation.
 */
public abstract class AbstractODataBasicRequest<T extends ODataResponse>
    extends AbstractODataRequest implements ODataBasicRequest<T> {

  /**
   * Constructor.
   *
   * @param odataClient client instance getting this request
   * @param method request method.
   * @param uri OData request URI.
   */
  public AbstractODataBasicRequest(final CommonODataClient<?> odataClient, final HttpMethod method, final URI uri) {
    super(odataClient, method, uri);
  }

  @Override
  public void setFormat(final ODataFormat format) {
    if (format != null) {
      final String formatString = format.getContentType(odataClient.getServiceVersion()).toContentTypeString();
      setAccept(formatString);
      setContentType(formatString);
    }
  }

  @Override
  public final Future<T> asyncExecute() {
    return odataClient.getConfiguration().getExecutor().submit(new Callable<T>() {
      @Override
      public T call() throws Exception {
        return execute();
      }
    });
  }

  /**
   * Gets payload as an InputStream.
   *
   * @return InputStream for entire payload.
   */
  protected abstract InputStream getPayload();

  /**
   * Serializes the full request into the given batch request.
   *
   * @param req destination batch request.
   */
  public void batch(final CommonODataBatchRequest req) {
    batch(req, null);
  }

  /**
   * Serializes the full request into the given batch request.
   * <p>
   * This method have to be used to serialize a changeset item with the specified contentId.
   *
   * @param req destination batch request.
   * @param contentId contentId of the changeset item.
   */
  public void batch(final CommonODataBatchRequest req, final String contentId) {
    try {
      req.rawAppend(toByteArray());
      if (StringUtils.isNotBlank(contentId)) {
        req.rawAppend((ODataBatchConstants.CHANGESET_CONTENT_ID_NAME + ": " + contentId).getBytes());
        req.rawAppend(ODataStreamer.CRLF);
      }
      req.rawAppend(ODataStreamer.CRLF);

      final InputStream payload = getPayload();
      if (payload != null) {
        req.rawAppend(IOUtils.toByteArray(getPayload()));
      }
    } catch (IOException e) {
      throw new IllegalStateException(e);
    }
  }
}
