/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.controller;

import java.lang.management.GarbageCollectorMXBean;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryPoolMXBean;
import java.lang.management.MemoryUsage;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.components.Validator;
import org.apache.nifi.controller.ConfigurationContext;
import org.apache.nifi.processor.DataUnit;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.reporting.AbstractReportingTask;
import org.apache.nifi.reporting.Bulletin;
import org.apache.nifi.reporting.InitializationException;
import org.apache.nifi.reporting.ReportingContext;
import org.apache.nifi.reporting.ReportingInitializationContext;
import org.apache.nifi.reporting.Severity;
import org.apache.nifi.util.FormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Tags(value={"monitor", "memory", "heap", "jvm", "gc", "garbage collection", "warning"})
@CapabilityDescription(value="Checks the amount of Java Heap available in the JVM for a particular JVM Memory Pool. If the amount of space used exceeds some configurable threshold, will warn (via a log message and System-Level Bulletin) that the memory pool is exceeding this threshold.")
public class MonitorMemory
extends AbstractReportingTask {
    public static final PropertyDescriptor MEMORY_POOL_PROPERTY = new PropertyDescriptor.Builder().name("Memory Pool").description("The name of the JVM Memory Pool to monitor").required(true).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).defaultValue(null).build();
    public static final PropertyDescriptor THRESHOLD_PROPERTY = new PropertyDescriptor.Builder().name("Usage Threshold").description("Indicates the threshold at which warnings should be generated").required(true).addValidator((Validator)new ThresholdValidator()).defaultValue("65%").build();
    public static final PropertyDescriptor REPORTING_INTERVAL = new PropertyDescriptor.Builder().name("Reporting Interval").description("Indicates how often this reporting task should report bulletins while the memory utilization exceeds the configured threshold").required(false).addValidator(StandardValidators.TIME_PERIOD_VALIDATOR).defaultValue(null).build();
    public static final Pattern PERCENTAGE_PATTERN = Pattern.compile("\\d{1,2}%");
    public static final Pattern DATA_SIZE_PATTERN = DataUnit.DATA_SIZE_PATTERN;
    public static final Pattern TIME_PERIOD_PATTERN = FormatUtils.TIME_DURATION_PATTERN;
    private static final Logger logger = LoggerFactory.getLogger(MonitorMemory.class);
    private volatile MemoryPoolMXBean monitoredBean;
    private volatile String threshold = "65%";
    private volatile long lastReportTime = 0L;
    private volatile long reportingIntervalMillis;
    private volatile boolean lastValueWasExceeded = false;
    private final List<GarbageCollectorMXBean> garbageCollectorBeans = new ArrayList<GarbageCollectorMXBean>();
    private long schedulingPeriodMillis;

    protected void init(ReportingInitializationContext config) throws InitializationException {
        this.schedulingPeriodMillis = config.getSchedulingPeriod(TimeUnit.MILLISECONDS);
    }

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        ArrayList<PropertyDescriptor> descriptors = new ArrayList<PropertyDescriptor>(3);
        descriptors.add(MEMORY_POOL_PROPERTY);
        descriptors.add(THRESHOLD_PROPERTY);
        descriptors.add(REPORTING_INTERVAL);
        return descriptors;
    }

    @OnScheduled
    public void onConfigured(ConfigurationContext config) throws InitializationException {
        String thresholdValue;
        String desiredMemoryPoolName = config.getProperty(MEMORY_POOL_PROPERTY).getValue();
        this.threshold = thresholdValue = config.getProperty(THRESHOLD_PROPERTY).getValue().trim();
        Long reportingIntervalValue = config.getProperty(REPORTING_INTERVAL).asTimePeriod(TimeUnit.MILLISECONDS);
        this.reportingIntervalMillis = reportingIntervalValue == null ? this.schedulingPeriodMillis : reportingIntervalValue;
        List<MemoryPoolMXBean> memoryPoolBeans = ManagementFactory.getMemoryPoolMXBeans();
        for (MemoryPoolMXBean memoryPoolMXBean : memoryPoolBeans) {
            String memoryPoolName = memoryPoolMXBean.getName();
            if (!desiredMemoryPoolName.equals(memoryPoolName)) continue;
            this.monitoredBean = memoryPoolMXBean;
            if (DATA_SIZE_PATTERN.matcher(thresholdValue).matches()) {
                long bytes = DataUnit.parseDataSize((String)thresholdValue, (DataUnit)DataUnit.B).longValue();
                if (!memoryPoolMXBean.isCollectionUsageThresholdSupported()) continue;
                memoryPoolMXBean.setCollectionUsageThreshold(bytes);
                continue;
            }
            String percentage = thresholdValue.substring(0, thresholdValue.length() - 1);
            double pct = Double.parseDouble(percentage) / 100.0;
            long calculatedThreshold = (long)((double)memoryPoolMXBean.getUsage().getMax() * pct);
            if (!memoryPoolMXBean.isCollectionUsageThresholdSupported()) continue;
            memoryPoolMXBean.setCollectionUsageThreshold(calculatedThreshold);
        }
        for (GarbageCollectorMXBean garbageCollectorMXBean : ManagementFactory.getGarbageCollectorMXBeans()) {
            for (String memoryPoolName : garbageCollectorMXBean.getMemoryPoolNames()) {
                if (!desiredMemoryPoolName.equals(memoryPoolName)) continue;
                this.garbageCollectorBeans.add(garbageCollectorMXBean);
            }
        }
        if (this.monitoredBean == null) {
            throw new InitializationException("Found no JVM Memory Pool with name " + desiredMemoryPoolName + "; will not monitor Memory Pool");
        }
    }

    private long calculateThresholdBytes(long maxBytes) {
        if (DATA_SIZE_PATTERN.matcher(this.threshold).matches()) {
            return DataUnit.parseDataSize((String)this.threshold, (DataUnit)DataUnit.B).longValue();
        }
        String percentage = this.threshold.substring(0, this.threshold.length() - 1);
        double pct = Double.parseDouble(percentage) / 100.0;
        return (long)((double)maxBytes * pct);
    }

    public void onTrigger(ReportingContext context) {
        MemoryPoolMXBean bean = this.monitoredBean;
        if (bean == null) {
            return;
        }
        MemoryUsage usage = bean.getCollectionUsage();
        if (usage == null) {
            logger.warn("{} could not determine memory usage for pool with name {}", (Object)this, (Object)context.getProperty(MEMORY_POOL_PROPERTY));
            return;
        }
        boolean exceeded = usage.getUsed() > this.calculateThresholdBytes(usage.getMax());
        double percentageUsed = (double)usage.getUsed() / (double)usage.getMax() * 100.0;
        if (exceeded) {
            if (System.currentTimeMillis() < this.reportingIntervalMillis + this.lastReportTime && this.lastReportTime > 0L) {
                return;
            }
            this.lastReportTime = System.currentTimeMillis();
            this.lastValueWasExceeded = true;
            String message = String.format("Memory Pool '%1$s' has exceeded the configured Threshold of %2$s, having used %3$s / %4$s (%5$.2f%%)", bean.getName(), this.threshold, FormatUtils.formatDataSize((double)usage.getUsed()), FormatUtils.formatDataSize((double)usage.getMax()), percentageUsed);
            logger.warn("{}", (Object)message);
            Bulletin bulletin = context.createBulletin("Memory Management", Severity.WARNING, message);
            context.getBulletinRepository().addBulletin(bulletin);
        } else if (this.lastValueWasExceeded) {
            this.lastValueWasExceeded = false;
            this.lastReportTime = System.currentTimeMillis();
            String message = String.format("Memory Pool '%1$s' is no longer exceeding the configured Threshold of %2$s; currently using %3$s / %4$s (%5$.2f%%)", bean.getName(), this.threshold, FormatUtils.formatDataSize((double)usage.getUsed()), FormatUtils.formatDataSize((double)usage.getMax()), percentageUsed);
            logger.info("{}", (Object)message);
            Bulletin bulletin = context.createBulletin("Memory Management", Severity.INFO, message);
            context.getBulletinRepository().addBulletin(bulletin);
        }
    }

    private static class ThresholdValidator
    implements Validator {
        private ThresholdValidator() {
        }

        public ValidationResult validate(String subject, String input, ValidationContext context) {
            if (!PERCENTAGE_PATTERN.matcher(input).matches() && !DATA_SIZE_PATTERN.matcher(input).matches()) {
                return new ValidationResult.Builder().input(input).subject(subject).valid(false).explanation("Valid value is a number in the range of 0-99 followed by a percent sign (e.g. 65%) or a Data Size (e.g. 100 MB)").build();
            }
            return new ValidationResult.Builder().input(input).subject(subject).valid(true).build();
        }
    }
}

