/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.lookup.maxmind;

import com.maxmind.db.InvalidDatabaseException;
import com.maxmind.geoip2.model.AnonymousIpResponse;
import com.maxmind.geoip2.model.CityResponse;
import com.maxmind.geoip2.model.ConnectionTypeResponse;
import com.maxmind.geoip2.model.DomainResponse;
import com.maxmind.geoip2.model.IspResponse;
import com.maxmind.geoip2.record.Country;
import com.maxmind.geoip2.record.Location;
import com.maxmind.geoip2.record.Subdivision;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnEnabled;
import org.apache.nifi.annotation.lifecycle.OnStopped;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.resource.ResourceCardinality;
import org.apache.nifi.components.resource.ResourceType;
import org.apache.nifi.controller.AbstractControllerService;
import org.apache.nifi.controller.ConfigurationContext;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.lookup.LookupFailureException;
import org.apache.nifi.lookup.RecordLookupService;
import org.apache.nifi.lookup.maxmind.AnonymousIpSchema;
import org.apache.nifi.lookup.maxmind.CitySchema;
import org.apache.nifi.lookup.maxmind.ContainerSchema;
import org.apache.nifi.lookup.maxmind.DatabaseReader;
import org.apache.nifi.lookup.maxmind.IspSchema;
import org.apache.nifi.serialization.record.MapRecord;
import org.apache.nifi.serialization.record.Record;
import org.apache.nifi.util.StopWatch;

@Tags(value={"lookup", "enrich", "ip", "geo", "ipgeo", "maxmind", "isp", "domain", "cellular", "anonymous", "tor"})
@CapabilityDescription(value="A lookup service that provides several types of enrichment information for IP addresses. The service is configured by providing a MaxMind Database file and specifying which types of enrichment should be provided for an IP Address or Hostname. Each type of enrichment is a separate lookup, so configuring the service to provide all of the available enrichment data may be slower than returning only a portion of the available enrichments. In order to use this service, a lookup must be performed using key of 'ip' and a value that is a valid IP address or hostname. View the Usage of this component and choose to view Additional Details for more information, such as the Schema that pertains to the information that is returned.")
public class IPLookupService
extends AbstractControllerService
implements RecordLookupService {
    private volatile String databaseFile = null;
    private static final String IP_KEY = "ip";
    private static final Set<String> REQUIRED_KEYS = Stream.of("ip").collect(Collectors.toSet());
    private volatile DatabaseReader databaseReader = null;
    private volatile String databaseChecksum = null;
    private volatile long databaseLastRefreshAttempt = -1L;
    private final Lock dbWriteLock = new ReentrantLock();
    static final long REFRESH_THRESHOLD_MS = 300000L;
    static final PropertyDescriptor GEO_DATABASE_FILE = new PropertyDescriptor.Builder().name("database-file").displayName("MaxMind Database File").description("Path to Maxmind IP Enrichment Database File").required(true).identifiesExternalResource(ResourceCardinality.SINGLE, ResourceType.FILE, new ResourceType[0]).expressionLanguageSupported(ExpressionLanguageScope.VARIABLE_REGISTRY).build();
    static final PropertyDescriptor LOOKUP_CITY = new PropertyDescriptor.Builder().name("lookup-city").displayName("Lookup Geo Enrichment").description("Specifies whether or not information about the geographic information, such as cities, corresponding to the IP address should be returned").allowableValues(new String[]{"true", "false"}).defaultValue("true").expressionLanguageSupported(ExpressionLanguageScope.NONE).required(true).build();
    static final PropertyDescriptor LOOKUP_ISP = new PropertyDescriptor.Builder().name("lookup-isp").displayName("Lookup ISP").description("Specifies whether or not information about the Information Service Provider corresponding to the IP address should be returned").expressionLanguageSupported(ExpressionLanguageScope.NONE).allowableValues(new String[]{"true", "false"}).defaultValue("false").required(true).build();
    static final PropertyDescriptor LOOKUP_DOMAIN = new PropertyDescriptor.Builder().name("lookup-domain").displayName("Lookup Domain Name").description("Specifies whether or not information about the Domain Name corresponding to the IP address should be returned. If true, the lookup will contain second-level domain information, such as foo.com but will not contain bar.foo.com").expressionLanguageSupported(ExpressionLanguageScope.NONE).allowableValues(new String[]{"true", "false"}).defaultValue("false").required(true).build();
    static final PropertyDescriptor LOOKUP_CONNECTION_TYPE = new PropertyDescriptor.Builder().name("lookup-connection-type").displayName("Lookup Connection Type").description("Specifies whether or not information about the Connection Type corresponding to the IP address should be returned. If true, the lookup will contain a 'connectionType' field that (if populated) will contain a value of 'Dialup', 'Cable/DSL', 'Corporate', or 'Cellular'").expressionLanguageSupported(ExpressionLanguageScope.NONE).allowableValues(new String[]{"true", "false"}).defaultValue("false").required(true).build();
    static final PropertyDescriptor LOOKUP_ANONYMOUS_IP_INFO = new PropertyDescriptor.Builder().name("lookup-anonymous-ip").displayName("Lookup Anonymous IP Information").description("Specifies whether or not information about whether or not the IP address belongs to an anonymous network should be returned.").expressionLanguageSupported(ExpressionLanguageScope.NONE).allowableValues(new String[]{"true", "false"}).defaultValue("false").required(true).build();

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        ArrayList<PropertyDescriptor> properties = new ArrayList<PropertyDescriptor>();
        properties.add(GEO_DATABASE_FILE);
        properties.add(LOOKUP_CITY);
        properties.add(LOOKUP_ISP);
        properties.add(LOOKUP_DOMAIN);
        properties.add(LOOKUP_CONNECTION_TYPE);
        properties.add(LOOKUP_ANONYMOUS_IP_INFO);
        return properties;
    }

    @OnEnabled
    public void onEnabled(ConfigurationContext context) throws IOException {
        this.databaseFile = context.getProperty(GEO_DATABASE_FILE).evaluateAttributeExpressions().getValue();
        File dbFile = new File(this.databaseFile);
        String dbFileChecksum = this.getChecksum(dbFile);
        this.loadDatabase(dbFile, dbFileChecksum);
        this.databaseLastRefreshAttempt = System.currentTimeMillis();
    }

    private String getChecksum(File file) throws IOException {
        String fileChecksum;
        try (FileInputStream in = new FileInputStream(file);){
            fileChecksum = DigestUtils.md5Hex((InputStream)in);
        }
        return fileChecksum;
    }

    @OnStopped
    public void closeReader() throws IOException {
        DatabaseReader reader = this.databaseReader;
        if (reader != null) {
            reader.close();
        }
        this.databaseFile = null;
        this.databaseReader = null;
        this.databaseChecksum = null;
        this.databaseLastRefreshAttempt = -1L;
    }

    public Set<String> getRequiredKeys() {
        return REQUIRED_KEYS;
    }

    public Optional<Record> lookup(Map<String, Object> coordinates) throws LookupFailureException {
        if (coordinates == null) {
            return Optional.empty();
        }
        if (this.shouldAttemptDatabaseRefresh()) {
            try {
                this.refreshDatabase();
            }
            catch (IOException e) {
                throw new LookupFailureException("Failed to refresh database file: " + e.getMessage(), (Throwable)e);
            }
        }
        try {
            DatabaseReader databaseReader = this.databaseReader;
            return this.doLookup(databaseReader, coordinates);
        }
        catch (InvalidDatabaseException idbe) {
            if (this.dbWriteLock.tryLock()) {
                try {
                    this.getLogger().debug("Attempting to reload database after InvalidDatabaseException");
                    try {
                        File dbFile = new File(this.databaseFile);
                        String dbFileChecksum = this.getChecksum(dbFile);
                        this.loadDatabase(dbFile, dbFileChecksum);
                        this.databaseLastRefreshAttempt = System.currentTimeMillis();
                    }
                    catch (IOException ioe) {
                        throw new LookupFailureException("Error reloading database due to: " + ioe.getMessage(), (Throwable)ioe);
                    }
                    this.getLogger().debug("Attempting to retry lookup after InvalidDatabaseException");
                    try {
                        DatabaseReader databaseReader = this.databaseReader;
                        Optional<Record> optional = this.doLookup(databaseReader, coordinates);
                        return optional;
                    }
                    catch (Exception e) {
                        throw new LookupFailureException("Error performing look up: " + e.getMessage(), (Throwable)e);
                    }
                }
                finally {
                    this.dbWriteLock.unlock();
                }
            }
            throw new LookupFailureException("Failed to lookup a value for " + coordinates + " due to " + idbe.getMessage(), (Throwable)idbe);
        }
    }

    private Optional<Record> doLookup(DatabaseReader databaseReader, Map<String, Object> coordinates) throws LookupFailureException, InvalidDatabaseException {
        Record anonymousIpRecord;
        String connectionType;
        String domainName;
        Record ispRecord;
        Record geoRecord;
        InetAddress inetAddress;
        if (coordinates.get(IP_KEY) == null) {
            return Optional.empty();
        }
        String ipAddress = coordinates.get(IP_KEY).toString();
        try {
            inetAddress = InetAddress.getByName(ipAddress);
        }
        catch (IOException ioe) {
            this.getLogger().warn("Could not resolve the IP for value '{}'. This is usually caused by issue resolving the appropriate DNS record or providing the service with an invalid IP address", new Object[]{coordinates}, (Throwable)ioe);
            return Optional.empty();
        }
        if (this.getProperty(LOOKUP_CITY).asBoolean().booleanValue()) {
            CityResponse cityResponse;
            try {
                cityResponse = databaseReader.city(inetAddress);
            }
            catch (InvalidDatabaseException idbe) {
                throw idbe;
            }
            catch (Exception e) {
                throw new LookupFailureException("Failed to lookup City information for IP Address " + inetAddress, (Throwable)e);
            }
            geoRecord = this.createRecord(cityResponse);
        } else {
            geoRecord = null;
        }
        if (this.getProperty(LOOKUP_ISP).asBoolean().booleanValue()) {
            IspResponse ispResponse;
            try {
                ispResponse = databaseReader.isp(inetAddress);
            }
            catch (InvalidDatabaseException idbe) {
                throw idbe;
            }
            catch (Exception e) {
                throw new LookupFailureException("Failed to lookup ISP information for IP Address " + inetAddress, (Throwable)e);
            }
            ispRecord = this.createRecord(ispResponse);
        } else {
            ispRecord = null;
        }
        if (this.getProperty(LOOKUP_DOMAIN).asBoolean().booleanValue()) {
            DomainResponse domainResponse;
            try {
                domainResponse = databaseReader.domain(inetAddress);
            }
            catch (InvalidDatabaseException idbe) {
                throw idbe;
            }
            catch (Exception e) {
                throw new LookupFailureException("Failed to lookup Domain information for IP Address " + inetAddress, (Throwable)e);
            }
            domainName = domainResponse == null ? null : domainResponse.getDomain();
        } else {
            domainName = null;
        }
        if (this.getProperty(LOOKUP_CONNECTION_TYPE).asBoolean().booleanValue()) {
            ConnectionTypeResponse.ConnectionType type;
            ConnectionTypeResponse connectionTypeResponse;
            try {
                connectionTypeResponse = databaseReader.connectionType(inetAddress);
            }
            catch (InvalidDatabaseException idbe) {
                throw idbe;
            }
            catch (Exception e) {
                throw new LookupFailureException("Failed to lookup Domain information for IP Address " + inetAddress, (Throwable)e);
            }
            connectionType = connectionTypeResponse == null ? null : ((type = connectionTypeResponse.getConnectionType()) == null ? null : type.name());
        } else {
            connectionType = null;
        }
        if (this.getProperty(LOOKUP_ANONYMOUS_IP_INFO).asBoolean().booleanValue()) {
            AnonymousIpResponse anonymousIpResponse;
            try {
                anonymousIpResponse = databaseReader.anonymousIp(inetAddress);
            }
            catch (InvalidDatabaseException idbe) {
                throw idbe;
            }
            catch (Exception e) {
                throw new LookupFailureException("Failed to lookup Anonymous IP Information for IP Address " + inetAddress, (Throwable)e);
            }
            anonymousIpRecord = this.createRecord(anonymousIpResponse);
        } else {
            anonymousIpRecord = null;
        }
        if (geoRecord == null && ispRecord == null && domainName == null && connectionType == null && anonymousIpRecord == null) {
            return Optional.empty();
        }
        return Optional.ofNullable(this.createContainerRecord(geoRecord, ispRecord, domainName, connectionType, anonymousIpRecord));
    }

    private boolean shouldAttemptDatabaseRefresh() {
        return System.currentTimeMillis() - this.databaseLastRefreshAttempt >= 300000L;
    }

    private void refreshDatabase() throws IOException {
        if (this.dbWriteLock.tryLock()) {
            try {
                if (this.shouldAttemptDatabaseRefresh()) {
                    File dbFile = new File(this.databaseFile);
                    String dbFileChecksum = this.getChecksum(dbFile);
                    if (!dbFileChecksum.equals(this.databaseChecksum)) {
                        this.loadDatabase(dbFile, dbFileChecksum);
                    } else {
                        this.getLogger().debug("Checksum hasn't changed, database will not be reloaded");
                    }
                    this.databaseLastRefreshAttempt = System.currentTimeMillis();
                }
                this.getLogger().debug("Acquired write lock, but no longer need to reload the database");
            }
            finally {
                this.dbWriteLock.unlock();
            }
        } else {
            this.getLogger().debug("Unable to acquire write lock, skipping reload of database");
        }
    }

    private void loadDatabase(File dbFile, String dbFileChecksum) throws IOException {
        StopWatch stopWatch = new StopWatch(true);
        DatabaseReader reader = new DatabaseReader.Builder(dbFile).build();
        stopWatch.stop();
        this.getLogger().info("Completed loading of Maxmind Database.  Elapsed time was {} milliseconds.", new Object[]{stopWatch.getDuration(TimeUnit.MILLISECONDS)});
        this.databaseReader = reader;
        this.databaseChecksum = dbFileChecksum;
    }

    private Record createRecord(CityResponse city) {
        if (city == null) {
            return null;
        }
        HashMap<String, Object> values = new HashMap<String, Object>();
        values.put(CitySchema.CITY.getFieldName(), city.getCity().getName());
        Location location = city.getLocation();
        values.put(CitySchema.ACCURACY.getFieldName(), location.getAccuracyRadius());
        values.put(CitySchema.METRO_CODE.getFieldName(), location.getMetroCode());
        values.put(CitySchema.TIMEZONE.getFieldName(), location.getTimeZone());
        values.put(CitySchema.LATITUDE.getFieldName(), location.getLatitude());
        values.put(CitySchema.LONGITUDE.getFieldName(), location.getLongitude());
        values.put(CitySchema.CONTINENT.getFieldName(), city.getContinent().getName());
        values.put(CitySchema.POSTALCODE.getFieldName(), city.getPostal().getCode());
        values.put(CitySchema.COUNTRY.getFieldName(), this.createRecord(city.getCountry()));
        Object[] subdivisions = new Object[city.getSubdivisions().size()];
        int i = 0;
        for (Subdivision subdivision : city.getSubdivisions()) {
            subdivisions[i++] = this.createRecord(subdivision);
        }
        values.put(CitySchema.SUBDIVISIONS.getFieldName(), subdivisions);
        return new MapRecord(CitySchema.GEO_SCHEMA, values);
    }

    private Record createRecord(Subdivision subdivision) {
        if (subdivision == null) {
            return null;
        }
        HashMap<String, String> values = new HashMap<String, String>(2);
        values.put(CitySchema.SUBDIVISION_NAME.getFieldName(), subdivision.getName());
        values.put(CitySchema.SUBDIVISION_ISO.getFieldName(), subdivision.getIsoCode());
        return new MapRecord(CitySchema.SUBDIVISION_SCHEMA, values);
    }

    private Record createRecord(Country country) {
        if (country == null) {
            return null;
        }
        HashMap<String, String> values = new HashMap<String, String>(2);
        values.put(CitySchema.COUNTRY_NAME.getFieldName(), country.getName());
        values.put(CitySchema.COUNTRY_ISO.getFieldName(), country.getIsoCode());
        return new MapRecord(CitySchema.COUNTRY_SCHEMA, values);
    }

    private Record createRecord(IspResponse isp) {
        if (isp == null) {
            return null;
        }
        HashMap<String, Object> values = new HashMap<String, Object>(4);
        values.put(IspSchema.ASN.getFieldName(), isp.getAutonomousSystemNumber());
        values.put(IspSchema.ASN_ORG.getFieldName(), isp.getAutonomousSystemOrganization());
        values.put(IspSchema.NAME.getFieldName(), isp.getIsp());
        values.put(IspSchema.ORG.getFieldName(), isp.getOrganization());
        return new MapRecord(IspSchema.ISP_SCHEMA, values);
    }

    private Record createRecord(AnonymousIpResponse anonymousIp) {
        if (anonymousIp == null) {
            return null;
        }
        HashMap<String, Boolean> values = new HashMap<String, Boolean>(5);
        values.put(AnonymousIpSchema.ANONYMOUS.getFieldName(), anonymousIp.isAnonymous());
        values.put(AnonymousIpSchema.ANONYMOUS_VPN.getFieldName(), anonymousIp.isAnonymousVpn());
        values.put(AnonymousIpSchema.HOSTING_PROVIDER.getFieldName(), anonymousIp.isHostingProvider());
        values.put(AnonymousIpSchema.PUBLIC_PROXY.getFieldName(), anonymousIp.isPublicProxy());
        values.put(AnonymousIpSchema.TOR_EXIT_NODE.getFieldName(), anonymousIp.isTorExitNode());
        return new MapRecord(AnonymousIpSchema.ANONYMOUS_IP_SCHEMA, values);
    }

    private Record createContainerRecord(Record geoRecord, Record ispRecord, String domainName, String connectionType, Record anonymousIpRecord) {
        HashMap<String, Object> values = new HashMap<String, Object>(4);
        values.put("geo", geoRecord);
        values.put("isp", ispRecord);
        values.put("domainName", domainName);
        values.put("connectionType", connectionType);
        values.put("anonymousIp", anonymousIpRecord);
        MapRecord containerRecord = new MapRecord(ContainerSchema.CONTAINER_SCHEMA, values);
        return containerRecord;
    }
}

