/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.nar;

import java.io.File;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.Arrays;
import java.util.Collection;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import org.apache.nifi.nar.NarCloseable;
import org.apache.nifi.nar.NarProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class NarProviderTask
implements Runnable {
    private static final Logger LOGGER = LoggerFactory.getLogger(NarProviderTask.class);
    private static final String NAR_EXTENSION = "nar";
    private final String id = UUID.randomUUID().toString();
    private final NarProvider narProvider;
    private final ClassLoader narProviderClassLoader;
    private final long pollTimeInMs;
    private final File extensionDirectory;
    private volatile boolean stopped = false;

    NarProviderTask(NarProvider narProvider, ClassLoader narProviderClassLoader, File extensionDirectory, long pollTimeInMs) {
        this.narProvider = narProvider;
        this.narProviderClassLoader = narProviderClassLoader;
        this.pollTimeInMs = pollTimeInMs;
        this.extensionDirectory = extensionDirectory;
    }

    @Override
    public void run() {
        LOGGER.info("Nar provider task is started");
        while (!this.stopped) {
            try {
                Collection availableNars;
                LOGGER.debug("Task starts fetching NARs from provider");
                Set<String> loadedNars = this.getLoadedNars();
                try (NarCloseable narCloseable = NarCloseable.withComponentNarLoader((ClassLoader)this.narProviderClassLoader);){
                    availableNars = this.narProvider.listNars();
                }
                for (String availableNar : availableNars) {
                    InputStream inputStream;
                    if (loadedNars.contains(availableNar)) continue;
                    long startedAt = System.currentTimeMillis();
                    try (NarCloseable narCloseable = NarCloseable.withComponentNarLoader((ClassLoader)this.narProviderClassLoader);){
                        inputStream = this.narProvider.fetchNarContents(availableNar);
                    }
                    File tempFile = new File(this.extensionDirectory, ".tmp_" + this.id + ".nar");
                    File targetFile = new File(this.extensionDirectory, availableNar);
                    Files.copy(inputStream, tempFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
                    tempFile.renameTo(targetFile);
                    LOGGER.info("Downloaded NAR {} in {} ms", (Object)availableNar, (Object)(System.currentTimeMillis() - startedAt));
                }
                LOGGER.debug("Task finished fetching NARs from provider");
            }
            catch (Throwable e) {
                LOGGER.error("Error during reaching the external source", e);
            }
            try {
                Thread.sleep(this.pollTimeInMs);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LOGGER.warn("NAR autoloader external source task is interrupted");
                this.stopped = true;
            }
        }
    }

    private Set<String> getLoadedNars() {
        return Arrays.stream(this.extensionDirectory.listFiles(file -> file.isFile() && file.getName().toLowerCase().endsWith(".nar"))).map(file -> file.getName()).collect(Collectors.toSet());
    }

    void stop() {
        LOGGER.info("Nar provider task is stopped");
        this.stopped = true;
    }
}

