/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processor.util.file.transfer;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.flowfile.attributes.CoreAttributes;
import org.apache.nifi.logging.ComponentLog;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.FlowFileAccessException;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.file.transfer.FileInfo;
import org.apache.nifi.processor.util.file.transfer.FileTransfer;
import org.apache.nifi.util.StopWatch;
import org.apache.nifi.util.StringUtils;

public abstract class PutFileTransfer<T extends FileTransfer>
extends AbstractProcessor {
    public static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("FlowFiles that are successfully sent will be routed to success").build();
    public static final Relationship REL_FAILURE = new Relationship.Builder().name("failure").description("FlowFiles that failed to send to the remote system; failure is usually looped back to this processor").build();
    public static final Relationship REL_REJECT = new Relationship.Builder().name("reject").description("FlowFiles that were rejected by the destination system").build();
    private static final Set<Relationship> RELATIONSHIPS = Set.of(REL_SUCCESS, REL_FAILURE, REL_REJECT);

    public Set<Relationship> getRelationships() {
        return RELATIONSHIPS;
    }

    protected abstract T getFileTransfer(ProcessContext var1);

    protected void beforePut(FlowFile flowFile, ProcessContext context, T transfer) throws IOException {
    }

    protected void afterPut(FlowFile flowFile, ProcessContext context, T transfer) throws IOException {
    }

    public void onTrigger(ProcessContext context, ProcessSession session) {
        FlowFile flowFile = session.get();
        if (flowFile == null) {
            return;
        }
        ComponentLog logger = this.getLogger();
        String hostname = context.getProperty(FileTransfer.HOSTNAME).evaluateAttributeExpressions(flowFile).getValue();
        int maxNumberOfFiles = context.getProperty(FileTransfer.BATCH_SIZE).asInteger();
        int fileCount = 0;
        try (Object transfer = this.getFileTransfer(context);){
            do {
                hostname = context.getProperty(FileTransfer.HOSTNAME).evaluateAttributeExpressions(flowFile).getValue();
                String rootPath = context.getProperty(FileTransfer.REMOTE_PATH).evaluateAttributeExpressions(flowFile).getValue();
                String workingDirPath = StringUtils.isBlank((String)rootPath) ? transfer.getHomeDirectory(flowFile) : transfer.getAbsolutePath(flowFile, rootPath);
                boolean rejectZeroByteFiles = context.getProperty(FileTransfer.REJECT_ZERO_BYTE).asBoolean();
                ConflictResult conflictResult = this.identifyAndResolveConflictFile(context.getProperty(FileTransfer.CONFLICT_RESOLUTION).getValue(), transfer, workingDirPath, flowFile, rejectZeroByteFiles, logger);
                if (conflictResult.isTransfer()) {
                    StopWatch stopWatch = new StopWatch();
                    stopWatch.start();
                    this.beforePut(flowFile, context, transfer);
                    FlowFile flowFileToTransfer = flowFile;
                    AtomicReference<Object> fullPathRef = new AtomicReference<Object>(null);
                    session.read(flowFile, in -> {
                        try (BufferedInputStream bufferedIn = new BufferedInputStream(in);){
                            if (workingDirPath != null && context.getProperty(FileTransfer.CREATE_DIRECTORY).asBoolean().booleanValue()) {
                                transfer.ensureDirectoryExists(flowFileToTransfer, new File(workingDirPath));
                            }
                            fullPathRef.set(transfer.put(flowFileToTransfer, workingDirPath, conflictResult.getFileName(), bufferedIn));
                        }
                    });
                    this.afterPut(flowFile, context, transfer);
                    stopWatch.stop();
                    String dataRate = stopWatch.calculateDataRate(flowFile.getSize());
                    long millis = stopWatch.getDuration(TimeUnit.MILLISECONDS);
                    logger.info("Successfully transferred {} to {} on remote host {} in {} milliseconds at a rate of {}", new Object[]{flowFile, fullPathRef.get(), hostname, millis, dataRate});
                    Object fullPathWithSlash = fullPathRef.get();
                    if (!((String)fullPathWithSlash).startsWith("/")) {
                        fullPathWithSlash = "/" + (String)fullPathWithSlash;
                    }
                    String destinationUri = transfer.getProtocolName() + "://" + hostname + (String)fullPathWithSlash;
                    session.getProvenanceReporter().send(flowFile, destinationUri, millis);
                }
                if (conflictResult.isPenalize()) {
                    flowFile = session.penalize(flowFile);
                }
                session.transfer(flowFile, conflictResult.getRelationship());
                session.commitAsync();
            } while (this.isScheduled() && this.getRelationships().size() == context.getAvailableRelationships().size() && ++fileCount < maxNumberOfFiles && (flowFile = session.get()) != null);
        }
        catch (IOException e) {
            context.yield();
            logger.error("Unable to transfer {} to remote host {} ", new Object[]{flowFile, hostname, e});
            flowFile = session.penalize(flowFile);
            session.transfer(flowFile, REL_FAILURE);
        }
        catch (FlowFileAccessException e) {
            context.yield();
            logger.error("Unable to transfer {} to remote host {}", new Object[]{flowFile, hostname, e.getCause()});
            flowFile = session.penalize(flowFile);
            session.transfer(flowFile, REL_FAILURE);
        }
        catch (ProcessException e) {
            context.yield();
            logger.error("Routing to failure since unable to transfer {} to remote host {}", new Object[]{flowFile, hostname, e.getCause()});
            flowFile = session.penalize(flowFile);
            session.transfer(flowFile, REL_FAILURE);
        }
    }

    private ConflictResult identifyAndResolveConflictFile(String conflictResolutionType, T transfer, String path, FlowFile flowFile, boolean rejectZeroByteFiles, ComponentLog logger) throws IOException {
        long sizeInBytes;
        Relationship destinationRelationship = REL_SUCCESS;
        Object fileName = flowFile.getAttribute(CoreAttributes.FILENAME.key());
        boolean transferFile = true;
        boolean penalizeFile = false;
        if (rejectZeroByteFiles && (sizeInBytes = flowFile.getSize()) == 0L) {
            logger.warn("Rejecting {} because it is zero bytes", new Object[]{flowFile});
            return new ConflictResult(REL_REJECT, false, (String)fileName, true);
        }
        if (conflictResolutionType.equalsIgnoreCase("NONE")) {
            return new ConflictResult(destinationRelationship, transferFile, (String)fileName, penalizeFile);
        }
        FileInfo remoteFileInfo = transfer.getRemoteFileInfo(flowFile, path, (String)fileName);
        if (remoteFileInfo == null) {
            return new ConflictResult(destinationRelationship, transferFile, (String)fileName, penalizeFile);
        }
        if (remoteFileInfo.isDirectory()) {
            logger.warn("Resolving conflict by rejecting {} due to conflicting filename with a directory or file already on remote server", new Object[]{flowFile});
            return new ConflictResult(REL_REJECT, false, (String)fileName, false);
        }
        logger.info("Discovered a filename conflict on the remote server for {} so handling using configured Conflict Resolution of {}", new Object[]{flowFile, conflictResolutionType});
        switch (conflictResolutionType.toUpperCase()) {
            case "REJECT": {
                destinationRelationship = REL_REJECT;
                transferFile = false;
                penalizeFile = false;
                logger.warn("Resolving conflict by rejecting {} due to conflicting filename with a directory or file already on remote server", new Object[]{flowFile});
                break;
            }
            case "REPLACE": {
                destinationRelationship = REL_SUCCESS;
                transferFile = true;
                penalizeFile = false;
                logger.info("Resolving filename conflict for {} with remote server by deleting remote file and replacing with flow file", new Object[]{flowFile});
                break;
            }
            case "RENAME": {
                boolean uniqueNameGenerated = false;
                for (int i = 1; i < 100 && !uniqueNameGenerated; ++i) {
                    String possibleFileName = i + "." + (String)fileName;
                    FileInfo renamedFileInfo = transfer.getRemoteFileInfo(flowFile, path, possibleFileName);
                    boolean bl = uniqueNameGenerated = renamedFileInfo == null;
                    if (!uniqueNameGenerated) continue;
                    fileName = possibleFileName;
                    logger.info("Attempting to resolve filename conflict for {} on the remote server by using a newly generated filename of: {}", new Object[]{flowFile, fileName});
                    destinationRelationship = REL_SUCCESS;
                    transferFile = true;
                    penalizeFile = false;
                    break;
                }
                if (uniqueNameGenerated) break;
                destinationRelationship = REL_REJECT;
                transferFile = false;
                penalizeFile = false;
                logger.warn("Could not determine a unique name after 99 attempts for.  Switching resolution mode to REJECT for {}", new Object[]{flowFile});
                break;
            }
            case "IGNORE": {
                destinationRelationship = REL_SUCCESS;
                transferFile = false;
                penalizeFile = false;
                logger.info("Resolving conflict for {}  by not transferring file and and still considering the process a success.", new Object[]{flowFile});
                break;
            }
            case "FAIL": {
                destinationRelationship = REL_FAILURE;
                transferFile = false;
                penalizeFile = true;
                logger.warn("Resolved filename conflict for {} as configured by routing to FAILURE relationship.", new Object[]{flowFile});
            }
        }
        return new ConflictResult(destinationRelationship, transferFile, (String)fileName, penalizeFile);
    }

    private static class ConflictResult {
        final Relationship relationship;
        final boolean transferFile;
        final String newFileName;
        final boolean penalizeFile;

        public ConflictResult(Relationship relationship, boolean transferFileVal, String newFileNameVal, boolean penalizeFileVal) {
            this.relationship = relationship;
            this.transferFile = transferFileVal;
            this.newFileName = newFileNameVal;
            this.penalizeFile = penalizeFileVal;
        }

        public boolean isTransfer() {
            return this.transferFile;
        }

        public boolean isPenalize() {
            return this.penalizeFile;
        }

        public String getFileName() {
            return this.newFileName;
        }

        public Relationship getRelationship() {
            return this.relationship;
        }
    }
}

