/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.box;

import com.box.sdk.BoxAPIConnection;
import com.box.sdk.BoxAPIResponseException;
import com.box.sdk.BoxFile;
import com.box.sdk.Metadata;
import java.io.InputStream;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.behavior.WritesAttributes;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.SeeAlso;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.box.controllerservices.BoxClientService;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.processors.box.FetchBoxFile;
import org.apache.nifi.processors.box.ListBoxFile;
import org.apache.nifi.processors.box.ListBoxFileMetadataTemplates;
import org.apache.nifi.processors.box.UpdateBoxFileMetadataInstance;
import org.apache.nifi.processors.box.utils.BoxDate;
import org.apache.nifi.serialization.RecordReader;
import org.apache.nifi.serialization.RecordReaderFactory;
import org.apache.nifi.serialization.record.Record;
import org.apache.nifi.serialization.record.RecordField;
import org.apache.nifi.serialization.record.RecordFieldType;

@InputRequirement(value=InputRequirement.Requirement.INPUT_REQUIRED)
@Tags(value={"box", "storage", "metadata", "templates", "create"})
@CapabilityDescription(value="Creates a metadata instance for a Box file using a specified template with values from the flowFile content. \nThe Box API requires newly created templates to be created with the scope set as enterprise so no scope is required. \nThe input record should be a flat key-value object where each field name is used as the metadata key.\n")
@SeeAlso(value={ListBoxFileMetadataTemplates.class, UpdateBoxFileMetadataInstance.class, ListBoxFile.class, FetchBoxFile.class})
@WritesAttributes(value={@WritesAttribute(attribute="box.id", description="The ID of the file for which metadata was created"), @WritesAttribute(attribute="box.template.key", description="The template key used for metadata creation"), @WritesAttribute(attribute="error.code", description="The error code returned by Box"), @WritesAttribute(attribute="error.message", description="The error message returned by Box")})
public class CreateBoxFileMetadataInstance
extends AbstractProcessor {
    public static final PropertyDescriptor FILE_ID = new PropertyDescriptor.Builder().name("File ID").description("The ID of the file for which to create metadata.").required(true).defaultValue("${box.id}").expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).build();
    public static final PropertyDescriptor TEMPLATE_KEY = new PropertyDescriptor.Builder().name("Template Key").description("The key of the metadata template to use for creation.").required(true).expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).build();
    public static final PropertyDescriptor RECORD_READER = new PropertyDescriptor.Builder().name("Record Reader").description("The Record Reader to use for parsing the incoming data").required(true).identifiesControllerService(RecordReaderFactory.class).build();
    public static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("A FlowFile is routed to this relationship after metadata has been successfully created.").build();
    public static final Relationship REL_FAILURE = new Relationship.Builder().name("failure").description("A FlowFile is routed to this relationship if an error occurs during metadata creation.").build();
    public static final Relationship REL_FILE_NOT_FOUND = new Relationship.Builder().name("file not found").description("FlowFiles for which the specified Box file was not found will be routed to this relationship.").build();
    public static final Relationship REL_TEMPLATE_NOT_FOUND = new Relationship.Builder().name("template not found").description("FlowFiles for which the specified metadata template was not found will be routed to this relationship.").build();
    private static final Set<Relationship> RELATIONSHIPS = Set.of(REL_SUCCESS, REL_FAILURE, REL_FILE_NOT_FOUND, REL_TEMPLATE_NOT_FOUND);
    private static final List<PropertyDescriptor> PROPERTY_DESCRIPTORS = List.of(BoxClientService.BOX_CLIENT_SERVICE, FILE_ID, TEMPLATE_KEY, RECORD_READER);
    private volatile BoxAPIConnection boxAPIConnection;

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return PROPERTY_DESCRIPTORS;
    }

    public Set<Relationship> getRelationships() {
        return RELATIONSHIPS;
    }

    @OnScheduled
    public void onScheduled(ProcessContext context) {
        BoxClientService boxClientService = (BoxClientService)context.getProperty(BoxClientService.BOX_CLIENT_SERVICE).asControllerService(BoxClientService.class);
        this.boxAPIConnection = boxClientService.getBoxApiConnection();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void onTrigger(ProcessContext context, ProcessSession session) throws ProcessException {
        FlowFile flowFile = session.get();
        if (flowFile == null) {
            return;
        }
        String fileId = context.getProperty(FILE_ID).evaluateAttributeExpressions(flowFile).getValue();
        String templateKey = context.getProperty(TEMPLATE_KEY).evaluateAttributeExpressions(flowFile).getValue();
        RecordReaderFactory recordReaderFactory = (RecordReaderFactory)context.getProperty(RECORD_READER).asControllerService(RecordReaderFactory.class);
        try (InputStream inputStream = session.read(flowFile);
             RecordReader recordReader = recordReaderFactory.createRecordReader(flowFile, inputStream, this.getLogger());){
            Metadata metadata = new Metadata();
            ArrayList<String> errors = new ArrayList<String>();
            Record record = recordReader.nextRecord();
            if (record != null) {
                this.processRecord(record, metadata, errors);
            } else {
                errors.add("No records found in input");
            }
            if (!errors.isEmpty()) {
                flowFile = session.putAttribute(flowFile, "error.message", String.join((CharSequence)", ", errors));
                session.transfer(flowFile, REL_FAILURE);
                return;
            }
            if (metadata.getOperations().isEmpty()) {
                flowFile = session.putAttribute(flowFile, "error.message", "No valid metadata key-value pairs found in the input");
                session.transfer(flowFile, REL_FAILURE);
                return;
            }
            BoxFile boxFile = this.getBoxFile(fileId);
            boxFile.createMetadata(templateKey, metadata);
        }
        catch (BoxAPIResponseException e) {
            flowFile = session.putAttribute(flowFile, "error.code", String.valueOf(e.getResponseCode()));
            flowFile = session.putAttribute(flowFile, "error.message", e.getMessage());
            if (e.getResponseCode() != 404) {
                this.getLogger().error("Couldn't create metadata for file with id [{}]", new Object[]{fileId, e});
                session.transfer(flowFile, REL_FAILURE);
                return;
            }
            String errorBody = e.getResponse();
            if (errorBody != null && errorBody.toLowerCase().contains("specified metadata template not found")) {
                this.getLogger().warn("Box metadata template with key {} was not found.", new Object[]{templateKey});
                session.transfer(flowFile, REL_TEMPLATE_NOT_FOUND);
                return;
            }
            this.getLogger().warn("Box file with ID {} was not found.", new Object[]{fileId});
            session.transfer(flowFile, REL_FILE_NOT_FOUND);
            return;
        }
        catch (Exception e) {
            this.getLogger().error("Error processing metadata creation for Box file [{}]", new Object[]{fileId, e});
            flowFile = session.putAttribute(flowFile, "error.message", e.getMessage());
            session.transfer(flowFile, REL_FAILURE);
            return;
        }
        Map<String, String> attributes = Map.of("box.id", fileId, "box.template.key", templateKey);
        flowFile = session.putAllAttributes(flowFile, attributes);
        session.getProvenanceReporter().create(flowFile, "%s%s/metadata/enterprise/%s".formatted("https://app.box.com/file/", fileId, templateKey));
        session.transfer(flowFile, REL_SUCCESS);
    }

    private void processRecord(Record record, Metadata metadata, List<String> errors) {
        if (record == null) {
            errors.add("No record found in input");
            return;
        }
        List fields = record.getSchema().getFields();
        if (fields.isEmpty()) {
            errors.add("Record has no fields");
            return;
        }
        for (RecordField field : fields) {
            this.addValueToMetadata(metadata, record, field);
        }
    }

    private void addValueToMetadata(Metadata metadata, Record record, RecordField field) {
        if (record.getValue(field) == null) {
            return;
        }
        RecordFieldType fieldType = field.getDataType().getFieldType();
        String fieldName = field.getFieldName();
        String path = "/" + fieldName;
        if (this.isNumber(fieldType)) {
            metadata.add(path, record.getAsDouble(fieldName).doubleValue());
        } else if (this.isDate(fieldType)) {
            LocalDate date = record.getAsLocalDate(fieldName, null);
            metadata.add(path, BoxDate.of(date).format());
        } else if (this.isArray(fieldType)) {
            List<String> values = Arrays.stream(record.getAsArray(fieldName)).filter(Objects::nonNull).map(Object::toString).toList();
            metadata.add(path, values);
        } else {
            metadata.add(path, record.getAsString(fieldName));
        }
    }

    private boolean isNumber(RecordFieldType fieldType) {
        boolean isInteger = RecordFieldType.BIGINT.equals((Object)fieldType) || RecordFieldType.BIGINT.isWiderThan(fieldType);
        boolean isFloat = RecordFieldType.DECIMAL.equals((Object)fieldType) || RecordFieldType.DECIMAL.isWiderThan(fieldType);
        return isInteger || isFloat;
    }

    private boolean isDate(RecordFieldType fieldType) {
        return RecordFieldType.DATE.equals((Object)fieldType);
    }

    private boolean isArray(RecordFieldType fieldType) {
        return RecordFieldType.ARRAY.equals((Object)fieldType);
    }

    BoxFile getBoxFile(String fileId) {
        return new BoxFile(this.boxAPIConnection, fileId);
    }
}

