/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.mahout.math.jet.random.engine;

import org.apache.mahout.math.MahoutTestCase;
import org.junit.Test;

import java.util.Date;

/**
 * Tests the Mersenne Twister against the reference implementation 991029/mt19937-2.c which can be
 * found at
 * <p/>
 * http://www.math.sci.hiroshima-u.ac.jp/~m-mat/MT/VERSIONS/C-LANG/991029/mt19937-2.c
 * <p/>
 * That implementation isn't exactly what our random number generator package will produce because
 * of two things: <ul> <li> The conversion to double is a bit different.  Ours uses nextLong, the
 * twister reference uses nextInt() which gives fewer sig figs. <li> The seed initialization which
 * was modified in Colt at some point.  Rather than argue with the new implementation, this test
 * uses a legacy method to initialize the PRNG state. <ul>
 */
public final class MersenneTwisterTest extends MahoutTestCase {

  /**
   * Convert an unsigned int stored in a long to a double in a fashion compatible with the C
   * reference implementation
   *
   * @param y The long value to convert
   * @return A double in the range [0..1)
   */
  private static double toDouble(long y) {
    return (y & 0xffffffffL) * 2.3283064365386963e-10;
  }

  @Test
  public void test10001() {
    MersenneTwister r = new MersenneTwister();
    r.setReferenceSeed(4357);

    // compares results against 1000 values from the reference implementation in C
    int i = 0;
    for (long x : reference1) {
      int y = r.nextInt();
      assertEquals("t-ref-int-" + i, x, y);
      i++;
    }

    r.setReferenceSeed(4357);
    i = 0;
    for (Double x : ref1) {
      assertEquals("t-ref-double-" + i, x, toDouble(r.nextInt()), 1.0e-7);
      i++;
    }
  }

  @Test
  public void testRegression() {
    RandomEngine r = new MersenneTwister(42);
    int i = 0;
    for (double x : reference3) {
      assertEquals("t-regression-"+i, x, r.nextDouble(), 1.0e-7);
      i++;
    }

  }

  @Test
  public void testDateConstructor() {
    RandomEngine r1 = new MersenneTwister(1275264362);
    RandomEngine r2 = new MersenneTwister(new Date(1275264362));
    for (int i = 0; i < 100; i++) {
      assertEquals("date-"+i, r1.nextInt(), r2.nextInt());
    }
  }

  // output of first 1000 values of nextInt() as determined by a slight modification of
  // the reference C implementation
  private final long[] reference1 = {
          -1427748157, 1585203162, -1181843167, -1341066457, -1831172428,
          -812701500, 1164297043, -696771727, 589972756, -182733429,
          767115311, -201891849, 1322433849, -937881972, -994918828,
          -645502951, -618362664, 1475054104, -1693033057, -874162432,
          -1802576116, 28597038, 1901037238, 1209433535, -714649522,
          -1806669844, 79873538, -986483224, -1381070953, -128771275,
          1930853421, -981423403, -1691237282, -1467414215, 1952080899,
          1405101208, 1959413290, -2072970131, -184835146, 1025637693,
          -624744829, -364164339, -1713958958, 1350592177, -387423596,
          -280265689, 935323183, 1742076441, -1111952306, -519637647,
          -1624564711, 631538770, -946857884, -1419824656, 1153190187,
          1034841106, -79535765, -925821475, -1850999675, 489641649,
          1500338112, -540983840, 714552875, 1260890119, -1342298899,
          -957774496, 555421782, 74804828, 1151768793, 1410928776,
          1120209439, -2066769033, -237465635, -1180052315, -1750907701,
          -762815280, -2007402377, 2032846052, 1496843742, -961876751,
          1817300007, 250812909, 1159923179, 685307794, -753202492,
          461984834, -1147951877, -923915622, 740493100, 1639766659,
          890836426, 1816610558, 740465142, 185885280, -35093869,
          -336704055, -29526693, 921638366, -140388124, 805230647,
          -1280559346, 1805221957, 561367725, -1589170300, 1548190021,
          -1783658974, 1037370857, -826584078, -2031446834, 100585698,
          1357493989, -680658300, 171284736, -2012746585, -771924827,
          -1946140299, 107252578, -18732539, -827495828, 226242558,
          1135756939, 1172828762, 1603997125, -1903426140, -1142976832,
          1103261409, -1140155964, 1113077002, -758341955, -1342546593,
          1239175044, -1726279891, -1447125076, -996227715, -529145016,
          1818781797, 1788591806, 1276077776, 949399679, 539881640,
          -1779593820, -1016654008, -2099702785, -1154248338, -2049180834,
          -322038658, 927276666, 511152533, -1508946065, -405678848,
          1460095448, -813645541, -1649241752, -1737367565, -1286613991,
          -1956797971, -579180754, -1967302698, 1670522570, 398515741,
          -382719009, 1871479044, 819271481, 1603628159, -2051995122,
          -1088019581, -1300651034, -815599548, -1866550794, -1289775408,
          596767713, -18581025, 629871933, 1632138881, 2115083989,
          801211500, -467095127, 1960552366, 1795052222, -1315487384,
          87338528, -1145722701, 643926064, -777924694, 2067922190,
          -1494259262, 196411345, 1750850708, 1469706228, -1383121853,
          1988965279, -1494611845, -1489487232, -2102956279, -620060911,
          -1613254841, -67574825, -1566118070, -965262195, 1906618821,
          -1152667674, 1154270206, -131272675, 1798769436, -281117605,
          2028970854, -2078898651, 1849210321, -187737964, -1779714343,
          -772687562, 1983720317, 2024886240, -757295103, -364749686,
          72039252, 1780891549, -766599792, 251685934, 1188909432,
          1947153461, 370404364, -480969831, 162450456, -477729904,
          -1270154148, 114391043, -1104216657, -1557949265, 1006449521,
          -648240330, 287956073, 1149908372, 487404965, 2110874122,
          -125178374, -1586846588, 1662089132, 509477256, 1099251950,
          -1355413830, -178051486, -2057816328, -1762332032, 1924974342,
          -2019883858, 2001527850, 85819681, 1193976695, -1433625679,
          1014135054, 996772449, -130739282, 1090591843, 2130489362,
          769151670, 995393661, 930169499, 53462752, 966539569,
          1906551696, -833969969, -413728139, -2018807943, 119179965,
          1765229953, -2026794687, 1085009887, -1619195910, 458208333,
          1454994770, 1017850103, -1645532781, -1295503958, 388687986,
          -879677508, -1271340169, 1525959519, -1842566946, 2054320272,
          -663278028, -2045782330, 2108263178, -1728324110, 614451766,
          1697197121, 1906804322, 1965059615, -609843883, 1808965526,
          -956124083, -530429769, -390790033, 654414333, -925784101,
          172707682, 1789534658, 99114865, 197192909, 1307666933,
          -1019527615, -1132743980, -2135742864, -1863142718, 479324619,
          1925592859, -1667087835, 792010626, 472668456, -117229595,
          -1695684170, -860776396, 1904530818, 1875549291, -1512584973,
          1854703958, 335047755, 782741003, -945192623, 2070491163,
          1444691247, -161427997, -1273883797, -1436594303, -45273895,
          -1179475147, -1469181637, -1070813498, -143531925, -8674298,
          1256416790, 1517863100, 745532812, -1537840807, -281538946,
          -1871419122, -2106555353, -465504136, 1559482295, 745669519,
          -854995128, 161134094, -590261774, 2092939826, -567201176,
          1109991787, -728753805, 1293706699, 1909349236, -1874770772,
          -1990220607, -659823429, -743007503, -6348644, 669034647,
          -553855348, -2095051428, -2126855025, 1697649112, 1070896705,
          -1504289523, 929567277, -476701944, 1552321291, 1984195592,
          2012174866, -981280665, -1107700217, -551504005, -838950404,
          -594420119, 673536746, -613800459, 639979024, 908071910,
          285351138, -814104229, -1568903254, -861750524, -115021904,
          -675957356, -1389846974, 1106151469, 2104819134, -858223142,
          -1779443430, -1923129246, 1019585697, -667014853, 875737047,
          370873717, -1354073499, 685464258, 748098950, -497781789,
          1051282146, -2012134278, 477323378, -242142488, -884198479,
          -1586803086, -384887197, 801512007, -1129707433, -1196045924,
          202576730, -1238107410, -1121307603, 1703200672, 1431918718,
          1548107888, -196076398, -269838779, 1676726643, -1109727040,
          1377418328, 682127434, 229406958, 536272508, 1302848757,
          -2072324870, -893647687, 342519951, -803149926, 880356224,
          -2006463724, -1734780468, 904726294, -1392156102, -2055422122,
          1380829270, 1525997083, 799156132, -430755091, 768044736,
          -1719451388, -594570907, 1384921240, 1784666717, 2072282702,
          -2021855225, 73034983, 1893910430, 554586922, 1002452453,
          244890730, 1914712138, -1191824507, -194588306, 15082512,
          -156127661, 286473904, 28127927, -9249080, -1041617435,
          848606688, 751913560, -1382303292, -29779765, -1007560893,
          -1222490607, 1465740219, 1119222960, 1706572418, 1637696965,
          1301369982, 994855959, -1122712575, -1507911605, -257360398,
          2001758001, -1226516195, 1913702326, 164846220, -908248841,
          -1934384520, 538955390, 1122472306, -2131179400, -474142027,
          473138414, 403084459, 780682841, 1916392897, 2137309162,
          -1321836898, 1560974537, 383098889, -1471156037, 95656605,
          1100898974, 2054803549, 1432869091, 39447374, -429164754,
          -2051585445, 326718209, -530310478, -439567291, -470318143,
          1760031250, -404864950, 38307054, -1046092764, 997675099,
          1302054089, 1134435193, -536378330, -438351386, -299093902,
          1997896248, -1953476615, -155946522, -75641431, -1414340477,
          1553845962, 739906589, -1626694751, 1218350904, 1726725379,
          -346060453, 1266205878, -1282094066, -2054418268, 377137844,
          907277693, -1734159739, 211027090, 1865109379, 1680481889,
          -1288411097, 1195935070, 538932664, 781359924, 1623787727,
          2139406835, 695091061, 1267298086, 644554442, -1617615690,
          1079279370, -1749865717, 536974673, -885076301, 328220108,
          -466845841, -1322702066, -1655966850, 208138402, -974382512,
          -931090593, -2133184188, -1267198043, -1476603032, 1836513581,
          -1594878134, -583857690, -1587570764, -2042444461, 542106924,
          290861120, 65506152, -98553653, 358020078, 911900450,
          -1492901891, -1106133249, 616484753, -118794916, -802153351,
          795652043, 342823405, -443791797, -1817348094, 974180631,
          2070155077, -1246351341, -799883912, 650122283, 14857981,
          1511525369, 1296338035, 240671856, -126886561, -2133396802,
          148980922, 962978395, 1307706563, -970011125, 1322718131,
          1734073640, -1905434541, 382604108, 432552756, 311652177,
          2123865787, -137050255, -1140600969, -1316759712, 1290868994,
          2007821863, -1034107165, -1654418680, -890855147, 1612513188,
          -1236187897, -7333467, -1372147252, -1261951798, -783395381,
          991947766, 643957286, -1858245905, -718349669, -1917970862,
          1222099653, -619578703, 1992634707, -1512305261, 1200611853,
          159589769, 679751733, -1133018403, 589403970, 1044672925,
          1965277612, 1713774522, 1215426812, 1813238647, -1248133092,
          1409105205, -1043888977, -762698605, -1678957447, 1386097548,
          174492254, -316159836, -1473858659, 2107299989, -833690268,
          491302117, 2132304916, -112677511, 2030842100, 926165801,
          1843009221, 4893907, 1091318508, -1715991292, -865551137,
          -829964760, -247893586, -824303510, 489006494, 1589495825,
          -1807323926, 50052787, -110072294, 478354171, -212240106,
          -1311579972, -306748207, -1347854915, -1556993453, 650841551,
          207036108, 469279577, -1611389253, -1277334795, 163011627,
          1693305508, -390991392, -1271239405, -1328800623, 1330783281,
          714904080, 487395952, -1100790740, 132096950, -822975875,
          -961687998, 964375359, 1852875757, -896777589, 1367148533,
          -1266600542, 1975064118, 1410547156, 930454876, 1797368724,
          1024933376, -78619762, -1368171896, -1514005974, -1344795504,
          254604891, -1833902950, -1193911864, -228510260, -2018919711,
          2005729740, 702724217, 352362105, -262963916, -426257329,
          -1232517259, 1984810322, 1122228626, 1836441070, -1245806501,
          -789776028, 1416228722, -1283534029, -675996896, -1660586550,
          521960665, 1258860577, -648003051, 482700194, 576607559,
          1025672363, 1619715344, -1363125918, -1911294560, -126546834,
          -395156910, -1569812412, 1643576117, -181522341, 1695213625,
          929397946, -164580871, -1016717391, 338698100, -1673405423,
          760291850, 1827462673, -1340989859, -1492476436, 952114118,
          1564374766, 1510012037, -1777737089, -1415712186, 1376858550,
          545337582, 275466025, 1766675460, 841474537, -2015920460,
          160777324, 1685578416, 97066962, 70420376, -1597258768,
          1312050661, 277966306, -1782698970, -734927361, -587293556,
          1802069091, -2080657689, -615003873, -1495278282, 2064212423,
          -1776297165, 1071037431, -1005864183, -1462724740, 626819162,
          -1473066947, -89564995, -1202424254, 1327800001, 2111287012,
          -937976900, 2106560895, 980585039, -1920505273, -321986311,
          1815212590, -1360852458, -1827983614, -1647880750, 1917133448,
          757801743, -1494540856, -832888837, 1864979383, -583359823,
          -1050492371, 1082752058, 1953813482, -722323323, 1845637828,
          1693567413, 1911166886, 13616514, 1527215536, 944105697,
          -574011490, -1283841468, -635113261, -803797107, 1206734392,
          -1830776329, 259012728, 351640621, 348063345, 237157049,
          -1702230489, -822882825, 967215995, 1941684155, -961498118,
          -51144549, -1457068641, -2121321987, 2144297387, -209192573,
          -1555878233, 1721981340, 1029091741, -431252539, -1125424606,
          1736267037, 1087293093, -553560177, -1291607021, -255814672,
          965607200, 1494605756, -1594157835, 1862970278, 2060187203,
          -1658445282, -1998000848, 2138904618, 830770044, -1885127998,
          946077889, 676163933, -1957194037, 1967578486, 1364110453,
          1120981439, 1505148538, -1118074866, -506466807, -2073526847,
          -1417161351, 1848017526, 1060239654, -1313859409, 394732496,
          775694702, -1829536644, 256035109, 672200319, -260601206,
          652227015, 224664253, -1462432907, -1704215028, 1616461983,
          474981047, 1729743343, 1033065486, -615149661, 1945677673,
          -152843588, -853261901, -671603644, -1406112499, -445972447,
          -1520498331, 733621242, 1559397411, 610848083, 1861857918,
          290039215, 1969085557, -1345508317, -148246158, 1709209040,
          -1190325503, -1880289029, -592854369, 1922014713, -1288666211,
          -1737534601, 1953275956, -819604769, -854143275, 1879890923,
          -175971242, 524339283, -2046891972, 1608569809, 1946682979,
          1598268490, -831293677, 456566908, 1513321122, 1918511241,
          -1289528135, 1218124360, -1680766564, 1479017575, -1789824800,
          -1160853575, 2091183380, -2062027073, 1740181808, -2092634913,
          2146620821, 2123780007, -894876271, -1152629704, 1128001831,
          -639377490, -595026093, 871253103, 1532083500, 883298808,
          2042462485, 1141767415, -989784431, -2048640156, 1481051047,
          -1827730611, 1053025675, 1128030915, 579308527, -200625939,
          -982376809, -1656203101, -70360337, 975347930, -1514279015,
          -1045859905, 1737859059, 562230366, -1187729158, 1630808317,
          -971015197, -1606723684, 1676966491, 1391319909, 1149056605,
          160522031, 1888652620, 1204167962, -1695150667, -198350693,
          -2141516092, -2060503969, -2040859804, -1496108351, 1904589021,
          -1212687768, 618798780, -1132067082, 746972615, -142651982,
          -653590385, 347959902, 847266746, -851701254, -82388269,
          2141521991, -1925233602, -242968079, -1860599362, -2020834628,
          -355708780, 365865295, 1762948411, -529670161, -10071373,
          1178072377, 1164762067, -1990293332, -1693036209, 660951723,
          324246673, 1237854749, 1390986182, -1171550610, -1065493866,
          647145616, -2066410839, -810069615, -440596341, -580317728,
          1924448690, 275933853, -1040838222, 1742363440, 1030650439
  };

  // the first 1000 values output by the reference implementation
  // data from http://www.math.sci.hiroshima-u.ac.jp/~m-mat/MT/VERSIONS/C-LANG/991029/mt19937-2.out
  private final double[] ref1 = {
          0.66757648, 0.36908387, 0.72483069, 0.68775863, 0.57364694,
          0.81077819, 0.27108403, 0.83777019, 0.13736374, 0.95745406,
          0.17860795, 0.95299339, 0.30790312, 0.78163234, 0.76835241,
          0.84970713, 0.85602622, 0.34343780, 0.60581002, 0.79646820,
          0.58030504, 0.00665827, 0.44261972, 0.28159319, 0.83360769,
          0.57935190, 0.01859701, 0.77031648, 0.67844436, 0.97001810,
          0.44956184, 0.77149456, 0.60622814, 0.65834100, 0.45450425,
          0.32715062, 0.45621146, 0.51734903, 0.95696471, 0.23879988,
          0.85454026, 0.91521138, 0.60093783, 0.31445925, 0.90979591,
          0.93474556, 0.21777190, 0.40560878, 0.74110343, 0.87901243,
          0.62175155, 0.14704158, 0.77954247, 0.66942131, 0.26849801,
          0.24094272, 0.98148164, 0.78444039, 0.56903055, 0.11400358,
          0.34932469, 0.87404238, 0.16636981, 0.29357386, 0.68747168,
          0.77700075, 0.12931921, 0.01741686, 0.26816707, 0.32850746,
          0.26081908, 0.51879284, 0.94471072, 0.72524766, 0.59233503,
          0.82239323, 0.53261521, 0.47330885, 0.34851109, 0.77604562,
          0.42312313, 0.05839693, 0.27006566, 0.15956065, 0.82463138,
          0.10756423, 0.73272163, 0.78488413, 0.17240949, 0.38178793,
          0.20741402, 0.42296261, 0.17240298, 0.04327979, 0.99182907,
          0.92160498, 0.99312528, 0.21458565, 0.96731334, 0.18748237,
          0.70184654, 0.42031099, 0.13070361, 0.62999246, 0.36046608,
          0.58470953, 0.24153172, 0.80754590, 0.52701693, 0.02341943,
          0.31606620, 0.84152189, 0.03988034, 0.53137092, 0.82027225,
          0.54687890, 0.02497169, 0.99563849, 0.80733361, 0.05267620,
          0.26443902, 0.27307048, 0.37345968, 0.55682407, 0.73387997,
          0.25687306, 0.73453675, 0.25915843, 0.82343475, 0.68741401,
          0.28851792, 0.59806914, 0.66306494, 0.76804766, 0.87679883,
          0.42346814, 0.41643898, 0.29711001, 0.22104934, 0.12570099,
          0.58565603, 0.76329179, 0.51112485, 0.73125562, 0.52288791,
          0.92501953, 0.21589842, 0.11901197, 0.64867112, 0.90554553,
          0.33995496, 0.81055838, 0.61600598, 0.59548759, 0.70043684,
          0.54439747, 0.86514897, 0.54195165, 0.38894885, 0.09278668,
          0.91089129, 0.43573767, 0.19075151, 0.37337378, 0.52223265,
          0.74667570, 0.69716858, 0.81010343, 0.56540978, 0.69970076,
          0.13894581, 0.99567377, 0.14665349, 0.38001195, 0.49245637,
          0.18654659, 0.89124594, 0.45647667, 0.41794316, 0.69371423,
          0.02033509, 0.73324065, 0.14992572, 0.81887529, 0.48147565,
          0.65209065, 0.04573058, 0.40765170, 0.34219265, 0.67796685,
          0.46309207, 0.65200856, 0.65320173, 0.51036734, 0.85563082,
          0.62438484, 0.98426651, 0.63535972, 0.77525738, 0.44391929,
          0.73162364, 0.26874948, 0.96943570, 0.41880865, 0.93454721,
          0.47240659, 0.51596869, 0.43055283, 0.95628885, 0.58562796,
          0.82009466, 0.46187088, 0.47145557, 0.82367849, 0.91507510,
          0.01677295, 0.41464613, 0.82151208, 0.05860020, 0.27681455,
          0.45335699, 0.08624149, 0.88801548, 0.03782344, 0.88876984,
          0.70426919, 0.02663374, 0.74290453, 0.63726167, 0.23433229,
          0.84906979, 0.06704500, 0.26773391, 0.11348281, 0.49147618,
          0.97085464, 0.63053349, 0.38698528, 0.11862192, 0.25593954,
          0.68441813, 0.95854416, 0.52087730, 0.58967510, 0.44819302,
          0.52970914, 0.46601702, 0.01998145, 0.27799436, 0.66620801,
          0.23612172, 0.23207917, 0.96955989, 0.25392320, 0.49604321,
          0.17908208, 0.23175815, 0.21657196, 0.01244777, 0.22504003,
          0.44390366, 0.80582624, 0.90367141, 0.52995965, 0.02774875,
          0.41099963, 0.52810009, 0.25262355, 0.62300157, 0.10668494,
          0.33876737, 0.23698669, 0.61686954, 0.69836698, 0.09049847,
          0.79518412, 0.70399305, 0.35529014, 0.57099395, 0.47830871,
          0.84556855, 0.52367918, 0.49086827, 0.59759319, 0.14306320,
          0.39515950, 0.44396248, 0.45752609, 0.85800966, 0.42118261,
          0.77738501, 0.87649970, 0.90901211, 0.15236771, 0.78444909,
          0.04021164, 0.41665851, 0.02307698, 0.04591255, 0.30446493,
          0.76262273, 0.73626249, 0.50273361, 0.56620328, 0.11160146,
          0.44833702, 0.61185087, 0.18440434, 0.11005170, 0.97270536,
          0.60519276, 0.79958488, 0.44343314, 0.43668535, 0.64782387,
          0.43183192, 0.07800938, 0.18224609, 0.77993019, 0.48207379,
          0.33636839, 0.96241462, 0.70340082, 0.66551682, 0.98945885,
          0.72538204, 0.65792949, 0.75068180, 0.96658137, 0.99798036,
          0.29253233, 0.35340504, 0.17358288, 0.64194354, 0.93444911,
          0.56427628, 0.50952936, 0.89161637, 0.36309527, 0.17361471,
          0.80093093, 0.03751695, 0.86256897, 0.48730053, 0.86793819,
          0.25844010, 0.83032378, 0.30121456, 0.44455501, 0.56349591,
          0.53661566, 0.84637289, 0.82700508, 0.99852184, 0.15577177,
          0.87104550, 0.51220783, 0.50480298, 0.39526474, 0.24933757,
          0.64975530, 0.21643175, 0.88900918, 0.36142797, 0.46198154,
          0.46849597, 0.77152779, 0.74209345, 0.87159297, 0.80466664,
          0.86160078, 0.15681999, 0.85708844, 0.14900673, 0.21142697,
          0.06643849, 0.81045159, 0.63471125, 0.79935807, 0.97321938,
          0.84261641, 0.67640103, 0.25754596, 0.49006639, 0.80017935,
          0.58569104, 0.55223658, 0.23739079, 0.84469850, 0.20389842,
          0.08635077, 0.68473020, 0.15959708, 0.17418036, 0.88410115,
          0.24477070, 0.53151348, 0.11113551, 0.94362181, 0.79413150,
          0.63054362, 0.91038647, 0.18661656, 0.73696949, 0.72152386,
          0.04716607, 0.71173065, 0.73892523, 0.39655731, 0.33339456,
          0.36044696, 0.95434741, 0.93717326, 0.39039334, 0.74162154,
          0.32070520, 0.15882017, 0.05341297, 0.12486067, 0.30334311,
          0.51749927, 0.79193143, 0.07974914, 0.81300209, 0.20497391,
          0.53283376, 0.59608995, 0.21064801, 0.67586340, 0.52143474,
          0.32149937, 0.35529888, 0.18606804, 0.89970701, 0.17882435,
          0.59965903, 0.86156567, 0.32245210, 0.41552510, 0.48249092,
          0.52925015, 0.01700478, 0.44096038, 0.12912483, 0.23340165,
          0.05701807, 0.44580366, 0.72250673, 0.95469388, 0.00351167,
          0.96364870, 0.06669990, 0.00654904, 0.99784653, 0.75747954,
          0.19758164, 0.17506852, 0.67815743, 0.99306636, 0.76540895,
          0.71536673, 0.34126924, 0.26058940, 0.39734235, 0.38130604,
          0.30299881, 0.23163295, 0.73859811, 0.64891197, 0.94007861,
          0.46607060, 0.71442944, 0.44556854, 0.03838125, 0.78853184,
          0.54961601, 0.12548533, 0.26134595, 0.50379613, 0.88960521,
          0.11016112, 0.09385041, 0.18176689, 0.44619499, 0.49763107,
          0.69223586, 0.36344271, 0.08919716, 0.65746979, 0.02227179,
          0.25632302, 0.47842123, 0.33361583, 0.00918456, 0.90007729,
          0.52232804, 0.07607001, 0.87652747, 0.89765526, 0.89049552,
          0.40978921, 0.90573503, 0.00891906, 0.75643755, 0.23228934,
          0.30315809, 0.26413128, 0.87511469, 0.89793836, 0.93036177,
          0.46517147, 0.54517078, 0.96369087, 0.98238836, 0.67069820,
          0.36178296, 0.17227293, 0.62125561, 0.28366943, 0.40203458,
          0.91942652, 0.29481153, 0.70148921, 0.52166847, 0.08780925,
          0.21124205, 0.59623447, 0.04913357, 0.43425462, 0.39126768,
          0.70001841, 0.27845033, 0.12548004, 0.18192453, 0.37806754,
          0.49811947, 0.16183850, 0.29506583, 0.15007203, 0.62336950,
          0.25128931, 0.59257764, 0.12502416, 0.79392712, 0.07641970,
          0.89130398, 0.69203443, 0.61444017, 0.04846100, 0.77313389,
          0.78321358, 0.50332935, 0.70495746, 0.65620157, 0.42759664,
          0.62866350, 0.86406004, 0.63036488, 0.52445634, 0.12621910,
          0.06772138, 0.01525184, 0.97705369, 0.08335805, 0.21231837,
          0.65240669, 0.74245828, 0.14353654, 0.97234090, 0.81323412,
          0.18525218, 0.07981979, 0.89667167, 0.57686567, 0.22681910,
          0.48199554, 0.70981122, 0.81376251, 0.15136839, 0.00345939,
          0.35192942, 0.30182722, 0.05603578, 0.97045692, 0.50327985,
          0.03468732, 0.22421088, 0.30447416, 0.77415169, 0.30796931,
          0.40374548, 0.55635645, 0.08908196, 0.10071154, 0.07256218,
          0.49450104, 0.96809050, 0.73443314, 0.69341799, 0.30055386,
          0.46748246, 0.75922816, 0.61480063, 0.79258162, 0.37544248,
          0.71217758, 0.99829254, 0.68052207, 0.70617895, 0.81760155,
          0.23095584, 0.14993299, 0.56734341, 0.83274618, 0.55343761,
          0.28454225, 0.85574309, 0.46394642, 0.64788899, 0.27953923,
          0.03715739, 0.15826703, 0.73619860, 0.13723131, 0.24323187,
          0.45757685, 0.39901923, 0.28298861, 0.42217752, 0.70939637,
          0.32808287, 0.75695066, 0.82242039, 0.60908726, 0.32272598,
          0.04062714, 0.92638830, 0.65684054, 0.49064401, 0.80589136,
          0.11439019, 0.49646593, 0.97376522, 0.47284227, 0.21563978,
          0.42910902, 0.00113945, 0.25409239, 0.60046464, 0.79847317,
          0.80675877, 0.94228278, 0.80807688, 0.11385570, 0.37008334,
          0.57919961, 0.01165382, 0.97437180, 0.11137551, 0.95058400,
          0.69462399, 0.92857962, 0.68617807, 0.63748421, 0.15153586,
          0.04820435, 0.10926267, 0.62481920, 0.70259732, 0.03795410,
          0.39425341, 0.90896522, 0.70401651, 0.69061450, 0.30984713,
          0.16645158, 0.11348071, 0.74370218, 0.03075622, 0.80838600,
          0.77608956, 0.22453614, 0.43140626, 0.79120270, 0.31831407,
          0.70509658, 0.45985545, 0.32841860, 0.21663841, 0.41848252,
          0.23863590, 0.98169491, 0.68144766, 0.64749301, 0.68689040,
          0.05927982, 0.57301120, 0.72202073, 0.94679581, 0.52993362,
          0.46699535, 0.16361573, 0.08204070, 0.93877394, 0.90075423,
          0.71303221, 0.46212466, 0.26128921, 0.42757976, 0.70993807,
          0.81611594, 0.32974144, 0.70115395, 0.84260721, 0.61336456,
          0.12152844, 0.29310132, 0.84912503, 0.11238740, 0.13425191,
          0.23880796, 0.37711937, 0.68262252, 0.55499206, 0.97053602,
          0.90799536, 0.63449957, 0.38267489, 0.95773604, 0.39469768,
          0.21639232, 0.96168053, 0.76327704, 0.07885930, 0.61037994,
          0.17701924, 0.42548931, 0.68777647, 0.65250575, 0.22168134,
          0.36423438, 0.35157707, 0.58608833, 0.67037882, 0.32057486,
          0.12697130, 0.06413693, 0.41133618, 0.19592106, 0.53063194,
          0.03743389, 0.39245431, 0.02260016, 0.01639602, 0.62810921,
          0.30548560, 0.06471907, 0.58493305, 0.82888639, 0.86326006,
          0.41957691, 0.51555913, 0.85680825, 0.65185340, 0.48061191,
          0.58642359, 0.24937033, 0.76580399, 0.65943286, 0.14594271,
          0.65702487, 0.97914652, 0.72003879, 0.30915253, 0.49157231,
          0.78161023, 0.49047193, 0.22831025, 0.55284752, 0.92503172,
          0.42263712, 0.68315185, 0.57438940, 0.61632286, 0.44636741,
          0.17643947, 0.65202509, 0.80607796, 0.43422435, 0.86417596,
          0.75541319, 0.25209786, 0.45490765, 0.83182100, 0.42972104,
          0.39431439, 0.44497822, 0.00317034, 0.35558258, 0.21981674,
          0.86635254, 0.70108236, 0.85212617, 0.81285140, 0.28096475,
          0.57373917, 0.06030610, 0.08187271, 0.08103981, 0.05521743,
          0.60366858, 0.80840766, 0.22519752, 0.45208357, 0.77613377,
          0.98809198, 0.66074977, 0.50609124, 0.49925814, 0.95129356,
          0.63774387, 0.40093002, 0.23960409, 0.89959119, 0.73796666,
          0.40425617, 0.25315515, 0.87111423, 0.69927431, 0.94043851,
          0.22482295, 0.34799002, 0.62883121, 0.43375657, 0.47967471,
          0.61386312, 0.53480418, 0.49800254, 0.19342872, 0.56108443,
          0.22027592, 0.15743168, 0.54430525, 0.45811257, 0.31760671,
          0.26099883, 0.35044470, 0.73967791, 0.88207901, 0.51721941,
          0.67004141, 0.43027511, 0.24685628, 0.69409327, 0.09190582,
          0.18060550, 0.57402781, 0.05961282, 0.15650883, 0.93932405,
          0.15185844, 0.05230872, 0.65950080, 0.60320652, 0.37636188,
          0.11059014, 0.40273726, 0.24052930, 0.85677431, 0.45301338,
          0.96441333, 0.80133448, 0.84363009, 0.67261392, 0.89616395,
          0.64598139, 0.17080951, 0.36307550, 0.14222415, 0.43349758,
          0.06753002, 0.45846346, 0.68672443, 0.96548375, 0.39795624,
          0.72285575, 0.56221110, 0.86196534, 0.44750392, 0.69995902,
          0.59544870, 0.45478250, 0.80917089, 0.80112927, 0.43769621,
          0.95902850, 0.12208225, 0.52342082, 0.37452434, 0.45324745,
          0.37212588, 0.80644936, 0.10630277, 0.35234753, 0.44668821,
          0.69975833, 0.28361668, 0.60866604, 0.34436061, 0.58327394,
          0.72971771, 0.48689157, 0.51989691, 0.40516765, 0.51277047,
          0.49979911, 0.49448107, 0.79164538, 0.73163248, 0.26263339,
          0.85113333, 0.86145969, 0.20285442, 0.35671599, 0.20565903,
          0.47554786, 0.26583844, 0.76954785, 0.52301379, 0.34483407,
          0.57444831, 0.24517665, 0.26264016, 0.13488078, 0.95328813,
          0.77127258, 0.61438517, 0.98361796, 0.22709089, 0.64742944,
          0.75649177, 0.40462684, 0.13090446, 0.72346026, 0.37970215,
          0.77391791, 0.62590549, 0.39044919, 0.32394191, 0.26753559,
          0.03737445, 0.43973621, 0.28036720, 0.60531698, 0.95381788,
          0.50138943, 0.52025153, 0.52482530, 0.65166013, 0.44344669,
          0.71764913, 0.14407532, 0.73642009, 0.17391811, 0.96678625,
          0.84782413, 0.08101573, 0.19726966, 0.80169785, 0.98081749,
          0.49861194, 0.55174662, 0.94342959, 0.56679545, 0.52948777,
          0.91718010, 0.08518465, 0.41046841, 0.87667656, 0.99765508,
          0.27429135, 0.27119230, 0.53659872, 0.60580929, 0.15388982,
          0.07549456, 0.28821052, 0.32386421, 0.72722712, 0.75192038,
          0.15067533, 0.51887623, 0.81139097, 0.89741567, 0.86488425,
          0.44807063, 0.06424586, 0.75766097, 0.40567560, 0.23996701
  };

  private final double[] reference3 = {
          0.09575212, 0.03326592, 0.10697744, 0.37992924, 0.13844205,
          0.07550350, 0.89109388, 0.50151205, 0.79222822, 0.41823034,
          0.84440729, 0.66568635, 0.87830555, 0.17024311, 0.43956848,
          0.15568711, 0.44433865, 0.15509045, 0.25706894, 0.54578702,
          0.49590286, 0.22401731, 0.54676760, 0.01344236, 0.49878244,
          0.11425735, 0.92811815, 0.33364944, 0.45523584, 0.30330454,
          0.71281391, 0.76541035, 0.65384557, 0.36548962, 0.78606296,
          0.23527360, 0.55992981, 0.30954199, 0.31097754, 0.45424875,
          0.37274760, 0.27568572, 0.89592171, 0.66181069, 0.19410842,
          0.26527840, 0.57097220, 0.50241738, 0.33047523, 0.15719302,
          0.72335021, 0.48632592, 0.26988200, 0.02962598, 0.68539884,
          0.41837634, 0.74918846, 0.29888093, 0.91695999, 0.44307476,
          0.85652083, 0.86902892, 0.51888942, 0.04191434, 0.15804193,
          0.82571235, 0.41822514, 0.77878083, 0.24677425, 0.93067399,
          0.66147883, 0.08167767, 0.68635270, 0.21531198, 0.17917009,
          0.61210917, 0.89142440, 0.20540484, 0.98484786, 0.64105697,
          0.67740091, 0.69698981, 0.24651169, 0.74108039, 0.22530642,
          0.93012240, 0.60056313, 0.48043165, 0.60459893, 0.89849029,
          0.72360913, 0.44201254, 0.34065371, 0.86086940, 0.81661531,
          0.82741425, 0.54029302, 0.11725746, 0.64111068, 0.78480954,
          0.10389839, 0.25918715, 0.08772679, 0.04966821, 0.12496534,
          0.33111983, 0.90714372, 0.68193514, 0.42431445, 0.83658259,
          0.37504970, 0.14869862, 0.79012165, 0.10846471, 0.98522992,
          0.47527037, 0.02974793, 0.35251748, 0.80962036, 0.61765703,
          0.38585344, 0.48789858, 0.07826998, 0.28860830, 0.95594578,
          0.70463541, 0.40793755, 0.54573732, 0.64536751, 0.00739575,
          0.16901933, 0.34837234, 0.73592536, 0.32724020, 0.57088019,
          0.28976575, 0.24089392, 0.84049551, 0.78418605, 0.68421081,
          0.56295548, 0.22439076, 0.16466916, 0.57982696, 0.08512823,
          0.34178794, 0.66295924, 0.18114724, 0.25777769, 0.51590296,
          0.81920810, 0.01251322, 0.49352028, 0.39729409, 0.28260467,
          0.66368229, 0.31167985, 0.05018547, 0.07948601, 0.00742122,
          0.53983623, 0.04312077, 0.48228226, 0.03693601, 0.39157325,
          0.35947770, 0.20659274, 0.42678597, 0.24304387, 0.95389365,
          0.24411614, 0.52725440, 0.49916495, 0.28801917, 0.92581127,
          0.13417299, 0.91819667, 0.66061804, 0.97479890, 0.88356344,
          0.72971172, 0.91781935, 0.81123852, 0.66991017, 0.19069337,
          0.33610088, 0.48529086, 0.97480851, 0.61725675, 0.23778267,
          0.89281974, 0.46734440, 0.73223030, 0.63382758, 0.98261468,
          0.31080283, 0.40987343, 0.88972044, 0.91491452, 0.73864976,
          0.71977485, 0.22949882, 0.23827910, 0.75796538, 0.63373192,
          0.77852707, 0.13147285, 0.39905918, 0.98074901, 0.67505513,
          0.53726390, 0.29738812, 0.43486852, 0.37772844, 0.66651982,
          0.59846935, 0.65808432, 0.75209822, 0.63925437, 0.55695771,
          0.43820710, 0.61858674, 0.51274335, 0.47826917, 0.58277081,
          0.57037206, 0.27480447, 0.21507891, 0.31928376, 0.24264734,
          0.27603027, 0.30597543, 0.05130251, 0.16922754, 0.96421093,
          0.31156402, 0.68282019, 0.68952485, 0.04260403, 0.55311981,
          0.34666010, 0.02820792, 0.58097105, 0.03094771, 0.73916029,
          0.96790360, 0.68735213, 0.29706159, 0.35237517, 0.32556654,
          0.34507865, 0.71775189, 0.48004937, 0.08464617, 0.90295000,
          0.89509077, 0.48900486, 0.85891582, 0.78559599, 0.70929274,
          0.17588442, 0.98907009, 0.32978426, 0.78820522, 0.87447705,
          0.33872695, 0.00621719, 0.14675062, 0.17750487, 0.59685984,
          0.63547209, 0.00425963, 0.23090550, 0.96244983, 0.34157976,
          0.11382939, 0.87948627, 0.31331161, 0.36878708, 0.00581891,
          0.90011438, 0.48569161, 0.51841765, 0.77118893, 0.52467643,
          0.39202666, 0.20858830, 0.87396695, 0.15565967, 0.43254322,
          0.34562149, 0.86639746, 0.91036207, 0.02206924, 0.84579132,
          0.54798554, 0.64647403, 0.82308719, 0.00250840, 0.71748825,
          0.63908262, 0.79827802, 0.83448213, 0.22413254, 0.43744489,
          0.20714656, 0.93100558, 0.61709875, 0.98877387, 0.68023638,
          0.96291950, 0.64453665, 0.99467975, 0.32048655, 0.22497285,
          0.21878946, 0.23361169, 0.63424833, 0.48893507, 0.55624310,
          0.57561503, 0.33065869, 0.28598930, 0.02059915, 0.89955576,
          0.91003985, 0.26868726, 0.97613304, 0.38723648, 0.66504190,
          0.02092824, 0.41636886, 0.35077139, 0.15265648, 0.20790285,
          0.20421888, 0.34162863, 0.39413524, 0.91310634, 0.81056322,
          0.48583758, 0.53282213, 0.14793039, 0.86648513, 0.30037967,
          0.13204684, 0.64860530, 0.86108906, 0.98664413, 0.08942630,
          0.17791490, 0.19107823, 0.37702045, 0.72403257, 0.96539768,
          0.69525971, 0.40577687, 0.37139153, 0.55812589, 0.59373464,
          0.03421771, 0.78394764, 0.56237131, 0.69683049, 0.61979606,
          0.12955550, 0.66382787, 0.59509876, 0.79766276, 0.08659076,
          0.07288888, 0.64689550, 0.02016563, 0.86808399, 0.53495614,
          0.22494508, 0.31779084, 0.27205262, 0.74975765, 0.60174621,
          0.27383379, 0.83377514, 0.94198377, 0.47070628, 0.96111268,
          0.63857051, 0.62490258, 0.88250860, 0.05129671, 0.73319113,
          0.09881131, 0.88649673, 0.52707143, 0.70178227, 0.02416467,
          0.85358190, 0.34640508, 0.79175930, 0.43554792, 0.93983171,
          0.26700712, 0.54769506, 0.41191831, 0.43383408, 0.63333933,
          0.00788010, 0.91803682, 0.13659728, 0.47918280, 0.72078780,
          0.20186758, 0.98634244, 0.39079655, 0.62432433, 0.92814197,
          0.42111281, 0.90462521, 0.51787318, 0.04000191, 0.82254138,
          0.56604171, 0.84049756, 0.87863315, 0.44926471, 0.49165355,
          0.59618969, 0.39500911, 0.08301515, 0.42440598, 0.07148232,
          0.50067089, 0.63170103, 0.88784773, 0.93568234, 0.12767691,
          0.51460267, 0.04551480, 0.29679326, 0.71869878, 0.66886604,
          0.99559746, 0.21373819, 0.07705207, 0.90162303, 0.29944698,
          0.76965061, 0.39307939, 0.58977358, 0.59955108, 0.88001699,
          0.88839849, 0.79028170, 0.35853755, 0.38961951, 0.76400921,
          0.04481058, 0.20498015, 0.25184305, 0.31290529, 0.09688802,
          0.46950488, 0.28448810, 0.20749227, 0.21774246, 0.16754650,
          0.84214893, 0.75998650, 0.59543810, 0.15768158, 0.82985541,
          0.53741576, 0.62024092, 0.56164251, 0.84685099, 0.43875932,
          0.14376269, 0.12191657, 0.74036581, 0.63042238, 0.39947516,
          0.97869470, 0.03388022, 0.55623888, 0.00560876, 0.19804017,
          0.35809862, 0.57770986, 0.20684322, 0.86952049, 0.05314201,
          0.31115161, 0.75297924, 0.24038078, 0.25543901, 0.29558365,
          0.04859761, 0.35811325, 0.98469545, 0.98575424, 0.85483618,
          0.57789363, 0.23820230, 0.77587080, 0.63739416, 0.73807512,
          0.08509698, 0.91990975, 0.56027216, 0.59227258, 0.89669956,
          0.48393170, 0.78349119, 0.07688876, 0.49048711, 0.97676233,
          0.64727299, 0.38584462, 0.28145301, 0.01550599, 0.18945752,
          0.87379996, 0.02890444, 0.13732257, 0.21823246, 0.71756652,
          0.56966958, 0.44550803, 0.69540274, 0.81009273, 0.71420855,
          0.11768296, 0.40865700, 0.65500886, 0.50676682, 0.13591700,
          0.22430768, 0.98437986, 0.11323729, 0.25948878, 0.90992219,
          0.24566214, 0.44020692, 0.79119580, 0.40191868, 0.86418681,
          0.96511463, 0.05257337, 0.42441068, 0.97284508, 0.78583947,
          0.03145277, 0.13306677, 0.62284921, 0.06526968, 0.27227572,
          0.02920410, 0.85618323, 0.88531784, 0.69550380, 0.91717720,
          0.94659912, 0.74391481, 0.16259777, 0.94226300, 0.10532974,
          0.91059056, 0.85529121, 0.63953821, 0.18559760, 0.17911454,
          0.24281455, 0.87419743, 0.43099459, 0.91572761, 0.32962105,
          0.05089396, 0.25399158, 0.49165522, 0.95659705, 0.67785446,
          0.35475829, 0.74463888, 0.72561228, 0.25482160, 0.44891496,
          0.07720984, 0.48395681, 0.74351843, 0.39979001, 0.00487332,
          0.69655613, 0.33494366, 0.61969097, 0.70448508, 0.65012519,
          0.32901089, 0.81829251, 0.44918533, 0.95604007, 0.00820664,
          0.34594212, 0.04635539, 0.46375257, 0.83024265, 0.37547015,
          0.91336615, 0.41038603, 0.65918687, 0.11727323, 0.30263102,
          0.14217266, 0.99648433, 0.99743383, 0.02021849, 0.06917298,
          0.82707160, 0.79692345, 0.01885254, 0.34061080, 0.61211031,
          0.61339898, 0.61745751, 0.49587059, 0.79893049, 0.97833123,
          0.98060653, 0.41235724, 0.87134599, 0.65061048, 0.91216122,
          0.15553742, 0.75017715, 0.89782523, 0.00424658, 0.05994507,
          0.18287370, 0.30758177, 0.10222843, 0.82329823, 0.32028853,
          0.51689417, 0.70411537, 0.45353180, 0.08223335, 0.82436051,
          0.80887412, 0.43933198, 0.08687836, 0.53211483, 0.18266968,
          0.15658701, 0.72414266, 0.32735886, 0.15145334, 0.91787533,
          0.54332984, 0.32279609, 0.42173936, 0.07588914, 0.30419505,
          0.81017966, 0.96845370, 0.63627574, 0.84727451, 0.88742977,
          0.44747925, 0.24776310, 0.05030551, 0.13016318, 0.62416476,
          0.23559481, 0.24573308, 0.82449689, 0.40184810, 0.29491292,
          0.97200511, 0.25879377, 0.60255217, 0.43286781, 0.16246353,
          0.83972849, 0.57959594, 0.03197305, 0.24436894, 0.67692210,
          0.55185838, 0.92005375, 0.25633985, 0.10481763, 0.67853816,
          0.31928186, 0.75554830, 0.37089603, 0.60122019, 0.84765698,
          0.00033093, 0.82914790, 0.35126384, 0.13373781, 0.22649988,
          0.25832871, 0.00805221, 0.69034736, 0.09688938, 0.94542655,
          0.69852256, 0.69562397, 0.96922480, 0.36608056, 0.11822842,
          0.87258933, 0.98896530, 0.61255493, 0.42210611, 0.64365300,
          0.23857627, 0.91837564, 0.14547014, 0.51612699, 0.64192328,
          0.34246590, 0.14268754, 0.15859495, 0.77095702, 0.01080264,
          0.87087159, 0.63060450, 0.69481937, 0.84240541, 0.95379007,
          0.45310809, 0.51997102, 0.82978527, 0.67140555, 0.45856560,
          0.48439090, 0.29327285, 0.36266237, 0.49221888, 0.98523206,
          0.83644511, 0.98197702, 0.13521117, 0.59523541, 0.92477075,
          0.51307875, 0.63397493, 0.77868858, 0.15233450, 0.37777192,
          0.22528878, 0.13693927, 0.59983268, 0.08442964, 0.93192468,
          0.12760373, 0.88325929, 0.83974645, 0.51676707, 0.34376976,
          0.94219728, 0.98044638, 0.63594832, 0.65941087, 0.03715859,
          0.43524771, 0.73237678, 0.20493458, 0.79839755, 0.74117581,
          0.31915454, 0.84986064, 0.43777352, 0.54335319, 0.32167330,
          0.31855880, 0.26673096, 0.65461456, 0.97878388, 0.42818357,
          0.05243212, 0.84445490, 0.05629283, 0.67748546, 0.79044509,
          0.25858006, 0.63659558, 0.19212188, 0.87702207, 0.00454776,
          0.59516209, 0.36875898, 0.73059753, 0.45192937, 0.83777660,
          0.33062473, 0.62286961, 0.76417643, 0.87551423, 0.95845652,
          0.74296579, 0.86395090, 0.32583226, 0.92001191, 0.32171319,
          0.84006563, 0.60971240, 0.79027957, 0.41323460, 0.46600458,
          0.51279550, 0.97895957, 0.97138109, 0.66438339, 0.09826154,
          0.45081157, 0.01368416, 0.17961563, 0.89936614, 0.53304857,
          0.29234901, 0.18708516, 0.60935567, 0.28833607, 0.13783674,
          0.49096809, 0.96374449, 0.20760292, 0.29729179, 0.18633984,
          0.62029967, 0.57835982, 0.05922195, 0.76742521, 0.41023979,
          0.71555097, 0.83481165, 0.88031880, 0.09863277, 0.71271637,
          0.33321575, 0.03350630, 0.04950694, 0.11842818, 0.56167640,
          0.66612105, 0.60076075, 0.42829282, 0.92528926, 0.55599765,
          0.44648768, 0.62865139, 0.65400034, 0.95861368, 0.03078074,
          0.48366601, 0.79292197, 0.11419146, 0.41982703, 0.15805725,
          0.36365063, 0.57828509, 0.23531032, 0.66927232, 0.06579232,
          0.34134493, 0.43320389, 0.81604946, 0.75374402, 0.59252676,
          0.14314112, 0.08008821, 0.61607044, 0.47622726, 0.70278089,
          0.82210241, 0.35791651, 0.28425965, 0.70573468, 0.84738383,
          0.43737448, 0.79827193, 0.76220665, 0.55732352, 0.03427711,
          0.92281258, 0.27413625, 0.58199877, 0.30013308, 0.87995056,
          0.48320941, 0.37881629, 0.15332319, 0.73774346, 0.73654414,
          0.66289244, 0.22828983, 0.01796034, 0.63758637, 0.76623816,
          0.63569452, 0.68143683, 0.00839874, 0.93280417, 0.05507095,
          0.62650092, 0.24592312, 0.71943274, 0.60509154, 0.33123011,
          0.38506249, 0.39467195, 0.71827800, 0.97414449, 0.31963712,
          0.95827787, 0.71162246, 0.25389547, 0.09090820, 0.33619408,
          0.48412707, 0.28775361, 0.93853186, 0.77479483, 0.67790150,
          0.95794532, 0.53957660, 0.57953073, 0.76021089, 0.39418710,
          0.26653863, 0.05670721, 0.61318126, 0.67558325, 0.46325899,
          0.97914211, 0.86432830, 0.42864641, 0.99048174, 0.29400621,
          0.92266204, 0.04397024, 0.29926884, 0.24685589, 0.08042319,
          0.25212710, 0.78726916, 0.04105533, 0.93783659, 0.37391839,
          0.98121781, 0.72464585, 0.94671099, 0.70072935, 0.06362176,
          0.32143080, 0.30587660, 0.01336253, 0.22886753, 0.95124328,
          0.36812657, 0.53327065, 0.98800260, 0.14293064, 0.91433127,
          0.17528358, 0.58460345, 0.79436018, 0.85536464, 0.19883063,
          0.86772242, 0.90513821, 0.69579779, 0.16524102, 0.05741454,
          0.17712384, 0.69604158, 0.32330351, 0.25526924, 0.16796039,
          0.68508886, 0.23090268, 0.55306973, 0.75485248, 0.70351626,
          0.15102021, 0.80210782, 0.43654393, 0.93555827, 0.12228843,
          0.27130315, 0.10070153, 0.48069392, 0.60335391, 0.30162777,
          0.90613764, 0.42626484, 0.16670167, 0.87120354, 0.34972825

  };
}
