/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kylin.common.util;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Function;
import com.google.common.base.Strings;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.collect.Iterables;
import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import org.apache.commons.lang3.StringUtils;
import org.apache.curator.RetryPolicy;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.framework.imps.CuratorFrameworkState;
import org.apache.curator.retry.ExponentialBackoffRetry;
import org.apache.curator.utils.ZKPaths;
import org.apache.hadoop.conf.Configuration;
import org.apache.kylin.common.KylinConfig;
import org.apache.kylin.common.util.ClassUtil;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Shell;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZKUtil {
    private static final Logger logger = LoggerFactory.getLogger(ZKUtil.class);
    private static final KylinConfig defaultKylinConfig = KylinConfig.getInstanceFromEnv();
    private static final String zkChRoot = ZKUtil.fixPath(defaultKylinConfig.getZookeeperBasePath(), defaultKylinConfig.getClusterName());
    private static Cache<String, CuratorFramework> CACHE = CacheBuilder.newBuilder().removalListener((RemovalListener)new RemovalListener<String, CuratorFramework>(){

        public void onRemoval(RemovalNotification<String, CuratorFramework> notification) {
            logger.info("CuratorFramework for zkString " + (String)notification.getKey() + " is removed due to " + notification.getCause());
            CuratorFramework curator = (CuratorFramework)notification.getValue();
            try {
                curator.close();
            }
            catch (Exception ex) {
                logger.error("Error at closing " + curator, (Throwable)ex);
            }
        }
    }).expireAfterWrite(10000L, TimeUnit.DAYS).build();

    private static String fixPath(String parent, String child) {
        String path = ZKPaths.makePath((String)parent, (String)child);
        try {
            if (Shell.WINDOWS) {
                return new File(path).toURI().getPath();
            }
            return new File(path).getCanonicalPath();
        }
        catch (IOException e) {
            logger.error("get canonical path failed, use original path", (Throwable)e);
            return path;
        }
    }

    public static String getZKConnectString(KylinConfig config) {
        String zkString = config.getZookeeperConnectString();
        if (zkString == null && (zkString = ZKUtil.getZKConnectStringFromHBase()) == null) {
            throw new RuntimeException("Please set 'kylin.env.zookeeper-connect-string' in kylin.properties");
        }
        return zkString;
    }

    public static String getZkRootBasedPath(String path) {
        return zkChRoot + "/" + path;
    }

    public static CuratorFramework getZookeeperClient(KylinConfig config) {
        RetryPolicy retryPolicy = ZKUtil.getRetryPolicy(config);
        return ZKUtil.getZookeeperClient(ZKUtil.getZKConnectString(config), retryPolicy);
    }

    private static CuratorFramework getZookeeperClient(final String zkString, final RetryPolicy retryPolicy) {
        if (StringUtils.isEmpty((CharSequence)zkString)) {
            throw new IllegalArgumentException("ZOOKEEPER_QUORUM is empty!");
        }
        try {
            CuratorFramework instance = (CuratorFramework)CACHE.get((Object)zkString, (Callable)new Callable<CuratorFramework>(){

                @Override
                public CuratorFramework call() throws Exception {
                    return ZKUtil.newZookeeperClient(zkString, retryPolicy);
                }
            });
            if (instance.getState() != CuratorFrameworkState.STARTED) {
                logger.warn("curator for {} is closed by others unexpectedly, reinitialize a new one", (Object)zkString);
                CACHE.invalidate((Object)zkString);
                instance = ZKUtil.getZookeeperClient(zkString, retryPolicy);
            }
            return instance;
        }
        catch (Throwable e) {
            throw new RuntimeException(e);
        }
    }

    @VisibleForTesting
    public static CuratorFramework newZookeeperClient() {
        return ZKUtil.newZookeeperClient(KylinConfig.getInstanceFromEnv());
    }

    @VisibleForTesting
    public static CuratorFramework newZookeeperClient(KylinConfig config) {
        RetryPolicy retryPolicy = ZKUtil.getRetryPolicy(config);
        return ZKUtil.newZookeeperClient(ZKUtil.getZKConnectString(config), retryPolicy);
    }

    @VisibleForTesting
    public static CuratorFramework newZookeeperClient(String zkString, RetryPolicy retryPolicy) {
        if (zkChRoot == null) {
            throw new NullPointerException("zkChRoot must not be null");
        }
        logger.info("zookeeper connection string: {} with namespace {}", (Object)zkString, (Object)zkChRoot);
        CuratorFramework instance = ZKUtil.getCuratorFramework(zkString, zkChRoot, retryPolicy);
        instance.start();
        logger.info("new zookeeper Client start: " + zkString);
        ZKUtil.createZkChRootIfNecessary(instance, zkString);
        return instance;
    }

    private static RetryPolicy getRetryPolicy(KylinConfig config) {
        int baseSleepTimeMs = config.getZKBaseSleepTimeMs();
        int maxRetries = config.getZKMaxRetries();
        return new ExponentialBackoffRetry(baseSleepTimeMs, maxRetries);
    }

    private static synchronized void createZkChRootIfNecessary(CuratorFramework instance, String zkString) {
        try {
            ExponentialBackoffRetry retryPolicy = new ExponentialBackoffRetry(1000, 3);
            if (instance.checkExists().forPath("/") == null) {
                CuratorFramework tmpCurator = ZKUtil.getCuratorFramework(zkString, null, (RetryPolicy)retryPolicy);
                tmpCurator.start();
                tmpCurator.create().creatingParentsIfNeeded().forPath(zkChRoot);
                tmpCurator.close();
            }
        }
        catch (KeeperException.NodeExistsException e) {
            logger.warn("The chRoot znode {} has been created by others", (Object)zkChRoot);
        }
        catch (Exception e) {
            throw new RuntimeException("Fail to check or create znode for chRoot " + zkChRoot + " due to ", e);
        }
    }

    private static CuratorFramework getCuratorFramework(String zkString, String zkChRoot, RetryPolicy retryPolicy) {
        if (!Strings.isNullOrEmpty((String)zkChRoot)) {
            zkString = zkString + zkChRoot;
        }
        return CuratorFrameworkFactory.newClient((String)zkString, (int)120000, (int)15000, (RetryPolicy)retryPolicy);
    }

    private static String getZKConnectStringFromHBase() {
        Configuration hconf = null;
        try {
            Class<Object> hbaseConnClz = ClassUtil.forName("org.apache.kylin.storage.hbase.HBaseConnection", Object.class);
            hconf = (Configuration)hbaseConnClz.getMethod("getCurrentHBaseConfiguration", new Class[0]).invoke(null, new Object[0]);
        }
        catch (Throwable ex) {
            logger.warn("Failed to get zookeeper connect string from HBase configuration", ex);
            return null;
        }
        String serverList = hconf.get("hbase.zookeeper.quorum");
        final String port = hconf.get("hbase.zookeeper.property.clientPort");
        return StringUtils.join((Iterable)Iterables.transform(Arrays.asList(serverList.split(",")), (Function)new Function<String, String>(){

            @Nullable
            public String apply(String input) {
                return input + ":" + port;
            }
        }), (String)",");
    }

    public static void cleanZkPath(String path) {
        try (CuratorFramework zkClient = ZKUtil.newZookeeperClient();){
            zkClient.delete().deletingChildrenIfNeeded().forPath(path);
        }
    }

    static {
        Runtime.getRuntime().addShutdownHook(new Thread(new Runnable(){

            @Override
            public void run() {
                logger.info("Going to remove {} cached curator clients", (Object)CACHE.size());
                CACHE.invalidateAll();
            }
        }));
    }
}

