/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kylin.dict.lookup;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.NavigableSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import org.apache.kylin.common.KylinConfig;
import org.apache.kylin.common.persistence.ResourceStore;
import org.apache.kylin.dict.lookup.SnapshotTable;
import org.apache.kylin.dict.lookup.SnapshotTableSerializer;
import org.apache.kylin.metadata.TableMetadataManager;
import org.apache.kylin.metadata.model.TableDesc;
import org.apache.kylin.source.IReadableTable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SnapshotManager {
    private static final Logger logger = LoggerFactory.getLogger(SnapshotManager.class);
    private static final ConcurrentHashMap<String, Object> lockObjectMap = new ConcurrentHashMap();
    private KylinConfig config;
    private LoadingCache<String, SnapshotTable> snapshotCache;

    private SnapshotManager(KylinConfig config) {
        this.config = config;
        this.snapshotCache = CacheBuilder.newBuilder().removalListener((RemovalListener)new RemovalListener<String, SnapshotTable>(){

            public void onRemoval(RemovalNotification<String, SnapshotTable> notification) {
                logger.info("Snapshot with resource path {} is removed due to {}", notification.getKey(), (Object)notification.getCause());
            }
        }).maximumSize((long)config.getCachedSnapshotMaxEntrySize()).expireAfterWrite(1L, TimeUnit.DAYS).build((CacheLoader)new CacheLoader<String, SnapshotTable>(){

            public SnapshotTable load(String key) throws Exception {
                SnapshotTable snapshotTable = SnapshotManager.this.load(key, true);
                return snapshotTable;
            }
        });
    }

    public static SnapshotManager getInstance(KylinConfig config) {
        return config.getManager(SnapshotManager.class);
    }

    static SnapshotManager newInstance(KylinConfig config) throws IOException {
        return new SnapshotManager(config);
    }

    private Object getConcurrentObject(String resPath) {
        if (!lockObjectMap.containsKey(resPath)) {
            this.addObject(resPath);
        }
        return lockObjectMap.get(resPath);
    }

    private synchronized void addObject(String resPath) {
        if (!lockObjectMap.containsKey(resPath)) {
            lockObjectMap.put(resPath, new Object());
        }
    }

    public void wipeoutCache() {
        this.snapshotCache.invalidateAll();
    }

    public SnapshotTable getSnapshotTable(String resourcePath) throws IOException {
        try {
            SnapshotTable r = (SnapshotTable)this.snapshotCache.get((Object)resourcePath);
            if (r == null) {
                r = this.loadAndUpdateLocalCache(resourcePath);
            }
            return r;
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e.getCause());
        }
    }

    private SnapshotTable loadAndUpdateLocalCache(String snapshotResPath) throws IOException {
        SnapshotTable snapshotTable = this.load(snapshotResPath, true);
        this.snapshotCache.put((Object)snapshotTable.getResourcePath(), (Object)snapshotTable);
        return snapshotTable;
    }

    public List<SnapshotTable> getSnapshots(String tableName, IReadableTable.TableSignature sourceTableSignature) throws IOException {
        ArrayList result = Lists.newArrayList();
        String tableSnapshotsPath = SnapshotTable.getResourceDir(tableName);
        ResourceStore store = TableMetadataManager.getInstance(this.config).getStore();
        result.addAll(store.getAllResources(tableSnapshotsPath, SnapshotTableSerializer.INFO_SERIALIZER));
        if (sourceTableSignature != null) {
            String oldTableSnapshotsPath = SnapshotTable.getOldResourceDir(sourceTableSignature);
            result.addAll(store.getAllResources(oldTableSnapshotsPath, SnapshotTableSerializer.INFO_SERIALIZER));
        }
        return result;
    }

    public void removeSnapshot(String resourcePath) throws IOException {
        ResourceStore store = this.getStore();
        store.deleteResource(resourcePath);
        this.snapshotCache.invalidate((Object)resourcePath);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public SnapshotTable buildSnapshot(IReadableTable table, TableDesc tableDesc, KylinConfig cubeConfig) throws IOException {
        SnapshotTable snapshot = new SnapshotTable(table, tableDesc.getIdentity());
        snapshot.updateRandomUuid();
        Object object = this.getConcurrentObject(tableDesc.getIdentity());
        synchronized (object) {
            SnapshotTable reusableSnapshot = this.getReusableSnapShot(table, snapshot, tableDesc, cubeConfig);
            if (reusableSnapshot != null) {
                return this.updateDictLastModifiedTime(reusableSnapshot.getResourcePath());
            }
            snapshot.takeSnapshot(table, tableDesc);
            return this.trySaveNewSnapshot(snapshot);
        }
    }

    public SnapshotTable rebuildSnapshot(IReadableTable table, TableDesc tableDesc, String overwriteUUID) throws IOException {
        SnapshotTable snapshot = new SnapshotTable(table, tableDesc.getIdentity());
        snapshot.setUuid(overwriteUUID);
        snapshot.takeSnapshot(table, tableDesc);
        try {
            SnapshotTable existing = this.getSnapshotTable(snapshot.getResourcePath());
            snapshot.setLastModified(existing.getLastModified());
        }
        catch (Exception ex) {
            logger.error("Error reading {}, delete it and save rebuild", (Object)snapshot.getResourcePath(), (Object)ex);
        }
        this.save(snapshot);
        this.snapshotCache.put((Object)snapshot.getResourcePath(), (Object)snapshot);
        return snapshot;
    }

    private SnapshotTable getReusableSnapShot(IReadableTable table, SnapshotTable snapshot, TableDesc tableDesc, KylinConfig cubeConfig) throws IOException {
        String dup = this.checkDupByInfo(snapshot);
        if ((float)snapshot.getSignature().getSize() / 1024.0f / 1024.0f > (float)cubeConfig.getTableSnapshotMaxMB()) {
            throw new IllegalStateException("Table snapshot should be no greater than " + cubeConfig.getTableSnapshotMaxMB() + " MB, but " + tableDesc + " size is " + snapshot.getSignature().getSize());
        }
        if (dup != null) {
            logger.info("Identical input {}, reuse existing snapshot at {}", (Object)table.getSignature(), (Object)dup);
            return this.getSnapshotTable(dup);
        }
        return null;
    }

    public SnapshotTable trySaveNewSnapshot(SnapshotTable snapshotTable) throws IOException {
        String dupTable = this.checkDupByContent(snapshotTable);
        if (dupTable != null) {
            logger.info("Identical snapshot content {}, reuse existing snapshot at {}", (Object)snapshotTable, (Object)dupTable);
            return this.updateDictLastModifiedTime(dupTable);
        }
        this.save(snapshotTable);
        this.snapshotCache.put((Object)snapshotTable.getResourcePath(), (Object)snapshotTable);
        return snapshotTable;
    }

    private String checkDupByInfo(SnapshotTable snapshot) throws IOException {
        String resourceDir;
        ResourceStore store = this.getStore();
        NavigableSet<String> existings = store.listResources(resourceDir = snapshot.getResourceDir());
        if (existings == null) {
            return null;
        }
        IReadableTable.TableSignature sig = snapshot.getSignature();
        for (String existing : existings) {
            SnapshotTable existingTable = this.load(existing, false);
            if (existingTable == null || !sig.equals(existingTable.getSignature())) continue;
            return existing;
        }
        return null;
    }

    private String checkDupByContent(SnapshotTable snapshot) throws IOException {
        String resourceDir;
        ResourceStore store = this.getStore();
        NavigableSet<String> existings = store.listResources(resourceDir = snapshot.getResourceDir());
        if (existings == null) {
            return null;
        }
        for (String existing : existings) {
            SnapshotTable existingTable = this.load(existing, true);
            if (existingTable == null || !existingTable.equals(snapshot)) continue;
            return existing;
        }
        return null;
    }

    private SnapshotTable updateDictLastModifiedTime(String snapshotPath) throws IOException {
        ResourceStore store = this.getStore();
        long now = System.currentTimeMillis();
        store.updateTimestamp(snapshotPath, now);
        logger.info("Update snapshotTable {} lastModifiedTime to {}", (Object)snapshotPath, (Object)now);
        return this.loadAndUpdateLocalCache(snapshotPath);
    }

    private void save(SnapshotTable snapshot) throws IOException {
        ResourceStore store = this.getStore();
        String path = snapshot.getResourcePath();
        store.putBigResource(path, snapshot, System.currentTimeMillis(), SnapshotTableSerializer.FULL_SERIALIZER);
    }

    private SnapshotTable load(String resourcePath, boolean loadData) throws IOException {
        logger.info("Loading snapshotTable from {}, with loadData: {}", (Object)resourcePath, (Object)loadData);
        ResourceStore store = this.getStore();
        SnapshotTable table = store.getResource(resourcePath, loadData ? SnapshotTableSerializer.FULL_SERIALIZER : SnapshotTableSerializer.INFO_SERIALIZER);
        if (loadData) {
            logger.debug("Loaded snapshot at {}", (Object)resourcePath);
        }
        return table;
    }

    private ResourceStore getStore() {
        return ResourceStore.getStore(this.config);
    }
}

